/*
 * Copyright Staffan Gimåker 2007-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cmath>
#include <Eigen/Geometry>

#include "Frustum.hh"
#include "CullableEntity.hh"
#include "entities/Camera.hh"


using namespace peekabot;
using namespace peekabot::renderer;



Frustum::Frustum(const Camera &cam, float w_slack, float h_slack)
{
    calculate_frustum_bsphere(cam, w_slack, h_slack);
    calculate_frustum_planes(cam, w_slack, h_slack);
}


void Frustum::calculate_frustum_bsphere(
    const Camera &cam, float w_slack, float h_slack) throw()
{
    float far = cam.get_far_plane();
    float near = cam.get_near_plane();

    if( cam.is_orthographic() )
    {
        /**
         * <em>Bounding sphere for orthographic cameras:</em>
         * An orthographic frustum is described by a box, we need only
         * fix the longest side of this box. Once we have the longest side
         * of the frustum box, we multiply it with a the square root of two
         * to get the box's correponding bounding sphere.
         *
         * Ergo, its bounding sphere is:
         *
         * \f$\sqrt{2} \cdot
         * max\left(\{ \frac{m_far-m_near}{2}, half\_w, half\_h \}\right)\f$
         *
         * Where \f$ half\_h = d \tan{\frac{fov}{2}} \f$,
         * \f$ half\_w = d \tan{\frac{fov}{2} \frac{m\_w}{m\_h} } \f$, \e d
         * being the distance from the camera to the origin.
         */
        float d = cam.get_zoom_distance();
        float half_h = d*tanf(cam.get_fov_rad()/2);
        float half_w = d*tanf(cam.get_fov_rad()/2*cam.get_aspect_ratio());
        float tmp_w = /*2*m_half_w*/w_slack*far/near;
        float tmp_h = /*2*m_half_h*/h_slack*far/near;
        float extent = std::max((far-near)/2,
                                std::max(half_w+tmp_w, half_h+tmp_h));

        m_frustum_bsphere.set_pos(
            cam.get_viewer_mtow().translation() + (
                far-near)/2*cam.get_transformation().linear().col(0) );

        m_frustum_bsphere.set_radius(M_SQRT2*extent);
    }
    else
    {
        m_frustum_bsphere.set_pos(
            cam.get_viewer_mtow().translation() +
            (far+near)/2*cam.get_transformation().linear().col(0) );

        float tmp = (far-near)/2;
        float half_w = far * tanf(cam.get_horiz_fov_rad()/2);
        float half_h = far * tanf(cam.get_fov_rad()/2);
        m_frustum_bsphere.set_radius(sqrtf(half_h*half_h + half_w*half_w * tmp*tmp));
    }
}


void Frustum::calculate_frustum_planes(
    const Camera &cam, float w_slack, float h_slack) throw()
{
    const Eigen::Transform3f &cam_mtow = cam.get_viewer_mtow();

    // Camera view vector
    const Eigen::Vector3f vv(cam_mtow.linear().col(0));

    // Camera up vector
    const Eigen::Vector3f up(cam_mtow.linear().col(2));

    // Camera viewer position
    const Eigen::Vector3f p(cam.get_viewer_mtow().translation());

    if( !cam.is_orthographic() )
    {
        Eigen::Vector3f n; // temp. variable used to store the normal of all each plane

        float fov_rad = cam.get_fov_rad();
        float h_fov_rad = cam.get_horiz_fov_rad();

        n = (cam_mtow * Eigen::AngleAxisf(
                 h_fov_rad/2, Eigen::Vector3f::UnitZ())).linear() *
            Eigen::Vector3f(0, 1, 0);
        n.normalize();
        m_frustum_planes[LEFT].set_normal(n);
        m_frustum_planes[LEFT].set_d(-n.dot(p));


        n = (cam_mtow * Eigen::AngleAxisf(
                 -h_fov_rad/2, Eigen::Vector3f::UnitZ())).linear() *
            Eigen::Vector3f(0, -1, 0);
        n.normalize();
        m_frustum_planes[RIGHT].set_normal(n);
        m_frustum_planes[RIGHT].set_d(-n.dot(p));


        n = (cam_mtow * Eigen::AngleAxisf(
                 -fov_rad/2, Eigen::Vector3f::UnitY())).linear() *
            Eigen::Vector3f(0, 0, 1);
        n.normalize();
        m_frustum_planes[TOP].set_normal(n);
        m_frustum_planes[TOP].set_d(-n.dot(p));


        n = (cam_mtow * Eigen::AngleAxisf(
                 fov_rad/2, Eigen::Vector3f::UnitY())).linear() *
            Eigen::Vector3f(0, 0, -1);
        n.normalize();
        m_frustum_planes[BOTTOM].set_normal(n);
        m_frustum_planes[BOTTOM].set_d(-n.dot(p));


        n = cam_mtow.linear() * Eigen::Vector3f(1, 0, 0);
        n.normalize();
        m_frustum_planes[FAR].set_normal(n);
        m_frustum_planes[FAR].set_d(-n.dot(p + vv * cam.get_far_plane()));


        n = cam_mtow.linear() * Eigen::Vector3f(-1, 0, 0);
        n.normalize();
        m_frustum_planes[NEAR].set_normal(n);
        m_frustum_planes[NEAR].set_d(-n.dot(p + vv * cam.get_near_plane()));
    }
    else
    {
        const Eigen::Vector3f ortho_vector(cam_mtow.linear().col(1));

        Eigen::Vector3f n; // the normal

        float d = cam.get_zoom_distance();
        // Half the height of the near plane, in meters
        float half_h = d*tanf(cam.get_fov_rad()/2);
        // Half the width of the near plane, in meters
        float half_w = half_h * cam.get_aspect_ratio();


        n = cam_mtow.linear() * Eigen::Vector3f(0, 1, 0);
        n.normalize();
        m_frustum_planes[LEFT].set_normal(n);
        m_frustum_planes[LEFT].set_d(-n.dot(p + ortho_vector * half_w));


        n = cam_mtow.linear() * Eigen::Vector3f(0, -1, 0);
        n.normalize();
        m_frustum_planes[RIGHT].set_normal(n);
        m_frustum_planes[RIGHT].set_d(-n.dot(p - ortho_vector * half_w));


        n = cam_mtow.linear() * Eigen::Vector3f(0, 0, 1);
        n.normalize();
        m_frustum_planes[TOP].set_normal(n);
        m_frustum_planes[TOP].set_d(-n.dot(p + up * half_h));


        n = cam_mtow.linear() * Eigen::Vector3f(0, 0, -1);
        n.normalize();
        m_frustum_planes[BOTTOM].set_normal(n);
        m_frustum_planes[BOTTOM].set_d(-n.dot(p - up * half_h));


        n = cam_mtow.linear() * Eigen::Vector3f(1, 0, 0);
        n.normalize();
        m_frustum_planes[FAR].set_normal(n);
        m_frustum_planes[FAR].set_d(-n.dot(p + vv * cam.get_far_plane()));


        n = cam_mtow.linear() * Eigen::Vector3f(-1, 0, 0);
        n.normalize();
        m_frustum_planes[NEAR].set_normal(n);
        m_frustum_planes[NEAR].set_d(-n.dot(p + vv * cam.get_near_plane()));
    }
}


IntersectionTestResult Frustum::is_in_frustum(
    const BoundingSphere& bsphere, int &plane_mask) const throw()
{
    IntersectionTestResult retval = INSIDE;

    for( register int i = 5; i >= 0; --i )
    {
        if( ((plane_mask >> i) & 1) == 1 )
        {
            IntersectionTestResult test = bsphere.contained_in(m_frustum_planes[i]);

            if( test == DISJOINT )
            {
                plane_mask ^= (1 << i);
                retval = DISJOINT;
            }
            else if( test == INTERSECT && retval != DISJOINT )
                retval = INTERSECT;
        }
    }

    return retval;
}


IntersectionTestResult Frustum::is_in_frustum(
    const BoundingSphere& bsphere) const throw()
{
    IntersectionTestResult retval = INSIDE;

    for( int i = 0; i < 6; ++i )
    {
        IntersectionTestResult test = bsphere.contained_in(m_frustum_planes[i]);

        if( test == DISJOINT )
                return DISJOINT;

        if( test == INTERSECT )
            retval = INTERSECT;
    }

    return retval;
}


IntersectionTestResult Frustum::is_in_uncapped_frustum(
    const BoundingSphere& bsphere) const throw()
{
    IntersectionTestResult retval = INSIDE;

    for( int i = 0; i < 4; ++i )
    {
        IntersectionTestResult test = bsphere.contained_in(m_frustum_planes[i]);

        if( test == DISJOINT )
            return DISJOINT;

        if( test == INTERSECT )
            retval = INTERSECT;
    }

    return retval;
}


IntersectionTestResult Frustum::is_in_uncapped_frustum(
    const BoundingSphere& bsphere, int &plane_mask) const throw()
{
    IntersectionTestResult retval = INSIDE;

    for( register int i = 3; i >= 0; --i )
    {
        if( ((plane_mask >> i) & 1) == 1 )
        {
            IntersectionTestResult test = bsphere.contained_in(m_frustum_planes[i]);

            if( test == DISJOINT )
            {
                plane_mask ^= (1 << i);
                retval = DISJOINT;
            }
            else if( test == INTERSECT && retval != DISJOINT )
                retval = INTERSECT;
        }
    }

    return retval;
}
