/*
 * Copyright Staffan Gimåker 2007-2008, 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <GL/glew.h>
#include "Mesh.hh"
#include "MeshVisitor.hh"


using namespace peekabot;
using namespace peekabot::renderer;


// -------------- Mesh implementation ----------------


Mesh::Mesh()
    : m_index_offset(0),
      m_element_count(0)
{
}


Mesh::Mesh(boost::shared_ptr<VertexBuffer> vertices,
           boost::shared_ptr<VertexBuffer> colors,
           boost::shared_ptr<VertexBuffer> tex_coords,
           boost::shared_ptr<IndexBuffer> indices,
           uint32_t index_offset,
           uint32_t element_count)
    : m_indices(indices),
      m_index_offset(index_offset),
      m_element_count(element_count)
{
    if( vertices )
        get_state().get<statelets::VertexPointer>()->set_buffer(vertices);
    
    if( colors )
        get_state().get<statelets::ColorPointer>()->set_buffer(colors);
    
    if( tex_coords )
        get_state().get<statelets::TexCoordPointer>()->set_buffer(tex_coords);
}

Mesh::Mesh(const Mesh &mesh)
    : Renderable(mesh),
      m_indices(mesh.m_indices),
      m_index_offset(mesh.m_index_offset),
      m_element_count(mesh.m_element_count)
{
}

bool Mesh::has_vertices() const throw()
{
    return get_state().is_set<statelets::VertexPointer>();
}

boost::shared_ptr<VertexBuffer> &Mesh::get_vertices() throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::VertexPointer>() )
        throw std::runtime_error(
            "Failed to get mesh vertices because there are none.");
    return get_state().get<statelets::VertexPointer>()->get_buffer();
}

const boost::shared_ptr<VertexBuffer> &Mesh::get_vertices() const throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::VertexPointer>() )
        throw std::runtime_error(
            "Failed to get mesh vertices because there are none.");
    return get_state().get<statelets::VertexPointer>()->get_buffer();
}

bool Mesh::has_colors() const throw()
{
    return get_state().is_set<statelets::ColorPointer>();
}

boost::shared_ptr<VertexBuffer> &Mesh::get_colors() throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::ColorPointer>() )
        throw std::runtime_error(
            "Failed to get mesh colors because there are none.");
    return get_state().get<statelets::ColorPointer>()->get_buffer();
}

const boost::shared_ptr<VertexBuffer> &Mesh::get_colors() const throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::ColorPointer>() )
        throw std::runtime_error(
            "Failed to get mesh colors because there are none.");
    return get_state().get<statelets::ColorPointer>()->get_buffer();
}

bool Mesh::has_tex_coords() const throw()
{
    return get_state().is_set<statelets::TexCoordPointer>();
}

boost::shared_ptr<VertexBuffer> &Mesh::get_tex_coords() throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::TexCoordPointer>() )
        throw std::runtime_error(
            "Failed to get mesh tex coords because there are none.");
    return get_state().get<statelets::TexCoordPointer>()->get_buffer();
}

const boost::shared_ptr<VertexBuffer> &Mesh::get_tex_coords() const throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::TexCoordPointer>() )
        throw std::runtime_error(
            "Failed to get mesh tex coords because there are none.");
    return get_state().get<statelets::TexCoordPointer>()->get_buffer();
}

void Mesh::set_vertices(
    const boost::shared_ptr<VertexBuffer> &buffer,
    std::size_t stride, std::size_t offset)
{
    if( buffer )
    {
        get_state().get<statelets::VertexPointer>()->configure(
            buffer, stride, offset);
    }
    else
    {
        get_state().unset<statelets::VertexPointer>();
    }
}

void Mesh::set_colors(
    const boost::shared_ptr<VertexBuffer> &buffer,
    std::size_t stride, std::size_t offset)
{
    if( buffer )
    {
        get_state().get<statelets::ColorPointer>()->configure(
            buffer, stride, offset);
    }
    else
    {
        get_state().unset<statelets::ColorPointer>();
    }
}

void Mesh::set_tex_coords(boost::shared_ptr<VertexBuffer> tex_coords) throw()
{
    if( tex_coords )
        get_state().get<statelets::TexCoordPointer>()->set_buffer(tex_coords);
    else
        get_state().unset<statelets::TexCoordPointer>();
}

void Mesh::set_element_count(uint32_t element_count) throw()
{
    m_element_count = element_count;
}

void Mesh::set_index_offset(uint32_t offset) throw()
{
    m_index_offset = offset;
}


// ----------------- TriMesh implementation -------------------

TriMesh::TriMesh()
    : Mesh()
{
}


TriMesh::TriMesh(boost::shared_ptr<VertexBuffer> vertices,
                 boost::shared_ptr<VertexBuffer> normals,
                 boost::shared_ptr<VertexBuffer> colors,
                 boost::shared_ptr<VertexBuffer> tex_coords,
                 boost::shared_ptr<IndexBuffer> indices,
                 uint32_t index_offset, 
                 uint32_t element_count)
    : Mesh(vertices, colors, tex_coords, indices, index_offset, element_count)
{
    if( normals )
        get_state().get<statelets::NormalPointer>()->set_buffer(normals);
}

TriMesh::TriMesh(const TriMesh &mesh)
    : Mesh(mesh)
{
}

TriMesh *TriMesh::clone() const throw()
{
    return new TriMesh(*this);
}

bool TriMesh::has_normals() const throw()
{
    return get_state().is_set<statelets::NormalPointer>();
}

boost::shared_ptr<VertexBuffer> &TriMesh::get_normals() throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::NormalPointer>() )
        throw std::runtime_error(
            "Failed to get mesh normals because there are none.");
    return get_state().get<statelets::NormalPointer>()->get_buffer();
}

const boost::shared_ptr<VertexBuffer> &TriMesh::get_normals() const throw(std::runtime_error)
{
    if( !get_state().is_set<statelets::NormalPointer>() )
        throw std::runtime_error(
            "Failed to get mesh normals because there are none.");
    return get_state().get<statelets::NormalPointer>()->get_buffer();
}

uint32_t TriMesh::vertices_per_element() const throw()
{
    return 3;
}

void TriMesh::render(RenderContext &context) const
{
    if( has_vertices() && get_element_count() > 0 )
    {
        if( get_indices() )
        {
            get_indices()->draw_elements(
                GL_TRIANGLES, vertices_per_element()*get_element_count(),
                GL_UNSIGNED_INT, get_index_offset());
        }
        else
        {
            glDrawArrays(GL_TRIANGLES, 0, vertices_per_element()*get_element_count());
        }
    }
}

void TriMesh::set_normals(
    const boost::shared_ptr<VertexBuffer> &buffer,
    std::size_t stride, std::size_t offset)
{
    if( buffer )
    {
        get_state().get<statelets::NormalPointer>()->configure(
            buffer, stride, offset);
    }
    else
    {
        get_state().unset<statelets::NormalPointer>();
    }
}

void TriMesh::accept(MeshVisitor &visitor) throw()
{
    visitor.visit(*this);
}


// ----------------- LineMesh implementation -------------------

LineMesh::LineMesh()
    : Mesh()
{
    get_state().set(new statelets::Lighting(false));
}

LineMesh::LineMesh(boost::shared_ptr<VertexBuffer> vertices,
                   boost::shared_ptr<VertexBuffer> colors,
                   boost::shared_ptr<VertexBuffer> tex_coords,
                   boost::shared_ptr<IndexBuffer> indices,
                   uint32_t index_offset, 
                   uint32_t element_count)
    : Mesh(vertices, colors, tex_coords, indices, index_offset, element_count)
{
    get_state().set(new statelets::Lighting(false));
}

LineMesh::LineMesh(const LineMesh &mesh)
    : Mesh(mesh)
{
}

LineMesh *LineMesh::clone() const throw()
{
    return new LineMesh(*this);
}

uint32_t LineMesh::vertices_per_element() const throw()
{
    return 2;
}

void LineMesh::render(RenderContext &context) const
{
    if( has_vertices() && get_element_count() > 0 )
    {
        if( get_indices() )
        {
            get_indices()->draw_elements(
                GL_LINES, vertices_per_element()*get_element_count(),
                GL_UNSIGNED_INT, get_index_offset());
        }
        else
        {
            glDrawArrays(GL_LINES, 0, vertices_per_element()*get_element_count());
        }
    }
}

void LineMesh::accept(MeshVisitor &visitor) throw()
{
    visitor.visit(*this);
}


// ----------------- LineStripMesh implementation -------------------


LineStripMesh::LineStripMesh()
    : Mesh()
{
    get_state().set(new statelets::Lighting(false));
}

LineStripMesh::LineStripMesh(
    boost::shared_ptr<VertexBuffer> vertices,
    boost::shared_ptr<VertexBuffer> colors,
    boost::uint32_t element_count)
    : Mesh(
        vertices, colors,
        boost::shared_ptr<VertexBuffer>(),
        boost::shared_ptr<IndexBuffer>(), 0, element_count)
{
    get_state().set(new statelets::Lighting(false));
}


LineStripMesh::LineStripMesh(const LineStripMesh &mesh)
    : Mesh(mesh)
{
}


LineStripMesh *LineStripMesh::clone() const throw()
{
    return new LineStripMesh(*this);
}

boost::uint32_t LineStripMesh::vertices_per_element() const throw()
{
    return 1;
}

void LineStripMesh::render(RenderContext &context) const
{
    if( has_vertices() && get_element_count() > 0 )
    {
        glDrawArrays(GL_LINE_STRIP, 0, vertices_per_element()*get_element_count());
    }
}

void LineStripMesh::accept(MeshVisitor &visitor) throw()
{
    visitor.visit(*this);
}


// ----------------- PointMesh implementation -------------------

PointMesh::PointMesh()
    : Mesh()
{
    get_state().set(new statelets::Lighting(false));
}

PointMesh::PointMesh(boost::shared_ptr<VertexBuffer> vertices,
                     boost::shared_ptr<VertexBuffer> colors,
                     uint32_t element_count)
    : Mesh(vertices, colors, 
           boost::shared_ptr<VertexBuffer>(), 
           boost::shared_ptr<IndexBuffer>(), 
           0,
           element_count)
{
    get_state().set(new statelets::Lighting(false));
}

PointMesh::PointMesh(const PointMesh &mesh)
    : Mesh(mesh)
{
}

PointMesh *PointMesh::clone() const throw()
{
    return new PointMesh(*this);
}

uint32_t PointMesh::vertices_per_element() const throw()
{
    return 1;
}

void PointMesh::render(RenderContext &context) const
{
    if( has_vertices() && get_element_count() > 0 )
    {
        glDrawArrays(GL_POINTS, 0, get_element_count());
    }
}

void PointMesh::accept(MeshVisitor &visitor) throw()
{
    visitor.visit(*this);
}

