/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <cmath>

#include "MeshFactory.hh"
#include "Mesh.hh"

using namespace peekabot;
using namespace peekabot::renderer;


Mesh *MeshFactory::get_disc_mesh(int segments, float r) throw()
{
    // Vertices...
    Eigen::Vector3f *x = new Eigen::Vector3f[2*(segments+1)];

    x[0] = x[segments+1] = Eigen::Vector3f(0,0,0);
    for( int i = 0; i < segments; ++i )
    {
        float alpha = i*(3.14159*2/segments);
        x[i+1] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), 0);
        x[(segments+1)+i+1] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), 0);
    }

    boost::shared_ptr<VertexBuffer> vertices(new VertexBuffer);
    vertices->set_data(x, sizeof(Eigen::Vector3f)*2*(segments+1));

    // Normals...
    for( int i = 0; i <= segments; ++i )
    {
        x[i] = Eigen::Vector3f(0, 0, 1);
        x[i+(segments+1)] = Eigen::Vector3f(0, 0, -1);
    }

    boost::shared_ptr<VertexBuffer> normals(new VertexBuffer);
    normals->set_data(x, sizeof(Eigen::Vector3f)*2*(segments+1));

    delete[] x;

    // Indices...
    boost::uint32_t *inds = new boost::uint32_t[2*3*segments];

    inds[0] = 0;
    inds[1] = segments;
    inds[2] = 1;

    inds[3*segments+0] = (segments+1)+0;
    inds[3*segments+1] = (segments+1)+1;
    inds[3*segments+2] = (segments+1)+segments;

    for( int i = 1; i < segments; ++i )
    {
        inds[3*i+0] = 0;
        inds[3*i+1] = i;
        inds[3*i+2] = i+1;

        inds[3*(segments+i)+0] = (segments+1);
        inds[3*(segments+i)+1] = (segments+1)+i+1;
        inds[3*(segments+i)+2] = (segments+1)+i;
    }

    boost::shared_ptr<IndexBuffer> indices(new IndexBuffer);
    indices->set_data(inds, sizeof(Eigen::Vector3f)*2*segments);

    delete[] inds;


    return new TriMesh(
        vertices, normals, 
        boost::shared_ptr<VertexBuffer>(), 
        boost::shared_ptr<VertexBuffer>(), 
        indices, 0, 2*segments);
}


Mesh *MeshFactory::get_cylinder_mesh(int segments, float r, float h) throw()
{
    //
    // If you have found a bug in this code, and I am quite confident that 
    // you will, you might as well re-write it from scratch at once, because 
    // it's so hairy and ugly you will most likely die if you try to wrap your
    // head around it.
    //

    // Vertices...
    Eigen::Vector3f *x = new Eigen::Vector3f[2*(segments+1) + 2*segments];

    x[0] = Eigen::Vector3f(0,0,h/2);
    x[segments+1] = Eigen::Vector3f(0,0,-h/2);
    for( int i = 0; i < segments; ++i )
    {
        // Top/bottom:
        float alpha = i*(3.14159*2/segments);
        x[i+1] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), h/2);
        x[(segments+1)+i+1] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), -h/2);

        // Sides:
        x[2*(segments+1)+2*i+0] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), h/2);
        x[2*(segments+1)+2*i+1] = Eigen::Vector3f(r*cosf(alpha), r*sinf(alpha), -h/2);
    }

    boost::shared_ptr<VertexBuffer> vertices(new VertexBuffer);
    vertices->set_data(
        x, sizeof(Eigen::Vector3f)*(2*(segments+1) + 2*segments));

    // Normals...
    //
    // Top/bottom:
    for( int i = 0; i <= segments; ++i )
    {
        x[i] = Eigen::Vector3f(0, 0, 1);
        x[i+(segments+1)] = Eigen::Vector3f(0, 0, -1);
    }
    // Sides:
    for( int i = 0; i < segments; ++i )
    {
        float alpha = i*(3.14159*2/segments);
        x[2*(segments+1)+2*i+0] = Eigen::Vector3f(cosf(alpha), sinf(alpha), 0);
        x[2*(segments+1)+2*i+1] = Eigen::Vector3f(cosf(alpha), sinf(alpha), 0);
    }


    boost::shared_ptr<VertexBuffer> normals(new VertexBuffer);
    normals->set_data(
        x, sizeof(Eigen::Vector3f)*(2*(segments+1) + 2*segments));

    delete[] x;

    // Indices...
    boost::uint32_t *inds = new boost::uint32_t[3*(2*segments+2*segments)];

    // Top/bottom:
    inds[0] = 0;
    inds[1] = segments;
    inds[2] = 1;

    inds[3*segments+0] = (segments+1)+0;
    inds[3*segments+1] = (segments+1)+1;
    inds[3*segments+2] = (segments+1)+segments;

    for( int i = 1; i < segments; ++i )
    {
        inds[3*i+0] = 0;
        inds[3*i+1] = i;
        inds[3*i+2] = i+1;

        inds[3*(segments+i)+0] = (segments+1);
        inds[3*(segments+i)+1] = (segments+1)+i+1;
        inds[3*(segments+i)+2] = (segments+1)+i;
    }

    // Sides:
    for( int i = 0; i < segments-1; ++i )
    {
        inds[3*(2*segments+2*i)+0] = 2*(segments+1)+2*i+0;
        inds[3*(2*segments+2*i)+1] = 2*(segments+1)+2*i+1;
        inds[3*(2*segments+2*i)+2] = 2*(segments+1)+2*(i+1)+1;

        inds[3*(2*segments+2*i+1)+0] = 2*(segments+1)+2*i+0;
        inds[3*(2*segments+2*i+1)+1] = 2*(segments+1)+2*(i+1)+1;
        inds[3*(2*segments+2*i+1)+2] = 2*(segments+1)+2*(i+1)+0;
    }

    inds[3*(2*segments+2*(segments-1))+0] = 2*(segments+1)+2*(segments-1)+0;
    inds[3*(2*segments+2*(segments-1))+1] = 2*(segments+1)+2*(segments-1)+1;
    inds[3*(2*segments+2*(segments-1))+2] = 2*(segments+1)+2*0+1;
    
    inds[3*(2*segments+2*(segments-1)+1)+0] = 2*(segments+1)+2*(segments-1)+0;
    inds[3*(2*segments+2*(segments-1)+1)+1] = 2*(segments+1)+2*0+1;
    inds[3*(2*segments+2*(segments-1)+1)+2] = 2*(segments+1)+2*0+0;



    boost::shared_ptr<IndexBuffer> indices(new IndexBuffer);
    indices->set_data(
        inds, sizeof(boost::uint32_t)*(3*(2*segments+2*segments)));

    delete[] inds;


    return new TriMesh(
        vertices, normals, 
        boost::shared_ptr<VertexBuffer>(), 
        boost::shared_ptr<VertexBuffer>(), 
        indices, 0, 2*segments+2*segments);
}


Mesh *MeshFactory::get_icosphere_mesh(int subdiv, float r) throw()
{
    const double X = 0.525731112119133606;
    const double Z = 0.850650808352039932;

    static Eigen::Vector3f vert_data[12] =
    {
        Eigen::Vector3f(-X, -Z, 0.0), Eigen::Vector3f(X, -Z, 0.0), 
        Eigen::Vector3f(-X, Z, 0.0), Eigen::Vector3f(X, Z, 0.0),
        Eigen::Vector3f(0.0, -X, Z), Eigen::Vector3f(0.0, X, Z), 
        Eigen::Vector3f(0.0, -X, -Z), Eigen::Vector3f(0.0, X, -Z),
        Eigen::Vector3f(Z, 0.0, X), Eigen::Vector3f(-Z, 0.0, X), 
        Eigen::Vector3f(Z, 0.0, -X), Eigen::Vector3f(-Z, 0.0, -X)
    };

    static boost::uint32_t ind_data[20][3] =
    {
        {0, 1, 4}, {0, 4, 9}, {9, 4, 5}, {4, 8, 5}, {4, 1, 8},
        {8, 1, 10}, {8, 10, 3}, {5, 8, 3}, {5, 3, 2}, {2, 3, 7},
        {7, 3, 10}, {7, 10, 6}, {7, 6, 11}, {11, 6, 0}, {0, 6, 1},
        {6, 10, 1}, {9, 11, 0}, {9, 2, 11}, {9, 5, 2}, {7, 11, 2}
    };

    std::vector<Eigen::Vector3f> verts;
    std::vector<boost::uint32_t> inds;

    for( int i = 0; i < 12; ++i )
        verts.push_back(vert_data[i]);


    for( int i = 0; i < 20; ++i )
    {
        subdivide(
            ind_data[i][0], ind_data[i][1], ind_data[i][2], 
            verts, inds, subdiv);
    }

    int n_tris = inds.size()/3;
    int n_verts = verts.size();

    for( size_t i = 0; i < verts.size(); ++i )
        verts[i] *= r;

    boost::shared_ptr<VertexBuffer> vertices(new VertexBuffer);
    vertices->set_data(&verts[0], sizeof(Eigen::Vector3f)*n_verts);

    boost::shared_ptr<IndexBuffer> indices(new IndexBuffer);
    indices->set_data(&inds[0], 3*sizeof(boost::uint32_t)*n_tris);


    return new TriMesh(
        vertices, vertices, //normals, 
        boost::shared_ptr<VertexBuffer>(), 
        boost::shared_ptr<VertexBuffer>(), 
        indices, 0, n_tris);
}


Mesh *MeshFactory::get_circle_mesh(int segments, float r) throw()
{
    std::vector<Eigen::Vector3f> verts;

    for( int i = 0; i <= segments; ++i )
    {
        verts.push_back(
            Eigen::Vector3f(
                r*cosf(2*M_PI*i/segments),
                r*sinf(2*M_PI*i/segments),
                0));
    }

    boost::shared_ptr<VertexBuffer> vertices(new VertexBuffer);
    vertices->set_data(&verts[0], sizeof(Eigen::Vector3f)*verts.size());

    return new LineStripMesh(
        vertices,
        boost::shared_ptr<VertexBuffer>(),
        segments+1);
}


void MeshFactory::subdivide(
    boost::uint32_t v0_idx, boost::uint32_t v1_idx, boost::uint32_t v2_idx, 
    std::vector<Eigen::Vector3f> &vertices,
    std::vector<boost::uint32_t> &indices,
    int depth) throw()
{
    if( depth == 0 )
    {
        indices.push_back(v0_idx);
        indices.push_back(v1_idx);
        indices.push_back(v2_idx);
    }
    else
    {
        const Eigen::Vector3f &v0 = vertices[v0_idx];
        const Eigen::Vector3f &v1 = vertices[v1_idx];
        const Eigen::Vector3f &v2 = vertices[v2_idx];
        
        Eigen::Vector3f v01((v0 + v1) / 2);
        Eigen::Vector3f v12((v1 + v2) / 2);
        Eigen::Vector3f v20((v2 + v0) / 2);

        v01.normalize();
        v12.normalize();
        v20.normalize();

        int n = vertices.size();

        vertices.push_back(v01);
        vertices.push_back(v12);
        vertices.push_back(v20);

        subdivide(v0_idx, n+0, n+2, vertices, indices, depth - 1);
        subdivide(v1_idx, n+1, n+0, vertices, indices, depth - 1);
        subdivide(v2_idx, n+2, n+1, vertices, indices, depth - 1);
        subdivide(n+0, n+1, n+2, vertices, indices, depth - 1);
    }
}


Mesh *MeshFactory::get_cone_mesh(
    float r, float h,
    size_t segments, size_t lat_subdiv) throw()
{
    assert( h != 0 );

    std::vector<float> vertices;
    std::vector<boost::uint32_t> indices;
    std::vector<float> normals;


    // Generate cone bottom
    size_t mid_idx = 0;
    vertices.push_back(0);
    vertices.push_back(0);
    vertices.push_back(-h/2);
    normals.push_back(0);
    normals.push_back(0);
    normals.push_back(-1);

    for( size_t i = 0; i < segments; ++i )
    {
        double k = double(i)/segments;
        double alpha = k*2*M_PI;

        vertices.push_back(r*cos(alpha));
        vertices.push_back(r*sin(alpha));
        vertices.push_back(-h/2);

        normals.push_back(0);
        normals.push_back(0);
        normals.push_back(-1);

        // Triangles in CCW order
        indices.push_back((i+1==segments ? 0:i+1)+1);
        indices.push_back(i+1);
        indices.push_back(mid_idx);
    }

    // Generate cone sides
    size_t off = vertices.size()/3;
    const size_t N = 2+lat_subdiv;
    for( size_t i = 0; i < segments; ++i )
    {
        float alpha = 2*M_PI*i/segments;
        float beta = 2*M_PI*(i+0.5)/segments;

        // a
        Eigen::Vector3f a(r*cosf(alpha), r*sinf(alpha), -h/2);
        vertices.push_back(a(0));
        vertices.push_back(a(1));
        vertices.push_back(a(2));

        Eigen::Vector3f n_a(cosf(alpha), sinf(alpha), r/h);
        n_a.normalize();
        normals.push_back(n_a(0));
        normals.push_back(n_a(1));
        normals.push_back(n_a(2));

        // b (aka tip)
        Eigen::Vector3f b(0, 0, h/2);
        vertices.push_back(b(0));
        vertices.push_back(b(1));
        vertices.push_back(b(2));

        Eigen::Vector3f n_b(cosf(beta), sinf(beta), r/h);
        n_b.normalize();
        normals.push_back(n_b(0));
        normals.push_back(n_b(1));
        normals.push_back(n_b(2));

        for( size_t i = 0; i < lat_subdiv; ++i )
        {
            float k_a = double(i+1)/(lat_subdiv+1);
            float k_b = 1-k_a;

            Eigen::Vector3f c = a*k_a + b*k_b;

            vertices.push_back(c(0));
            vertices.push_back(c(1));
            vertices.push_back(c(2));

            Eigen::Vector3f n_c = n_a*k_a + n_b*k_b;
            normals.push_back(n_c(0));
            normals.push_back(n_c(1));
            normals.push_back(n_c(2));
        }

        // add triangles, in CCW order

        size_t j = (i+1) % segments;

        if( lat_subdiv < 1 )
        {
            // a -> a' -> b
            indices.push_back(N*i + 0 + off);
            indices.push_back(N*j + 0 + off);
            indices.push_back(N*i + 1 + off);
        }
        else
        {
            // Top-most triangle: c0 -> c0' -> b
            indices.push_back(N*i + 2 + off);
            indices.push_back(N*j + 2 + off);
            indices.push_back(N*i + 1 + off);

            // Bottom-most triangles: a -> a' -> cn'
            indices.push_back(N*i + 0 + off);
            indices.push_back(N*j + 0 + off);
            indices.push_back(N*j + 1+lat_subdiv + off);
            // a -> cn' -> cn
            indices.push_back(N*i + 0 + off);
            indices.push_back(N*j + 1+lat_subdiv + off);
            indices.push_back(N*i + 1+lat_subdiv + off);

            for( size_t k = 0; k < lat_subdiv-1; ++k )
            {
                // c_{k+1} -> c_{k+1}' -> c_k'
                indices.push_back(N*i + 3+k + off);
                indices.push_back(N*j + 3+k + off);
                indices.push_back(N*j + 2+k + off);

                // c_{k+1} -> c_k' -> c_k
                indices.push_back(N*i + 3+k + off);
                indices.push_back(N*j + 2+k + off);
                indices.push_back(N*i + 2+k + off);
            }
        }
    }

    assert( normals.size() == vertices.size() );

    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    vb->set_data(
        &vertices[0],
        sizeof(float)*vertices.size());

    boost::shared_ptr<IndexBuffer> ib(new IndexBuffer);
    ib->set_data(
        &indices[0],
        sizeof(boost::uint32_t)*indices.size());

    boost::shared_ptr<VertexBuffer> nb(new VertexBuffer);
    nb->set_data(
        &normals[0],
        sizeof(float)*normals.size());

    return new TriMesh(
        vb, nb,
        boost::shared_ptr<VertexBuffer>(),
        boost::shared_ptr<VertexBuffer>(),
        ib, 0, indices.size()/3);
}
