/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ObjectEntity.hh"
#include "statelets/Transformation.hh"


using namespace peekabot;
using namespace peekabot::renderer;


ObjectEntity::ObjectEntity()
    : m_layer(0),
      m_is_visible(true),
      m_is_selected(false),
      m_has_selected_ancestor(false),
      m_parent(0)
{
    m_mtow.setIdentity();
    m_template_state.set(new statelets::Transformation(m_mtow));
}


void ObjectEntity::set_transformation(const Eigen::Transform3f &mtow)
{
    m_mtow = mtow;
    on_transformation_set();
    m_on_transformation_set(this);
}


float ObjectEntity::get_x_scale() const throw()
{
    return m_mtow.linear().col(0).norm();
}


float ObjectEntity::get_y_scale() const throw()
{
    return m_mtow.linear().col(1).norm();
}


float ObjectEntity::get_z_scale() const throw()
{
    return m_mtow.linear().col(2).norm();
}


void ObjectEntity::set_opacity(float alpha)
{
    if( alpha > 1 || alpha < 0 )
        throw std::domain_error("Opacity must be in the range [0,1]");


    if( alpha < 1 )
    {
        m_template_state.set(
            boost::shared_ptr<Statelet>(
                new statelets::BlendMode(
                    GL_CONSTANT_ALPHA, GL_ONE_MINUS_CONSTANT_ALPHA)));

        m_template_state.set(
            boost::shared_ptr<Statelet>(
                new statelets::Alpha(alpha)));

        m_template_state.set(
            boost::shared_ptr<Statelet>(
                new statelets::Culling(false)));

        m_template_state.set(
            boost::shared_ptr<Statelet>(
                new statelets::LightModel(true)));

        m_template_state.set(
            boost::shared_ptr<Statelet>(
                new statelets::DepthMask(false)));
    }
    else
    {
        m_template_state.unset<statelets::BlendMode>();
        m_template_state.unset<statelets::Alpha>();
        m_template_state.unset<statelets::Culling>();
        m_template_state.unset<statelets::LightModel>();
        m_template_state.unset<statelets::DepthMask>();
    }
}


void ObjectEntity::set_color(const RGBColor &color)
{
    if( color.r > 1 || color.r < 0 ||
        color.g > 1 || color.g < 0 ||
        color.b > 1 || color.b < 0 )
        throw std::domain_error(
            "All color components must be in the range [0,1]");

    m_template_state.set(
        boost::shared_ptr<Statelet>(
            new statelets::Color(color)));
}


void ObjectEntity::set_name(uint32_t name)
{
    m_template_state.set(
        boost::shared_ptr<Statelet>(
            new statelets::Name(name)));
}


void ObjectEntity::set_layer(uint32_t layer)
{
    if( layer >= NUMBER_OF_LAYERS )
        throw std::domain_error("Layer must be in the range [0,max_layers)");

    m_layer = layer;
}


uint32_t ObjectEntity::get_layer() const throw()
{
    return m_layer;
}


void ObjectEntity::set_visibility(bool visible)
{
    if( visible != m_is_visible )
    {
        m_is_visible = visible;
        m_on_visibility_set(this);
    }
}


void ObjectEntity::set_selected(bool is_selected) throw()
{
    if( m_is_selected != is_selected )
    {
        m_is_selected = is_selected;
        m_on_selected_set(this);
    }
}


void ObjectEntity::set_has_selected_ancestor(bool has_selected_ancestor) throw()
{
    m_has_selected_ancestor = has_selected_ancestor;
}


ObjectEntity *ObjectEntity::get_parent()
{
    return m_parent;
}


const ObjectEntity *ObjectEntity::get_parent() const
{
    return m_parent;
}


void ObjectEntity::set_parent(ObjectEntity *parent)
{
    m_parent = parent;
}


const Eigen::Transform3f &ObjectEntity::get_parent_transformation()
    const throw()
{
    const static Eigen::Transform3f eye(Eigen::Matrix4f::Identity());
    if( get_parent() )
        return get_parent()->get_transformation();
    else
        return eye;
}
