/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/filesystem/operations.hpp>
#include <boost/scoped_array.hpp>
#include <sstream>

#include "PbmfLoader.hh"
#include "../MessageHub.hh"
#include "entities/MeshBased.hh"
#include "Mesh.hh"
#include "TextureManager.hh"
#include "statelets/Texture2D.hh"

#include "../serialization/DeserializationInterface.hh"
#include "../serialization/StreambufAdapter.hh"
#include "../serialization/Eigen.hh"
#include "../serialization/Types.hh"
#include "../Path.hh"


using namespace peekabot;
using namespace peekabot::renderer;


PbmfLoader::PbmfLoader() throw()
{
    reset();
}


void PbmfLoader::reset() throw()
{
    m_vertices  = 0;
    m_normals   = 0;
    m_indices   = 0;
    m_uv_coords = 0;
    m_colors    = 0;

    m_vertex_count = m_tri_count = 0;
}



PbmfLoader::ResourceType *PbmfLoader::load(const Path &path)
    throw(std::runtime_error)
{
    boost::filesystem::path exp_path = path.get_expanded_path();

    TheMessageHub::instance().publish(
        DEBUG_MESSAGE,
        "Loading pbmf model file '%s'",
        exp_path.string().c_str());

    reset();

    std::ifstream ifs(exp_path.string().c_str(), std::ios::binary);

    if( !ifs.is_open() )
        throw std::runtime_error(
            "Could not open file '" + exp_path.string() + "' for reading");

    try
    {
        bool big_endian = read_preamble(ifs);

        StreambufAdapter adapter(*ifs.rdbuf());
        DeserializationInterface buf(adapter, big_endian);
        
        read_header(buf);
        read_material(buf);
        read_data(buf);

        /// \todo This leaks memory, fix!
    }
    catch(...)
    {
        ifs.close();
        throw;
    }
    
    ifs.close();

    TheMessageHub::instance().publish(
        DEBUG_MESSAGE,
        "Model successfully loaded - %d vertices, %d triangles. Bounding sphere radius=%f, pos=(%f, %f, %f)", 
        m_vertex_count, m_tri_count, m_bsphere_radius, 
        m_bsphere_center(0), m_bsphere_center(1), m_bsphere_center(2));

    
    boost::shared_ptr<VertexBuffer> vertices(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> normals(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> colors;
    boost::shared_ptr<VertexBuffer> texcoords;
    boost::shared_ptr<IndexBuffer>  indices(new IndexBuffer);

    vertices->set_data(
        m_vertices, 3*sizeof(GLfloat)*m_vertex_count);
    normals->set_data(
        m_normals, 3*sizeof(GLfloat)*m_vertex_count);
    indices->set_data(
        m_indices, 3*sizeof(GLuint)*m_tri_count);

    if( m_has_uv_coords )
    {
        texcoords = boost::shared_ptr<VertexBuffer>(
            new VertexBuffer(VertexBuffer::STATIC_DRAW));
        texcoords->set_data(
            m_uv_coords, 2*sizeof(GLuint)*m_vertex_count);
    }

    if( m_has_colors )
    {
        colors = boost::shared_ptr<VertexBuffer>(new VertexBuffer);
        colors->set_data(
            m_colors, 3*sizeof(GLfloat)*m_vertex_count);
    }
    
    TriMesh mesh(
        vertices,
        normals,
        colors,
        texcoords,
        indices,
        0, m_tri_count);

    // If it has UV-coords and a texture we should load its texture
    if( m_has_uv_coords && m_texture_filename != "" )
    {
        // Look in the model's directory for the texture before searching the
        // other search directories
        Path tmp_path(m_texture_filename);
        tmp_path.prepend_search_dir(exp_path.branch_path());
        boost::filesystem::path texture_path = tmp_path.get_expanded_path();

        TheMessageHub::instance().publish(
            DEBUG_MESSAGE,
            "Loading texture '%s'",
            texture_path.string().c_str());


        try
        {
            boost::shared_ptr<Texture> tex =
                TheTextureManager::instance().get(texture_path.string());
            mesh.get_state().set(new statelets::Texture2D(tex));
        }
        catch(std::exception &e)
        {
            TheMessageHub::instance().publish(
                ERROR_MESSAGE,
                "Failed to load texture %s with error: %s",
                texture_path.string().c_str(), e.what());
        }
    }

    MeshBased *primitive = new MeshBased(
        1, BoundingSphere(m_bsphere_center, m_bsphere_radius), 0);
    primitive->set_lod(0, 0, &mesh);

    return primitive;
}


bool PbmfLoader::read_preamble(std::ifstream &ifs)
{
    char uid[4];
    ifs.read(uid, 4);
    if( ifs.gcount() != 4 )
        throw std::runtime_error("Invalid header");
   
    if( uid[0] != 'p' || uid[1] != 'b' || uid[2] != 'm' || uid[3] != 'f' )
      throw std::runtime_error("Incorrect file format identifier");

    uint8_t is_big_endian;
    ifs.read(reinterpret_cast<char *>(&is_big_endian), 1);
    if( ifs.gcount() != 1 )
        throw std::runtime_error("Invalid header");

    if( is_big_endian != 0 && is_big_endian != 1 )
    {
        throw std::runtime_error(
            "Invalid header - LE/BE field contains invalid data");
    }

    m_version = 0;
    ifs.read(reinterpret_cast<char *>(&m_version), 1);
    if( ifs.gcount()  != 1 )
        throw std::runtime_error("Invalid header");

    if( m_version > 0 )
    {
        std::stringstream ss;
        ss << "Unsupported PBMF file format version (version " 
           << (int)m_version << "), try upgrading to a more recent release "
           << "of peekabot";
        throw std::runtime_error(ss.str());
    }

    return is_big_endian != 0;
}

void PbmfLoader::read_header(DeserializationInterface &buf)
{
    buf >> m_vertex_count
        >> m_tri_count
        >> m_has_uv_coords
        >> m_has_colors;

    // Bounding sphere
    buf >> m_bsphere_radius
        >> m_bsphere_center;

    buf >> m_disable_backface_culling
        >> m_always_draw_back_to_front;
}

void PbmfLoader::read_material(DeserializationInterface &buf)
{
    buf >> m_ambient_reflectance
        >> m_diffuse_reflectance
        >> m_specular_reflectance
        >> m_emissitivity
        >> m_shininess
        >> m_texture_filename;
}

void PbmfLoader::read_data(DeserializationInterface &buf)
{
    // Allocate buffers
    m_vertices = new GLfloat[3*m_vertex_count];
    m_normals  = new GLfloat[3*m_vertex_count];
    m_indices  = new uint32_t[3*m_tri_count];

    if( m_has_uv_coords ) 
        m_uv_coords = new GLfloat[2*m_vertex_count];
    if( m_has_colors ) 
        m_colors = new boost::uint8_t[3*m_vertex_count];

    try
    {
        buf.load_binary(m_vertices, 3*m_vertex_count*sizeof(GLfloat));
        buf.load_binary(m_normals, 3*m_vertex_count*sizeof(GLfloat));

        if( m_has_uv_coords ) 
            buf.load_binary(m_uv_coords, 2*m_vertex_count*sizeof(GLfloat));

        if( m_has_colors ) 
        {
            boost::scoped_array<float> tmp(new float[3*m_vertex_count]);
            buf.load_binary(tmp.get(), 3*m_vertex_count*sizeof(GLfloat));
            for( std::size_t i = 0; i < 3*m_vertex_count; ++i )
                m_colors[i] = (boost::uint8_t)255*tmp[i];
        }

        buf.load_binary(m_indices, 3*m_tri_count*sizeof(uint32_t));
    }
    catch(...)
    {       
        // Free resources
        delete[] m_vertices;
        delete[] m_normals;
        delete[] m_indices;

        if( m_has_uv_coords ) 
            delete[] m_uv_coords;
        if( m_has_colors ) 
            delete[] m_colors;

        throw;
    }
}

