/*
 * Copyright Staffan Gimåker 2007-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/scoped_ptr.hpp>

#include "PrimitiveFactory.hh"
#include "MeshFactory.hh"
#include "entities/MeshBased.hh"
#include "entities/Circle.hh"


using namespace peekabot;
using namespace peekabot::renderer;


PrimitiveFactory::PrimitiveFactory() throw()
{
    generate_boilerplates();
}

void PrimitiveFactory::generate_boilerplates() throw()
{
    //
    // Disc
    //
    MeshBased *disc = new MeshBased(
        4, BoundingSphere(Eigen::Vector3f(0,0,0), 1.0f), 0.1f);

    boost::scoped_ptr<Mesh> disc1(MeshFactory::get_disc_mesh( 8, 1.0));
    boost::scoped_ptr<Mesh> disc2(MeshFactory::get_disc_mesh(16, 1.0));
    boost::scoped_ptr<Mesh> disc3(MeshFactory::get_disc_mesh(32, 1.0));
    boost::scoped_ptr<Mesh> disc4(MeshFactory::get_disc_mesh(64, 1.0));
    disc->set_lod(0,   0, disc1.get());
    disc->set_lod(1,  15, disc2.get());
    disc->set_lod(2,  75, disc3.get());
    disc->set_lod(3, 400, disc4.get());

    add_boilerplate(DISC, disc);

    //
    // Cylinder
    //
    MeshBased *cylinder = new MeshBased(
        4, BoundingSphere(Eigen::Vector3f(0,0,0), 1.0f), 0.1f);

    boost::scoped_ptr<Mesh> cyl1(MeshFactory::get_cylinder_mesh( 8, 1.0, 1));
    boost::scoped_ptr<Mesh> cyl2(MeshFactory::get_cylinder_mesh(16, 1.0, 1));
    boost::scoped_ptr<Mesh> cyl3(MeshFactory::get_cylinder_mesh(32, 1.0, 1));
    boost::scoped_ptr<Mesh> cyl4(MeshFactory::get_cylinder_mesh(64, 1.0, 1));
    cylinder->set_lod(0,   0, cyl1.get());
    cylinder->set_lod(1,  20, cyl2.get());
    cylinder->set_lod(2, 150, cyl3.get());
    cylinder->set_lod(3, 450, cyl4.get());

    add_boilerplate(CYLINDER, cylinder);

    //
    // Ico-sphere
    //
    MeshBased *icosphere = new MeshBased(
        4, BoundingSphere(Eigen::Vector3f(0,0,0), 1.0f), 0.1f);

    boost::scoped_ptr<Mesh> sphere1(MeshFactory::get_icosphere_mesh(0, 1.0));
    boost::scoped_ptr<Mesh> sphere2(MeshFactory::get_icosphere_mesh(1, 1.0));
    boost::scoped_ptr<Mesh> sphere3(MeshFactory::get_icosphere_mesh(2, 1.0));
    boost::scoped_ptr<Mesh> sphere4(MeshFactory::get_icosphere_mesh(3, 1.0));
    icosphere->set_lod(0,   0, sphere1.get());
    icosphere->set_lod(1,  10, sphere2.get());
    icosphere->set_lod(2,  35, sphere3.get());
    icosphere->set_lod(3, 150, sphere4.get());

    add_boilerplate(ICOSPHERE, icosphere);

    //
    // Circle
    //
    MeshBased *circle = new Circle(
        4, BoundingSphere(Eigen::Vector3f(0,0,0), 1.0f), 0.1f);

    boost::scoped_ptr<Mesh> circle1(MeshFactory::get_circle_mesh(8, 1.0));
    boost::scoped_ptr<Mesh> circle2(MeshFactory::get_circle_mesh(16, 1.0));
    boost::scoped_ptr<Mesh> circle3(MeshFactory::get_circle_mesh(32, 1.0));
    boost::scoped_ptr<Mesh> circle4(MeshFactory::get_circle_mesh(64, 1.0));
    circle->set_lod(0,   0, circle1.get());
    circle->set_lod(1,  10, circle2.get());
    circle->set_lod(2,  70, circle3.get());
    circle->set_lod(3, 200, circle4.get());

    add_boilerplate(CIRCLE, circle);

    //
    // Cone
    //

    // bounding sphere radius = sqrt( r^2 + (h/2)^2 )
    MeshBased *cone = new MeshBased(
        6, BoundingSphere(Eigen::Vector3f(0,0,0), sqrt(1 + 1./8)), 0.1f);

    boost::scoped_ptr<Mesh> cone1(MeshFactory::get_cone_mesh(1.0, 1,   4, 0));
    boost::scoped_ptr<Mesh> cone2(MeshFactory::get_cone_mesh(1.0, 1,   8, 1));
    boost::scoped_ptr<Mesh> cone3(MeshFactory::get_cone_mesh(1.0, 1,  16, 2));
    boost::scoped_ptr<Mesh> cone4(MeshFactory::get_cone_mesh(1.0, 1,  32, 2));
    boost::scoped_ptr<Mesh> cone5(MeshFactory::get_cone_mesh(1.0, 1,  64, 3));
    boost::scoped_ptr<Mesh> cone6(MeshFactory::get_cone_mesh(1.0, 1, 256, 3));
    cone->set_lod(0,   0, cone1.get());
    cone->set_lod(1,  10, cone2.get());
    cone->set_lod(2,  20, cone3.get());
    cone->set_lod(3, 100, cone4.get());
    cone->set_lod(4, 200, cone5.get());
    cone->set_lod(5, 400, cone6.get());

    add_boilerplate(CONE, cone);
}


PrimitiveFactory::~PrimitiveFactory() throw()
{
    for( PrimitiveMap::iterator it = m_boilerplates.begin();
         it != m_boilerplates.end(); ++it )
    {
        delete it->second;
    }

    m_boilerplates.clear();
}


MeshBased *PrimitiveFactory::get_disc() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(DISC);
    return it->second->clone();
}

MeshBased *PrimitiveFactory::get_cylinder() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(CYLINDER);
    return it->second->clone();
}

MeshBased *PrimitiveFactory::get_icosphere() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(ICOSPHERE);
    return it->second->clone();
}

MeshBased *PrimitiveFactory::get_circle() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(CIRCLE);
    return it->second->clone();
}

MeshBased *PrimitiveFactory::get_cone() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(CONE);
    return it->second->clone();
}

/*MeshBased *PrimitiveFactory::get_cube() const throw()
{
    PrimitiveMap::const_iterator it = m_boilerplates.find(CUBE);
    return it->second->clone();
}*/

void PrimitiveFactory::add_boilerplate(
    MeshType type, MeshBased *boilerplate) throw()
{
    m_boilerplates.insert(std::make_pair(type, boilerplate));
}
