/*
 * Copyright Staffan Gimåker 2006-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <sstream>
#include <GL/glew.h>
#include <Eigen/LU>

#include "RenderEngine.hh"
#include "RenderTraverser.hh"
#include "entities/Camera.hh"
#include "../MessageHub.hh"
#include "Statelet.hh"
#include "statelets/Texture2D.hh"
#include "statelets/Transformation.hh"


using namespace peekabot;
using namespace peekabot::renderer;


RenderEngine::RenderEngine()
    : m_bg_color(1, 1, 1),
      m_selected_color(0, 0.3, 1),
      m_ancestor_selected_color(0.5, 0.75, 1)
{
}


RenderEngine::~RenderEngine()
{
}


void RenderEngine::init()
{
    TheMessageHub::instance().publish(
        DEBUG_MESSAGE, 
        "Initializing GLEW...");
   
    // Setting this magical, undocumented flag will enable GLEW to report
    // that OpenGL >1.2 exists on Macs with crippled hardware (e.g. the
    // Mac Mini, which only supports OpenGL 1.2 hardware accelerated).
    //
    // Note that the extra set of GL functions it reports are OK after setting
    // this flag are software rendered through their LLVM stack!
    // But software rendering is better than nothing I guess?
    glewExperimental = GL_TRUE;

    // Initialize GLEW (Note that it is IMPORTANT that we have a valid
    // OpenGL context before trying to initialize GLEW).
    GLenum err = glewInit();
    if( GLEW_OK != err )
    {
        // Problem: glewInit failed, something is seriously wrong.
        throw std::runtime_error((const char *)glewGetErrorString(err));
    }


    // Log GL version, vendor, renderer and extensions
    TheMessageHub::instance().publish(
        LOG_MESSAGE, 
        "OpenGL vendor: %s", glGetString(GL_VENDOR));

    TheMessageHub::instance().publish(
        LOG_MESSAGE, 
        "OpenGL renderer: %s", glGetString(GL_RENDERER));

    TheMessageHub::instance().publish(
        LOG_MESSAGE, 
        "OpenGL version: %s", glGetString(GL_VERSION));

    TheMessageHub::instance().publish(
        LOG_MESSAGE, 
        "OpenGL extensions: %s", glGetString(GL_EXTENSIONS));


    // We depend on glBlendFunc and glBlendColor - they're not available
    // until OpenGL 1.4, or through the ARB_imaging extension available in 
    // OpenGL 1.2. Make sure we have either one or abort!
    if( !(GLEW_VERSION_1_4 || (GLEW_ARB_imaging && GLEW_VERSION_1_2)) )
    {
        throw std::runtime_error(
            "OpenGL 1.4 or OpenGL 1.2 and the ARB_imaging "
            "extension are required to run peekabot");
    }

    // If we don't have VBO support, print a nice little informational
    // message saying so...
    if( !GLEW_ARB_vertex_buffer_object )
    {
        TheMessageHub::instance().publish(
            INFO_MESSAGE, 
            "VBO support not available");
    }

    State baseline;

    // Enable lighting by default
    baseline.set(new statelets::Lighting(true));
    // Disable two-sided lighting by default
    baseline.set(new statelets::LightModel(false));
    // Disable blending by default
    baseline.set(new statelets::BlendMode(GL_ONE, GL_ZERO));
    // Enable backface culling by default
    baseline.set(new statelets::Culling(true));
    // Enable depth testing by default, using the GL_LESS as the depth func
    baseline.set(new statelets::DepthTest(statelets::DepthTest::LESS));
    // Enable depth mask writing by default
    baseline.set(new statelets::DepthMask(true));
    // Enable diffuse reflectance material color tracking
    baseline.set(new statelets::ColorMaterial(true));
    baseline.set(new statelets::CullFrontFaces(false));
    baseline.set(new statelets::PointParams(3, true));
    baseline.set(new statelets::LineParams(1, false, 0, 0));
    baseline.set(new statelets::PolygonMode(
                     statelets::PolygonMode::FRONT_AND_BACK,
                     statelets::PolygonMode::FILL));
    baseline.set(new statelets::StencilTest(false));
    baseline.set(new statelets::StencilFunc(
                     GL_ALWAYS, 1, static_cast<GLuint>(-1),
                     GL_KEEP, GL_KEEP, GL_REPLACE));
    // No texture by default
    baseline.set(new statelets::Texture2D());

    // Default to disabling all vertex buffers
    baseline.set(new statelets::VertexPointer());
    baseline.set(new statelets::NormalPointer());
    baseline.set(new statelets::ColorPointer());
    baseline.set(
        new statelets::TexCoordPointer(boost::shared_ptr<VertexBuffer>()));

    // Default material:
    baseline.set(
        new statelets::Material(
            RGBAColor(0.8,0.8,0.8,1),
            RGBAColor(0.2,0.2,0.2,1),
            RGBAColor(0,0,0,1),
            20));

    State::set_baseline(baseline);
}


void RenderEngine::init_context()
{
    //
    // Dial in the right settings for a few states that aren't
    // handled by the statelet system (at the moment, at least)
    //
    glShadeModel(GL_SMOOTH);
    glClearStencil(0);
    glEnable(GL_LIGHT0);

    // We need this since we do scaling with matrices.
    glEnable(GL_NORMALIZE);

    glEnable(GL_DEPTH_TEST);
    glEnable(GL_DITHER);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
    //glHint(GL_POLYGON_SMOOTH_HINT, GL_FASTEST);

    glLightModelf(GL_LIGHT_MODEL_LOCAL_VIEWER, 1);
    glLightModeli(GL_LIGHT_MODEL_COLOR_CONTROL, GL_SEPARATE_SPECULAR_COLOR);
}


void RenderEngine::render(
    unsigned int w, unsigned int h,
    Camera &camera,
    const bool *visible_layers)
{
    if( w < 1 || h < 1 )
        return;

    assert( visible_layers != 0 );

    // Setup viewport
    camera.set_viewport(0, 0, w, h);


    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);


    // Clear color, depth and stencil buffers
    glClearColor(m_bg_color.r, m_bg_color.g, m_bg_color.b, 0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

    // Set up the viewing transform and projection matrix
    use_camera(camera);


    // We're not doing picking, ignore object names
    State::OverrideSet os(State::get_override_set());
    os.insert(statelets::Name::get_type());
    State::push_baseline();
    State::set_baseline(State::get_baseline_state(), os);
    // Unconditionally reapply the baseline state
    State::apply_baseline();

    RenderTraverser rt(camera, visible_layers);
    m_scene_manager.traverse(camera, rt);

    rt.render_opaque();
    rt.render_translucent();
    rt.render_outlines(m_selected_color, m_ancestor_selected_color);
    rt.render_tripods();

    State::pop_baseline();
}


ObjectID RenderEngine::pick(
    unsigned int w, unsigned int h,
    Camera &camera,
    const bool *visible_layers,
    int x, int y, float sensitivity)
{
    if( w < 1 || h < 1 )
        return 0;

    assert( visible_layers != 0 );

    // Setup viewport
    camera.set_viewport(0, 0, w, h);


    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);


    // Clear color and depth buffers
    glClearColor(0, 0, 0, 0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // Set up the viewing transform and projection matrix
    use_camera(camera);


    State::OverrideSet os(State::get_override_set());
    os.insert(statelets::Texture2D::get_type());
    os.insert(statelets::Material::get_type());
    os.insert(statelets::ColorMaterial::get_type());
    os.insert(statelets::Color::get_type());
    os.insert(statelets::Alpha::get_type());
    os.insert(statelets::BlendMode::get_type());
    os.insert(statelets::ColorPointer::get_type());
    os.insert(statelets::TexCoordPointer::get_type());
    os.insert(statelets::NormalPointer::get_type());
    os.insert(statelets::Lighting::get_type());
    os.insert(statelets::StencilTest::get_type());
    os.insert(statelets::StencilFunc::get_type());
    os.insert(statelets::DepthTest::get_type());

    State::push_baseline();
    State baseline = State::get_baseline_state();
    // Make sure lighting is disabled
    baseline.set(new statelets::Lighting(false));
    // Everything not explicitly named should be assigned the invalid 0 ID
    baseline.set(new statelets::Name(0));
    State::set_baseline(baseline, os);
    // Unconditionally reapply the baseline state
    State::apply_baseline();

    RenderTraverser rt(camera, visible_layers);
    m_scene_manager.traverse(camera, rt);

    rt.render_opaque();
    rt.render_translucent();

    State::pop_baseline();


    // Extract the select object from the rendered output
    boost::uint8_t pixbuf[w*h*3];
    glPixelStorei(GL_PACK_ALIGNMENT, 1);
    glReadBuffer(GL_BACK);
    glReadPixels(0, 0, w, h, GL_RGB, GL_UNSIGNED_BYTE, pixbuf);

    return (pixbuf[3*(w*y + x) + 0]<<16 |
            pixbuf[3*(w*y + x) + 1]<<8 |
            pixbuf[3*(w*y + x) + 2]);
}


void RenderEngine::use_camera(Camera &camera) const throw()
{
    // Load the projection matrix
    glMatrixMode(GL_PROJECTION);
    camera.apply_projection_matrix();

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    Eigen::Matrix4f M;
    M.col(0) = Eigen::Vector4f( 0,  0, -1,  0);
    M.col(1) = Eigen::Vector4f(-1,  0,  0,  0);
    M.col(2) = Eigen::Vector4f( 0,  1,  0,  0);
    M.col(3) = Eigen::Vector4f( 0,  0,  0,  1);

    M = M * camera.get_viewer_mtow().inverse(Eigen::Affine);
    glLoadMatrixf(reinterpret_cast<const GLfloat *>(M.data()));

    // Set the light direction in *world* coordinates
    Eigen::Vector4f pos(1, -1, 2, 0);
    glLightfv(GL_LIGHT0, GL_POSITION, &pos(0));

    Eigen::Vector4f ambient(0.1, 0.1, 0.1, 1);
    glLightfv(GL_LIGHT0, GL_AMBIENT, &ambient(0));

    //Vector4r specular(0.5, 0.5, 0.5, 1);
    //glLightfv(GL_LIGHT0, GL_SPECULAR, &specular(0));
    //
    //Vector4r diffuse(0.9, 0.9, 0.9, 1);
    //glLightfv(GL_LIGHT0, GL_DIFFUSE, &diffuse(0));
}


void RenderEngine::set_background_color(const RGBColor &color)
{
    m_bg_color = color;
}


void RenderEngine::set_selected_color(const RGBColor &color)
{
    m_selected_color = color;
}


void RenderEngine::set_ancestor_selected_color(const RGBColor &color)
{
    m_ancestor_selected_color = color;
}
