/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/bind.hpp>
#include <boost/foreach.hpp>

#include "SceneManager.hh"
#include "SceneTraverser.hh"
#include "Entity.hh"
#include "ObjectEntity.hh"
#include "CullableEntity.hh"


using namespace peekabot;
using namespace peekabot::renderer;


SceneManager::SceneManager()
    : m_octree(100.0f, 2.0f) // TODO: use config values
{
}


SceneManager::~SceneManager()
{
    // TODO: disconnect entities ?
}


void SceneManager::add_entity(const Entity *entity)
{
    const CullableEntity *cullable =
        dynamic_cast<const CullableEntity *>(entity);

    const ObjectEntity *object =
        dynamic_cast<const ObjectEntity *>(entity);

    if( object )
    {
        object->m_on_visibility_set.connect(
            boost::bind(&SceneManager::on_visibility_set, this, _1));

        object->m_on_selected_set.connect(
            boost::bind(&SceneManager::on_selected_set, this, _1));

        if( cullable && object->is_visible() )
        {
            m_octree.add(cullable);

            cullable->m_on_world_bv_set.connect(
                boost::bind(&SceneManager::on_world_bv_set, this, _1));
        }
        else if( object->is_visible() )
        {
            m_non_cullable.insert(entity);
        }
    }
    else
    {
        m_non_cullable.insert(entity);
    }
}


void SceneManager::remove_entity(const Entity *entity)
{
    const CullableEntity *cullable =
        dynamic_cast<const CullableEntity *>(entity);

    const ObjectEntity *object =
        dynamic_cast<const ObjectEntity *>(entity);

    if( object )
    {
        object->m_on_visibility_set.disconnect(
            boost::bind(&SceneManager::on_visibility_set, this, _1));

        object->m_on_selected_set.disconnect(
            boost::bind(&SceneManager::on_selected_set, this, _1));

        if( object->is_selected() )
            m_selected.erase(object);

        if( cullable && object->is_visible() )
        {
            m_octree.remove(cullable);

            cullable->m_on_world_bv_set.disconnect(
                boost::bind(&SceneManager::on_world_bv_set, this, _1));
        }
        else if( object->is_visible() )
        {
            m_non_cullable.erase(object);
        }
    }
    else
    {
        m_non_cullable.erase(entity);
    }
}


void SceneManager::on_visibility_set(const ObjectEntity *entity)
{
    const CullableEntity *cullable(
        dynamic_cast<const CullableEntity *>(entity));

    if( cullable )
    {
        if( entity->is_visible() )
        {
            m_octree.add(cullable);

            cullable->m_on_world_bv_set.connect(
                boost::bind(&SceneManager::on_world_bv_set, this, _1));
        }
        else
        {
            m_octree.remove(cullable);

            cullable->m_on_world_bv_set.disconnect(
                boost::bind(&SceneManager::on_world_bv_set, this, _1));
        }
    }
    else
    {
        if( entity->is_visible() )
        {
            m_non_cullable.insert(entity);
        }
        else
        {
            m_non_cullable.erase(entity);
        }
    }
}


void SceneManager::on_selected_set(const ObjectEntity *entity)
{
    if( entity->is_selected() )
    {
        m_selected.insert(entity);
    }
    else
    {
        m_selected.erase(entity);
    }
}


void SceneManager::on_world_bv_set(const CullableEntity *entity)
{
    m_octree.update(entity);
}


void SceneManager::traverse(const Camera &camera, SceneTraverser &traverser)
{
    // 1. Emit cullable entities from the octree
    m_octree.traverse(camera, traverser);

    // 2. Emit non-cullable entities
    BOOST_FOREACH( const Entity *e, m_non_cullable )
    {
        traverser.traverse(e);
    }
}


const SceneManager::SelectedSet &SceneManager::get_selected() const throw()
{
    return m_selected;
}
