/*
 * Copyright Staffan Gimåker 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <iostream>
#include <iterator>
#include <algorithm>
#include <GL/glew.h>

#include "State.hh"


using namespace peekabot;
using namespace peekabot::renderer;


State::Baseline State::ms_baseline;
State::BaselineStack State::ms_baseline_stack;
State State::ms_current_state;
State::BaselineDelta State::ms_baseline_delta;


State::State() throw()
{
}


State::State(const State &state) throw()
{
    for( Statelets::const_iterator it = state.m_statelets.begin();
         it != state.m_statelets.end(); ++it )
    {
        if( it->second )
            set(it->second->clone());
    }
}


State &State::operator=(const State &state) throw()
{
    m_statelets.clear();

    for( Statelets::const_iterator it = state.m_statelets.begin();
         it != state.m_statelets.end(); ++it )
    {
        if( it->second )
            set(it->second->clone());
    }

    return *this;
}


/*State State::clone() const throw()
{
    State dup;

    for( Statelets::const_iterator it = m_statelets.begin();
         it != m_statelets.end(); ++it )
    {
        dup.set(it->second->clone());
    }

    return dup;
}*/


template<> void State::set<Statelet>(boost::shared_ptr<Statelet> s) throw()
{
    if( !s )
    {
        return;
    }
    else
    {
        m_statelets[s->_get_type()] = s;
    }
}


void State::set(Statelet *s) throw()
{
    if( s )
        set(boost::shared_ptr<Statelet>(s));
}


void State::apply(const State *template_state) const throw()
{
    /***********************************************
     *                                             *
     *        IF YOU MAKE ANY CHANGES HERE,        *
     *          PROFILE BEFORE YOU COMMIT!         *
     *                                             *
     ***********************************************/

    for( BaselineDelta::iterator it = ms_baseline_delta.begin();
         it != ms_baseline_delta.end(); )
    {
        BaselineDelta::iterator curr_it = it;
        ++it;

        // If A) not applied in state, B) nor in the template state or C) in 
        // the state/template but being overridden, revert to baseline:
        if( (m_statelets.find(*curr_it) == m_statelets.end() && // A)
             (!template_state ||                               // B)
              template_state->m_statelets.find(*curr_it) == template_state->m_statelets.end())
             ) || ms_baseline.second.count(*curr_it) != 0 )     // C)
        {
            // Apply baseline statelet and set current to curr_it
            boost::shared_ptr<Statelet> &base_s = 
                ms_baseline.first.m_statelets[*curr_it];

            boost::shared_ptr<Statelet> &curr_s = 
                ms_current_state.m_statelets[*curr_it];

            if( curr_s )
                curr_s->deactivate();

            if( base_s )
                base_s->activate();
            curr_s = base_s;

            ms_baseline_delta.erase(curr_it);
        }
    }


    for( Statelets::const_iterator it = m_statelets.begin();
         it != m_statelets.end(); ++it )
    {
        boost::shared_ptr<Statelet> &curr_s = 
            ms_current_state.m_statelets[it->first];

        if( (!curr_s || curr_s->get_impl() != it->second->get_impl()) &&
            ms_baseline.second.count(it->first) == 0 )
        {
            if( curr_s )
                curr_s->deactivate();

            it->second->activate();
            curr_s = it->second;

            if( ms_baseline.first.m_statelets[it->first] )
                ms_baseline_delta.insert(it->first);
        }
    }


    if( template_state )
    {
        for( Statelets::const_iterator it = template_state->m_statelets.begin();
             it != template_state->m_statelets.end(); ++it )
        {
            boost::shared_ptr<Statelet> &curr_s = 
                ms_current_state.m_statelets[it->first];
            
            if( m_statelets.find(it->first) == m_statelets.end() && // in state?
                (!curr_s || curr_s->get_impl() != it->second->get_impl()) &&
                ms_baseline.second.count(it->first) == 0 ) // overridden?
            {
                if( curr_s )
                    curr_s->deactivate();
                
                it->second->activate();
                curr_s = it->second;
                
                if( ms_baseline.first.m_statelets[it->first] )
                    ms_baseline_delta.insert(it->first);
            }
        }
    }
}






void State::apply_baseline() throw()
{
    // Deactivate current state
    for( Statelets::iterator it = ms_current_state.m_statelets.begin();
         it != ms_current_state.m_statelets.end(); ++it )
        if( it->second )
            it->second->deactivate();

    for( Statelets::iterator it = ms_baseline.first.m_statelets.begin();
         it != ms_baseline.first.m_statelets.end(); ++it )
        if( it->second )
            it->second->activate();

    ms_baseline_delta.clear();
    ms_current_state = ms_baseline.first;
}



const std::pair<State, State::OverrideSet> &State::get_baseline() throw()
{
    return ms_baseline;
}


const State &State::get_baseline_state() throw()
{
    return ms_baseline.first;
}


const State::OverrideSet &State::get_override_set() throw()
{
    return ms_baseline.second;
}


void State::set_baseline(
    const State &state, 
    const OverrideSet &os) throw()
{
    ms_baseline = std::make_pair(state, os);

    // Apply new baseline unonditionally:
    apply_baseline();
}


void State::push_baseline() throw()
{
    // Push previous baseline state
    ms_baseline_stack.push(ms_baseline);
}


void State::pop_baseline() throw()
{
    ms_baseline = ms_baseline_stack.top();
    ms_baseline_stack.pop();

    // Apply new baseline unonditionally:
    apply_baseline();
}
