/*
 * Copyright Staffan Gimåker 2007-2008, 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_STATELET_HH_INCLUDED
#define PEEKABOT_RENDERER_STATELET_HH_INCLUDED


#include <GL/glew.h>
#include <cassert>
#include <map>
#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>

#include "VertexBuffer.hh"
#include "Device.hh"


namespace peekabot
{
namespace renderer
{

    namespace statelets
    {
        typedef unsigned int StateletType;


        /**
         * \brief The interface for the implementation classes realizing the
         * functionality of Statelets.
         *
         * A Statelet implementation has only a few responsibilities:
         * \li Provide an activate() method that activates the statelet, i.e. 
         *     does what the statelet should do (e.g. toggle lighting).
         * \li \e Optional: Provide a deactivate() method that performs any
         *     necessary clean up.
         */
        class StateletImpl
        {
        public:
            virtual ~StateletImpl() {}

            virtual void activate() throw() = 0;

            virtual void deactivate() throw() {}
        };




        /**
         * \brief A statelet encapsulates a small part of an OpenGL rendering 
         * state, e.g. whether lighting is turning on, or the active stencil 
         * function.
         *
         * \sa TypedStatelet
         */
        class Statelet
        {
        public:
            Statelet() {}

            Statelet(const Statelet &s) : m_impl(s.m_impl) {}

            virtual ~Statelet() {}

            virtual Statelet *clone() const throw() = 0;
           
            inline void activate() throw() { m_impl->activate(); }

            inline void deactivate() throw() { m_impl->deactivate(); }

            inline const boost::shared_ptr<StateletImpl> &get_impl() 
                const throw() { return m_impl; }

            virtual StateletType _get_type() const throw() = 0;

        protected:
            Statelet(boost::shared_ptr<StateletImpl> impl);

            inline void set_impl(boost::shared_ptr<StateletImpl> impl) throw()
            {
                m_impl = impl;
            }

        protected:
            boost::shared_ptr<StateletImpl> m_impl;

            static StateletType ms_next_available_statelet_type;
        };




        /**
         * \brief This extension to the Statelet class provides a type 
         * identifier that uniquely identifies its type (as in class).
         *
         * The rationale is that we want to be able to put statelets in maps 
         * (and sets too) and easily be able to access and overwrite a statelet 
         * based on its type.
         *
         * Note that its functionality cannot be put in the Statelet base 
         * class. Why? The class providing the get_type() method must be 
         * templatized for obvious reasons, which conflicts with the implicit
         * requirement of being able to keep pointers to the statelet base 
         * class.
         */
        template<typename T>
        class TypedStatelet : public Statelet
        {
        protected:
            TypedStatelet(boost::shared_ptr<StateletImpl> impl) 
                : Statelet(impl) {}

            TypedStatelet(const TypedStatelet<T> &s) : Statelet(s) {}

        public:
            virtual TypedStatelet<T> *clone() const throw()
            {
                return new T(*dynamic_cast<const T *>(this));
            }

            inline static StateletType get_type() throw()
            {
                static StateletType type = ms_next_available_statelet_type++;
                return type;
            }

            virtual StateletType _get_type() const throw()
            {
                return get_type();
            }
        };



        typedef void (* ToggleFunc)(bool);

        template<ToggleFunc toggle_func, bool default_val>
        class BinaryStatelet : public TypedStatelet<BinaryStatelet<toggle_func, default_val> >
        {
            struct EnableImpl : public StateletImpl
            {
                virtual void activate() throw() { toggle_func(true); }
            };
            
            struct DisableImpl : public StateletImpl
            {
                virtual void activate() throw() { toggle_func(false); }
            };

            static boost::shared_ptr<StateletImpl> get_impl(bool enable) throw()
            {
                static boost::shared_ptr<StateletImpl> s_on(new EnableImpl());
                static boost::shared_ptr<StateletImpl> s_off(new DisableImpl());
                return enable ? s_on : s_off;
            }

        public:
            BinaryStatelet(bool enable) throw()
                : TypedStatelet<BinaryStatelet<toggle_func, default_val> >(
                    get_impl(enable)) {}

            BinaryStatelet() throw()
                : TypedStatelet<BinaryStatelet<toggle_func, default_val> >(
                    get_impl(default_val)) {}

            bool is_enabled() const throw()
            {
                return TypedStatelet<BinaryStatelet<toggle_func, default_val> >::m_impl == get_impl(true);
            }
            
            void toggle(bool enable) throw()
            {
                TypedStatelet<BinaryStatelet<toggle_func, default_val> >::m_impl = get_impl(enable); 
            }

            void enable() throw()
            {
                TypedStatelet<BinaryStatelet<toggle_func, default_val> >::m_impl = get_impl(true);
            }

            void disable() throw()
            {
                TypedStatelet<BinaryStatelet<toggle_func, default_val> >::m_impl = get_impl(false);
            }
        };

     
        /**
         * \brief A statelet that toggles lighting.
         */
        typedef BinaryStatelet<&Device::toggle_lighting, true> Lighting;

        /**
         * \brief Toggles stencil testing.
         *
         * \sa StencilFunc
         */
        typedef BinaryStatelet<&Device::toggle_stencil_test, false> StencilTest;


        /**
         * \brief Toggles depth buffer writing.
         */
        typedef BinaryStatelet<&Device::toggle_depth_mask, true> DepthMask;


        /**
         * \brief Toggles backface culling.
         *
         * Could easily be extended to allow front-face culling and other more
         * esoteric operations.
         */
        typedef BinaryStatelet<&Device::toggle_culling, true> Culling;

        /**
         * \brief Toggles material color tracking.
         *
         * \sa Color
         */
        typedef BinaryStatelet<&Device::toggle_color_material, true> ColorMaterial;

        typedef BinaryStatelet<&Device::toggle_front_face_culling, false> CullFrontFaces;




        /**
         * \brief A statelet that toggles two-sided lighting.
         */
        class LightModel : public TypedStatelet<LightModel>
        {
            class LightModelImpl : public StateletImpl
            {
                bool m_enable_two_sided_lighting;
            public:
                LightModelImpl(bool enable_two_sided_lighting) throw()
                    : m_enable_two_sided_lighting(enable_two_sided_lighting)
                {
                }

                virtual void activate() throw()
                {
                    if( m_enable_two_sided_lighting )
                        glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
                    else
                        glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
                }
            };


            boost::shared_ptr<StateletImpl>
            get_impl(bool enable) const throw()
            {
                if( enable )
                {
                    static boost::shared_ptr<StateletImpl> on(
                        new LightModelImpl(enable));
                    return on;
                }
                else
                {
                    static boost::shared_ptr<StateletImpl> off(
                        new LightModelImpl(enable));
                    return off;
                }
            }

        public:
            LightModel(bool enable_two_sided_lighting)
                : TypedStatelet<LightModel>(get_impl(enable_two_sided_lighting)) {}

            LightModel()
                : TypedStatelet<LightModel>(get_impl(false)) {}
        };



        /**
         * \brief A statelet that toggles and controls blending.
         */
        class BlendMode : public TypedStatelet<BlendMode>
        {
            class BlendModeImpl : public StateletImpl
            {
                GLenum m_src_factor;
                GLenum m_dst_factor;
            public:
                BlendModeImpl(GLenum src_factor,
                              GLenum dst_factor) throw()
                    : m_src_factor(src_factor),
                      m_dst_factor(dst_factor) {}

                virtual void activate() throw()
                {
                    if( m_src_factor == GL_ONE && m_dst_factor == GL_ZERO )
                        glDisable(GL_BLEND);
                    else
                    {
                        glBlendFunc(m_src_factor, m_dst_factor);
                        glEnable(GL_BLEND);
                    }
                }
            };

        public:
            BlendMode(GLenum src_factor,
                      GLenum dst_factor) throw()
                : TypedStatelet<BlendMode>(
                    boost::shared_ptr<StateletImpl>(
                        new BlendModeImpl(src_factor, dst_factor))) {}

            BlendMode() throw()
                : TypedStatelet<BlendMode>(
                    boost::shared_ptr<StateletImpl>(
                        new BlendModeImpl(GL_ONE, GL_ZERO))) {}
        };



        /**
         * \brief A statelet that toggles the alpha value used for blending.
         * 
         * Actually, the entire blending color is controlled but only the
         * alpha taken as a paramater.
         * If blending is disabled, this statelet will have no effect.
         *
         * \sa BlendMode
         */
        class Alpha : public TypedStatelet<Alpha>
        {
            class AlphaImpl : public StateletImpl
            {
                float m_alpha;
            public:
                AlphaImpl(float alpha) throw() : m_alpha(alpha) {}

                virtual void activate() throw()
                {
                    glBlendColor(1, 1, 1, m_alpha);
                }

                float get_alpha() const throw() { return m_alpha; }
            };

            boost::shared_ptr<StateletImpl>
            get_impl(float alpha) const throw()
            {
                static boost::shared_ptr<StateletImpl> alpha_tab[101];

                int idx = (int)roundf(alpha*100);

                if( idx > 100 )
                    idx = 100;
                else if(idx < 0)
                    idx = 0;

                if( !alpha_tab[idx] )
                    alpha_tab[idx] = boost::shared_ptr<StateletImpl>(
                        new AlphaImpl(idx/100.0f));

                return alpha_tab[idx];
            }


        public:
            Alpha(float alpha) throw()
                : TypedStatelet<Alpha>(
                    boost::shared_ptr<StateletImpl>(get_impl(alpha))) {}

            Alpha() throw()
                : TypedStatelet<Alpha>(
                    boost::shared_ptr<StateletImpl>(get_impl(1.0f))) {}

            void set(float alpha) throw()
            {
                m_impl = get_impl(alpha);
            }

            float get() const throw()
            {
                return boost::dynamic_pointer_cast<AlphaImpl>(m_impl)->get_alpha();
            }
        };



        /**
         * \brief Controls all material properties except diffuse color, e.g. 
         * shininess and specular reflectance.
         *
         * \sa Color
         */
        class Material : public TypedStatelet<Material>
        {
            class MaterialImpl : public StateletImpl
            {
                float m_specular_reflectance[4];
                float m_ambient_reflectance[4];
                float m_emissitivity[4];
                float m_shininess;
            public:
                MaterialImpl(
                    const RGBAColor &specular_reflectance,
                    const RGBAColor &ambient_reflectance,
                    const RGBAColor &emissitivity,
                    float shininess) throw()
                    : m_shininess(shininess)
                {
                    m_specular_reflectance[0] = specular_reflectance.r;
                    m_specular_reflectance[1] = specular_reflectance.g;
                    m_specular_reflectance[2] = specular_reflectance.b;
                    m_specular_reflectance[3] = specular_reflectance.a;
                    
                    m_ambient_reflectance[0] = ambient_reflectance.r;
                    m_ambient_reflectance[1] = ambient_reflectance.g;
                    m_ambient_reflectance[2] = ambient_reflectance.b;
                    m_ambient_reflectance[3] = ambient_reflectance.a;
                    
                    m_emissitivity[0] = emissitivity.r;
                    m_emissitivity[1] = emissitivity.g;
                    m_emissitivity[2] = emissitivity.b;
                    m_emissitivity[3] = emissitivity.a;
                }

                virtual void activate() throw()
                {
                    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, m_specular_reflectance);
                    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, m_ambient_reflectance);
                    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, m_emissitivity);
                    glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, m_shininess);
                }
            };


        public:
            Material(
                const RGBAColor &specular_reflectance,
                const RGBAColor &ambient_reflectance,
                const RGBAColor &emissitivity,
                float shininess) throw()
                : TypedStatelet<Material>(
                    boost::shared_ptr<StateletImpl>(
                        new MaterialImpl(
                            specular_reflectance,
                            ambient_reflectance,
                            emissitivity,
                            shininess))) {}

            Material() throw()
                : TypedStatelet<Material>(
                    boost::shared_ptr<StateletImpl>(
                        new MaterialImpl(
                            RGBAColor(1.0, 1.0, 1.0, 1),
                            RGBAColor(0.2, 0.2, 0.2, 1),
                            RGBAColor(0.0, 0.0, 0.0, 1),
                            20))) {}
        };


        /**
         * \brief Controls the diffuse and ambient reflectance.
         *
         * In reality, it could track specular reflectance etc. as well, if
         * the color material mode was set appropriately. But we shall assume
         * it is always left at its default of tracking the diffuse and ambient
         * reflectance.
         *
         * \sa Material, ColorMaterial
         */
        class Color : public TypedStatelet<Color>
        {
            class ColorImpl : public StateletImpl
            {
                RGBColor m_color;
            public:
                ColorImpl(const RGBColor &color) throw() : m_color(color)
                {
                }

                virtual void activate() throw()
                {
                    //glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, m_diffuse_reflectance);
                    
                    // Material only affects FACES (not lines and points), which is 
                    // really lame - thus we have to call glColor as well, in case
                    // the material is affecting a line/point
                    glColor3f(m_color.r, m_color.g, m_color.b);
                }

                const RGBColor &value() const throw() { return m_color; }

                RGBColor &value() throw() { return m_color; }
            };

        public:
            Color(const RGBColor &color) throw()
                : TypedStatelet<Color>(
                    boost::shared_ptr<StateletImpl>(
                        new ColorImpl(color))) {}

            Color() throw()
                : TypedStatelet<Color>(
                    boost::shared_ptr<StateletImpl>(
                        new ColorImpl(
                            RGBColor(0.8, 0.8, 0.8)))) {}


            const RGBColor &get() const throw()
            {
                return boost::dynamic_pointer_cast<ColorImpl>(m_impl)->value();
            }

            void set(const RGBColor &color) throw()
            {
                set_impl(boost::shared_ptr<StateletImpl>(new ColorImpl(color)));
            }
        };



        /**
         * \brief Controls the active stencil function.
         *
         * \sa StencilTest
         */
        class StencilFunc : public TypedStatelet<StencilFunc>
        {
            class StencilFuncImpl : public StateletImpl
            {
                GLenum m_stencil_func;
                GLint  m_stencil_func_ref;
                GLuint m_stencil_func_mask;
                
                GLenum m_stencil_op_ref;
                GLenum m_stencil_op_zfail;
                GLenum m_stencil_op_zpass;
            public:
                StencilFuncImpl(GLenum func,
                                GLint func_ref,
                                GLuint func_mask,
                                GLenum op_ref,
                                GLenum op_zfail,
                                GLenum op_zpass)
                    : m_stencil_func(func),
                      m_stencil_func_ref(func_ref),
                      m_stencil_func_mask(func_mask),
                      m_stencil_op_ref(op_ref),
                      m_stencil_op_zfail(op_zfail),
                      m_stencil_op_zpass(op_zpass)
                {
                }
                    
                virtual void activate() throw()
                {
                    glStencilFunc(
                        m_stencil_func, m_stencil_func_ref, m_stencil_func_mask);
                    glStencilOp(
                        m_stencil_op_ref, m_stencil_op_zfail, m_stencil_op_zpass);
                }
            };

        public:
            StencilFunc(GLenum func,
                        GLint func_ref,
                        GLuint func_mask,
                        GLenum op_ref,
                        GLenum op_zfail,
                        GLenum op_zpass) throw()
                : TypedStatelet<StencilFunc>(
                    boost::shared_ptr<StateletImpl>(
                        new StencilFuncImpl(
                            func, func_ref, func_mask,
                            op_ref, op_zfail, op_zpass))) {}

            StencilFunc() throw()
                : TypedStatelet<StencilFunc>(
                    boost::shared_ptr<StateletImpl>(
                        new StencilFuncImpl(
                            GL_NOTEQUAL, 1, static_cast<GLuint>(-1),
                            GL_KEEP, GL_KEEP, GL_REPLACE))) {}
        };


        /**
         * \brief Controls the depth test operation.
         */
        class DepthTest : public TypedStatelet<DepthTest>
        {
        public:
            enum DepthFunc
            {
                NEVER = GL_NEVER,
                LESS = GL_LESS,
                EQUAL = GL_EQUAL,
                LEQUAL = GL_LEQUAL,
                GREATER = GL_GREATER,
                NOTEQUAL = GL_NOTEQUAL,
                GEQUAL = GL_GEQUAL,
                ALWAYS = GL_ALWAYS
            };


        private:
            class DepthTestImpl : public StateletImpl
            {
                DepthFunc m_depth_func;
            public:
                DepthTestImpl(DepthFunc depth_func) throw()
                    : m_depth_func(depth_func) {}

                virtual void activate() throw()
                {
                    glDepthFunc(m_depth_func); 
                }
            };


            boost::shared_ptr<StateletImpl>
            get_impl(DepthFunc depth_func) const throw()
            {
                switch(depth_func)
                {
                    case NEVER:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case LESS:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }
                        
                    case EQUAL:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case LEQUAL:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case GREATER:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case NOTEQUAL:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case GEQUAL:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    case ALWAYS:
                    {
                        static boost::shared_ptr<StateletImpl> impl(
                            new DepthTestImpl(depth_func));
                        return impl;
                    }

                    default:
                        return boost::shared_ptr<StateletImpl>();
                }
            }

        public:
            DepthTest(DepthFunc depth_func) throw()
                : TypedStatelet<DepthTest>(get_impl(depth_func)) {}

            DepthTest() throw()
                : TypedStatelet<DepthTest>(get_impl(LESS)) {}
        };


        /**
         * \brief Controls which buffer is used to source vertex data to OpenGL
         * rendering commands (e.g. glDrawElements).
         */
        class VertexPointer : public TypedStatelet<VertexPointer>
        {
            class VertexPointerImpl : public StateletImpl
            {
                boost::shared_ptr<VertexBuffer> m_buffer;
                std::size_t m_stride;
                std::size_t m_offset;

            public:
                VertexPointerImpl(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset) throw()
                    : m_buffer(buffer),
                      m_stride(stride),
                      m_offset(offset) {}

                virtual void activate() throw()
                {
                    if( m_buffer )
                    {
                        glEnableClientState(GL_VERTEX_ARRAY);
                        m_buffer->source_vertex_data(
                            3, GL_FLOAT, m_stride, m_offset);
                    }
                    else
                    {
                        glDisableClientState(GL_VERTEX_ARRAY);
                    }
                }

                boost::shared_ptr<VertexBuffer> &get_buffer()
                {
                    return m_buffer;
                }

                void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
                {
                    m_buffer = buffer;
                }

                void set_stride(std::size_t stride)
                {
                    m_stride = stride;
                }

                void set_offset(std::size_t offset)
                {
                    m_offset = offset;
                }

                void configure(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset)
                {
                    m_buffer = buffer;
                    m_stride = stride;
                    m_offset = offset;
                }
            };

        public:
            VertexPointer(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
                : TypedStatelet<VertexPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new VertexPointerImpl(buffer, stride, offset))) {}

            VertexPointer()
                : TypedStatelet<VertexPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new VertexPointerImpl(
                            boost::shared_ptr<VertexBuffer>(), 0, 0))) {}

            boost::shared_ptr<VertexBuffer> &get_buffer()
            {
                return boost::dynamic_pointer_cast<VertexPointerImpl>(
                    m_impl)->get_buffer();
            }

            void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
            {
                boost::dynamic_pointer_cast<VertexPointerImpl>(
                    m_impl)->set_buffer(buffer);
            }

            void set_stride(std::size_t stride)
            {
                boost::dynamic_pointer_cast<VertexPointerImpl>(
                    m_impl)->set_stride(stride);
            }

            void set_offset(std::size_t offset)
            {
                boost::dynamic_pointer_cast<VertexPointerImpl>(
                    m_impl)->set_offset(offset);
            }

            void configure(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
            {
                boost::dynamic_pointer_cast<VertexPointerImpl>(
                    m_impl)->configure(buffer, stride, offset);
            }
        };


        /**
         * \brief Controls which buffer is used to source normal data to OpenGL
         * rendering commands (e.g. glDrawElements).
         */
        class NormalPointer : public TypedStatelet<NormalPointer>
        {
            class NormalPointerImpl : public StateletImpl
            {
                boost::shared_ptr<VertexBuffer> m_buffer;
                std::size_t m_stride;
                std::size_t m_offset;

            public:
                NormalPointerImpl(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset) throw()
                    : m_buffer(buffer),
                      m_stride(stride),
                      m_offset(offset) {}

                virtual void activate() throw()
                {
                    if( m_buffer )
                    {
                        glEnableClientState(GL_NORMAL_ARRAY);
                        m_buffer->source_normal_data(GL_FLOAT, m_stride, m_offset);
                    }
                    else
                    {
                        glDisableClientState(GL_NORMAL_ARRAY);
                    }
                }

                boost::shared_ptr<VertexBuffer> &get_buffer()
                {
                    return m_buffer;
                }

                void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
                {
                    m_buffer = buffer;
                }

                void set_stride(std::size_t stride)
                {
                    m_stride = stride;
                }

                void set_offset(std::size_t offset)
                {
                    m_offset = offset;
                }

                void configure(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset)
                {
                    m_buffer = buffer;
                    m_stride = stride;
                    m_offset = offset;
                }
            };

        public:
            NormalPointer(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
                : TypedStatelet<NormalPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new NormalPointerImpl(buffer, stride, offset))) {}

            NormalPointer()
                : TypedStatelet<NormalPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new NormalPointerImpl(
                            boost::shared_ptr<VertexBuffer>(), 0, 0))) {}

            boost::shared_ptr<VertexBuffer> &get_buffer()
            {
                return boost::dynamic_pointer_cast<NormalPointerImpl>(
                    m_impl)->get_buffer();
            }

            void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
            {
                boost::dynamic_pointer_cast<NormalPointerImpl>(
                    m_impl)->set_buffer(buffer);
            }

            void set_stride(std::size_t stride)
            {
                boost::dynamic_pointer_cast<NormalPointerImpl>(
                    m_impl)->set_stride(stride);
            }

            void set_offset(std::size_t offset)
            {
                boost::dynamic_pointer_cast<NormalPointerImpl>(
                    m_impl)->set_offset(offset);
            }

            void configure(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
            {
                boost::dynamic_pointer_cast<NormalPointerImpl>(
                    m_impl)->configure(buffer, stride, offset);
            }
        };


        /**
         * \brief Controls which buffer is used to source color data to OpenGL
         * rendering commands (e.g. glDrawElements).
         */
        class ColorPointer : public TypedStatelet<ColorPointer>
        {
            class ColorPointerImpl : public StateletImpl
            {
                boost::shared_ptr<VertexBuffer> m_buffer;
                std::size_t m_stride;
                std::size_t m_offset;

            public:
                ColorPointerImpl(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset) throw()
                    : m_buffer(buffer),
                      m_stride(stride),
                      m_offset(offset) {}

                virtual void activate() throw()
                {
                    if( m_buffer )
                    {
                        glEnableClientState(GL_COLOR_ARRAY);
                        m_buffer->source_color_data(3, GL_UNSIGNED_BYTE, m_stride, m_offset);
                    }
                    else
                    {
                        glDisableClientState(GL_COLOR_ARRAY);
                    }
                }

                boost::shared_ptr<VertexBuffer> &get_buffer()
                {
                    return m_buffer;
                }

                void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
                {
                    m_buffer = buffer;
                }

                void set_stride(std::size_t stride)
                {
                    m_stride = stride;
                }

                void set_offset(std::size_t offset)
                {
                    m_offset = offset;
                }

                void configure(
                    const boost::shared_ptr<VertexBuffer> &buffer,
                    std::size_t stride, std::size_t offset)
                {
                    m_buffer = buffer;
                    m_stride = stride;
                    m_offset = offset;
                }
            };

        public:
            ColorPointer(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
                : TypedStatelet<ColorPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new ColorPointerImpl(buffer, stride, offset))) {}

            ColorPointer()
                : TypedStatelet<ColorPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new ColorPointerImpl(
                            boost::shared_ptr<VertexBuffer>(), 0, 0))) {}

            boost::shared_ptr<VertexBuffer> &get_buffer()
            {
                return boost::dynamic_pointer_cast<ColorPointerImpl>(
                    m_impl)->get_buffer();
            }

            void set_buffer(const boost::shared_ptr<VertexBuffer> &buffer)
            {
                boost::dynamic_pointer_cast<ColorPointerImpl>(
                    m_impl)->set_buffer(buffer);
            }

            void set_stride(std::size_t stride)
            {
                boost::dynamic_pointer_cast<ColorPointerImpl>(
                    m_impl)->set_stride(stride);
            }

            void set_offset(std::size_t offset)
            {
                boost::dynamic_pointer_cast<ColorPointerImpl>(
                    m_impl)->set_offset(offset);
            }

            void configure(
                const boost::shared_ptr<VertexBuffer> &buffer,
                std::size_t stride, std::size_t offset)
            {
                boost::dynamic_pointer_cast<ColorPointerImpl>(
                    m_impl)->configure(buffer, stride, offset);
            }
        };



        /**
         * \brief Controls which buffer is used to source UV-coordinates to OpenGL
         * rendering commands (e.g. glDrawElements).
         */
        class TexCoordPointer : public TypedStatelet<TexCoordPointer>
        {
            class TexCoordPointerImpl : public StateletImpl
            {
                boost::shared_ptr<VertexBuffer> m_buffer;
            public:
                TexCoordPointerImpl(boost::shared_ptr<VertexBuffer> buffer) throw()
                    : m_buffer(buffer) {}

                virtual void activate() throw()
                {
                    if( m_buffer )
                    {
                        glEnableClientState(GL_TEXTURE_COORD_ARRAY);
                        m_buffer->source_texcoord_data(2, GL_FLOAT, 0, 0);
                    }
                    else
                        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
                }

                boost::shared_ptr<VertexBuffer> &get_buffer() throw()
                {
                    return m_buffer;
                }
                
                void set_buffer(boost::shared_ptr<VertexBuffer> buffer) throw()
                {
                    m_buffer = buffer;
                }
            };

        public:
            TexCoordPointer(boost::shared_ptr<VertexBuffer> buffer) throw()
                : TypedStatelet<TexCoordPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new TexCoordPointerImpl(buffer))) {}

            TexCoordPointer() throw()
                : TypedStatelet<TexCoordPointer>(
                    boost::shared_ptr<StateletImpl>(
                        new TexCoordPointerImpl(
                            boost::shared_ptr<VertexBuffer>()))) {}

            boost::shared_ptr<VertexBuffer> &get_buffer() throw()
            {
                return boost::dynamic_pointer_cast<TexCoordPointerImpl>(
                    m_impl)->get_buffer();
            }

            void set_buffer(boost::shared_ptr<VertexBuffer> buffer) throw()
            {
                set_impl(boost::shared_ptr<StateletImpl>(
                             new TexCoordPointerImpl(buffer)));
            }
        };



        /**
         * \brief Controls the fill mode (polygon mode) used when rendering.
         */
        class PolygonMode : public TypedStatelet<PolygonMode>
        {
        public:
            enum FaceEnum
            {
                FRONT = GL_FRONT,
                BACK = GL_BACK,
                FRONT_AND_BACK = GL_FRONT_AND_BACK
            };

            enum ModeEnum
            {
                POINT = GL_POINT,
                LINE = GL_LINE,
                FILL = GL_FILL
            };

        private:
            class PolygonModeImpl : public StateletImpl
            {
                FaceEnum m_face;
                ModeEnum m_mode;
            public:
                PolygonModeImpl(FaceEnum face, ModeEnum mode) throw()
                    : m_face(face), m_mode(mode) {}

                virtual void activate() throw()
                {
                    glPolygonMode(m_face, m_mode);
                }
            };


            /*boost::shared_ptr<StateletImpl>
            get_impl(bool enable) const throw()
            {
                if( enable )
                {
                    static boost::shared_ptr<StateletImpl> on(
                        new PolygonModeImpl(enable));
                    return on;
                }
                else
                {
                    static boost::shared_ptr<StateletImpl> off(
                        new PolygonModeImpl(enable));
                    return off;
                }
                }*/

        public:
            PolygonMode(FaceEnum face, ModeEnum mode) throw()
                : TypedStatelet<PolygonMode>(
                    boost::shared_ptr<StateletImpl>(
                        new PolygonModeImpl(
                            face, mode))) {}

            PolygonMode() throw()
                : TypedStatelet<PolygonMode>(
                    boost::shared_ptr<StateletImpl>(
                        new PolygonModeImpl(FRONT_AND_BACK, FILL))) {}
        };



        /**
         * \brief Controls the parameters governing line appearance - line 
         * width and stipple pattern.
         */
        class LineParams : public TypedStatelet<LineParams>
        {
            class LineParamsImpl : public StateletImpl
            {
                float m_line_width;
                bool   m_enable_line_stipple;
                int32_t  m_stipple_factor;
                uint16_t m_stipple_pattern;
            public:
                LineParamsImpl(float line_width, 
                               bool enable_line_stipple,
                               int32_t stipple_factor, 
                               uint16_t stipple_pattern) throw()
                    : m_line_width(line_width),
                      m_enable_line_stipple(enable_line_stipple),
                      m_stipple_factor(stipple_factor),
                      m_stipple_pattern(stipple_pattern) {}

                virtual void activate() throw()
                {
                    if( m_enable_line_stipple )
                    {
                        glLineStipple(m_stipple_factor, m_stipple_pattern);
                        glEnable(GL_LINE_STIPPLE);
                    }
                    else
                        glDisable(GL_LINE_STIPPLE);
                    
                    glLineWidth(m_line_width);
                }
            };

        public:
            LineParams(float line_width, 
                       bool enable_line_stipple,
                       int32_t stipple_factor, 
                       uint16_t stipple_pattern) throw()
                : TypedStatelet<LineParams>(
                    boost::shared_ptr<StateletImpl>(
                        new LineParamsImpl(line_width, enable_line_stipple,
                                           stipple_factor, stipple_pattern))) {}

            LineParams() throw()
                : TypedStatelet<LineParams>(
                    boost::shared_ptr<StateletImpl>(
                        new LineParamsImpl(1, false, 0, 0))) {}
        };



        /**
         * \brief Governs the size of rendered points.
         */
        class PointParams : public TypedStatelet<PointParams>
        {
            class PointParamsImpl : public StateletImpl
            {
                float m_point_size;
                bool m_smooth;
                
            public:
                PointParamsImpl(float point_size, bool smooth) throw()
                    : m_point_size(point_size), m_smooth(smooth) {}

                virtual void activate() throw()
                {
                    glPointSize(m_point_size);

                    Device::toggle_point_smoothing(m_smooth);
                }
            };

        public:
            PointParams(float point_size, bool smooth) throw()
                : TypedStatelet<PointParams>(
                    boost::shared_ptr<StateletImpl>(
                        new PointParamsImpl(point_size, smooth))) {}

            PointParams() throw()
                : TypedStatelet<PointParams>(
                    boost::shared_ptr<StateletImpl>(
                        new PointParamsImpl(1, true))) {}
        };


        /**
         * \brief Controls the \e name (as in glLoadName) used when rendering.
         *
         * \remark This is not thread-safe, so if we ever plan on doing
         * threaded rendering this will have to be revised.
         */
        class Name : public TypedStatelet<Name>
        {
            class NameImpl : public StateletImpl
            {
                typedef std::map<ObjectID, boost::weak_ptr<StateletImpl> > ImplTab;
                static ImplTab ms_impl_tab;

                const uint32_t m_id;

                NameImpl(uint32_t id) throw() : m_id(id) {}

            public:
                virtual ~NameImpl() throw()
                {
                    ms_impl_tab.erase(m_id);
                }

                virtual void activate() throw()
                {
                    boost::uint8_t r = (m_id>>16) & 0xFF;
                    boost::uint8_t g = (m_id>>8) & 0xFF;
                    boost::uint8_t b = m_id & 0xFF;
                    glColor3ub(r, g, b);
                    /*boost::uint8_t r = (m_id>>24) & 0xFF;
                    boost::uint8_t g = (m_id>>16) & 0xFF;
                    boost::uint8_t b = (m_id>>8) & 0xFF;
                    boost::uint8_t a = m_id & 0xFF;
                    glColor4ub(r, g, b, a);*/
                }

                uint32_t get() const throw()
                {
                    return m_id;
                }

                static boost::shared_ptr<StateletImpl> get_impl(uint32_t id) throw()
                {
                    ImplTab::iterator it = ms_impl_tab.find(id);
                    boost::shared_ptr<StateletImpl> impl;

                    // Is there an entry for the given ID? Is it still valid?
                    if( it != ms_impl_tab.end() && (impl = it->second.lock()) != 0 )
                        return impl;

                    impl = boost::shared_ptr<StateletImpl>(new NameImpl(id));
                    ms_impl_tab.insert(std::make_pair(id, impl));

                    return impl;
                }
            };

        public:
            Name(uint32_t id) throw()
                : TypedStatelet<Name>(NameImpl::get_impl(id)) {}

            Name() throw() : TypedStatelet<Name>(NameImpl::get_impl(0)) {}

            void set(uint32_t id) throw()
            {
                m_impl = NameImpl::get_impl(id);
            }

            uint32_t get() const throw()
            {
                return boost::dynamic_pointer_cast<NameImpl>(m_impl)->get();
            }
        };


    }

    using statelets::Statelet;
    typedef statelets::StateletType StateletType;

}
}


#endif // PEEKABOT_RENDERER_STATELET_HH_INCLUDED
