/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cmath>
#include <GL/glew.h>

#include "Camera.hh"
#include "../BoundingVolumes.hh"
#include "../Frustum.hh"
#include "../RenderContext.hh"
#include "../PrepareRenderContext.hh"

using namespace peekabot;
using namespace peekabot::renderer;


static const float DEG_TO_RAD = M_PI/180.0f;


Camera::Camera(bool is_ortho,
               float fov,
               float near,
               float far,
               float zoom_distance) throw()
    : m_fov(fov),
      m_fov_rad(m_fov * DEG_TO_RAD),
      m_is_ortho(is_ortho),
      m_near(near),
      m_far(far),
      m_zoom_distance(zoom_distance),
      m_x(0), m_y(0),
      m_w(100), m_h(100)
{
    calculate_bvs();
    calculate_viewer_mtow();
    get_state().set(new statelets::Lighting(false));
}


Camera::~Camera() throw()
{
}


Camera *Camera::clone() const throw()
{
    Camera *ret = new Camera(
        is_orthographic(), get_fov(),
        get_near_plane(), get_far_plane(),
        get_zoom_distance());
    ret->set_viewport(m_x, m_y, m_w, m_h);
    ret->set_transformation(get_transformation());
    return ret;
}


void Camera::get_renderables(PrepareRenderContext &context) const
{
    if( context.get_camera() != this )
    {
        context.prepare(this);
    }
}


void Camera::set_viewport(int32_t x, int32_t y, int32_t w, int32_t h) throw()
{
    // No need re-calculating the frustum volumes if the viewport 
    // hasn't changed.
    if( x != m_x || y != m_y || w != m_w || h != m_h )
    {
        m_x = x;
        m_y = y;
        m_w = w;
        m_h = h;

        invalidate_cached_frustums();
    }

    glViewport(x, y, w, h);
}


void Camera::get_viewport(int32_t &x, int32_t &y, int32_t &w, int32_t &h) const throw()
{
    x = m_x;
    y = m_y;
    w = m_w;
    h = m_h;
}


void Camera::set_fov(float fov) throw()
{
    if( fov != m_fov )
    {
        m_fov = fov;
        m_fov_rad = m_fov * DEG_TO_RAD;

        invalidate_cached_frustums();
        calculate_bvs();
    }
}


float Camera::get_fov() const throw()
{
    return m_fov;
}


void Camera::set_near_plane(float y_distance) throw()
{
    if( y_distance != m_near )
    {
        m_near = y_distance;

        invalidate_cached_frustums();
        calculate_bvs();
    }
}


void Camera::set_far_plane(float y_distance) throw()
{
    if( y_distance != m_far )
    {
        m_far = y_distance;

        invalidate_cached_frustums();
        calculate_bvs();
    }
}


void Camera::set_zoom_distance(float zoom_distance) throw()
{
    if( m_zoom_distance != zoom_distance )
    {
        m_zoom_distance = zoom_distance;

        invalidate_cached_frustums();
        calculate_bvs();
        calculate_viewer_mtow();
    }
}


float Camera::get_zoom_distance() const throw()
{
    return m_zoom_distance;
}


void Camera::set_orthographic(bool is_ortho) throw()
{
    if( is_ortho != m_is_ortho )
    {
        m_is_ortho = is_ortho;

        invalidate_cached_frustums();
        calculate_bvs();
    }
}


bool Camera::is_orthographic() const throw()
{
    return m_is_ortho;
}


void Camera::apply_projection_matrix() throw()
{
    glMatrixMode(GL_PROJECTION);

    if( is_orthographic() )
    {
        float d = m_zoom_distance;
        float aspect = static_cast<float>(m_w)/m_h;
        float half_h = d*tanf(m_fov_rad/2);
        float half_w = half_h*aspect;

        glOrtho(-half_w, half_w, -half_h, half_h, m_near, m_far);
    }
    else
    {
        gluPerspective(m_fov, static_cast<float>(m_w)/m_h, m_near, m_far);
    }

    glMatrixMode(GL_MODELVIEW);
}


void Camera::calculate_bvs() throw()
{
    // Calculate BVs
    //
    // The point furthest away from the center of the BV must either
    // lie in the corner of the frustum of at the edge of the slope
    // on the arrow.
    //
    // It can be proved that the point on the edge of the arrow's slope
    // that yields the greatest distance is (always) the tip of the arrow.
    // Thus, we need only consider two cases - the distance to the corner
    // and arrow tip respectively.

    BoundingSphere bsphere;

    float base = (m_near * tanf(m_fov_rad/2));
    float cornerdist = sqrtf(m_near*m_near/4 + base*M_SQRT2);
    float arrowtipdist = Eigen::Vector3f(0, m_near/2, base*1.55f).norm();

    if( std::max(arrowtipdist, cornerdist) > m_zoom_distance/2 )
    {
        bsphere.set_radius(std::max(arrowtipdist, cornerdist));
        bsphere.set_pos(Eigen::Vector3f(-m_near/2, 0, 0));
    }
    else
    {
        bsphere.set_radius(m_zoom_distance/2);
        bsphere.set_pos(Eigen::Vector3f(-m_zoom_distance/2, 0, 0));
    }

    set_bounding_sphere(bsphere);
}


void Camera::invalidate_cached_frustums() const throw()
{
    m_frustums.clear();
}


const boost::shared_ptr<Frustum> Camera::get_frustum(
    float w_slack, float h_slack) const throw()
{
    FrustumMap::const_iterator it = m_frustums.find(
        std::make_pair(w_slack, h_slack));
    
    if( it != m_frustums.end() )
        return it->second;
    else
    {
        return m_frustums.insert(
            std::make_pair(
                std::make_pair(w_slack, h_slack), 
                boost::shared_ptr<Frustum>( new Frustum(*this, w_slack, h_slack) )
                )).first->second;
    }
}


void Camera::render(RenderContext &context) const
{
    float offset = -m_zoom_distance;

    // Half the base width/height
    float base = (m_near * tanf(m_fov_rad/2));

    glBegin(GL_LINE_LOOP);
    glVertex3f(m_near+offset, -base,  base);
    glVertex3f(m_near+offset,  base,  base);
    glVertex3f(m_near+offset,  base, -base);
    glVertex3f(m_near+offset, -base, -base);
    glEnd();
    
    glBegin(GL_LINES);
    glVertex3f(offset,0,0);
    glVertex3f(m_near+offset, -base,  base);
    glVertex3f(offset,0,0);
    glVertex3f(m_near+offset,  base,  base);
    glVertex3f(offset,0,0);
    glVertex3f(m_near+offset,  base, -base);
    glVertex3f(offset,0,0);
    glVertex3f(m_near+offset, -base, -base);
    glEnd();

    // Draw up vector
    glBegin(GL_TRIANGLES);
    // Front faces...
    glVertex3f(m_near+offset, -base/4, base*1.25f);
    glVertex3f(m_near+offset,    0.0f, base*1.55f);
    glVertex3f(m_near+offset,  base/4, base*1.25f);

    glVertex3f(m_near+offset,    0.0f, base*1.55f);
    glVertex3f(m_near+offset, -base/4, base*1.25f);
    glVertex3f(m_near+offset, -base/2, base*1.25f);

    glVertex3f(m_near+offset,    0.0f, base*1.55f);
    glVertex3f(m_near+offset,  base/2, base*1.25f);
    glVertex3f(m_near+offset,  base/4, base*1.25f);

    glVertex3f(m_near+offset, -base/4, base);
    glVertex3f(m_near+offset, -base/4, base*1.25f);
    glVertex3f(m_near+offset,  base/4, base*1.25f);

    glVertex3f(m_near+offset, -base/4, base);
    glVertex3f(m_near+offset,  base/4, base*1.25f);
    glVertex3f(m_near+offset,  base/4, base);

    // and backfaces...
    glVertex3f(m_near+offset, -base/4, base*1.25f);
    glVertex3f(m_near+offset,  base/4, base*1.25f);
    glVertex3f(m_near+offset,    0.0f, base*1.55f);

    glVertex3f(m_near+offset,    0.0f, base*1.55f);
    glVertex3f(m_near+offset, -base/2, base*1.25f);
    glVertex3f(m_near+offset, -base/4, base*1.25f);

    glVertex3f(m_near+offset,    0.0f, base*1.55f);
    glVertex3f(m_near+offset,  base/4, base*1.25f);
    glVertex3f(m_near+offset,  base/2, base*1.25f);

    glVertex3f(m_near+offset, -base/4, base);
    glVertex3f(m_near+offset,  base/4, base*1.25f);
    glVertex3f(m_near+offset, -base/4, base*1.25f);

    glVertex3f(m_near+offset, -base/4, base);
    glVertex3f(m_near+offset,  base/4, base);
    glVertex3f(m_near+offset,  base/4, base*1.25f);
    glEnd();


    glPushAttrib(GL_LINE_BIT);

    glLineStipple(3, 43690);
    glEnable(GL_LINE_STIPPLE);

    glBegin(GL_LINES);
    glVertex3f(0,0,0);
    glVertex3f(offset,0,0);
    glEnd();

    glPopAttrib();
}


void Camera::on_transformation_set()
{
    calculate_bvs();
    invalidate_cached_frustums();
    calculate_viewer_mtow();
    CullableEntity::on_transformation_set();
}


void Camera::calculate_viewer_mtow()
{
    m_viewer_mtow =
        get_transformation() *
        Eigen::Translation3f(-m_zoom_distance, 0, 0);
}

