/*
 * Copyright Staffan Gimåker 2006-2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_CAMERA_HH_INCLUDED
#define PEEKABOT_RENDERER_CAMERA_HH_INCLUDED


#include "../CullableEntity.hh"
#include "../Renderable.hh"

#include <cmath>
#include <boost/shared_ptr.hpp>



namespace peekabot
{
    namespace renderer
    {
        class Frustum;

        class Camera : public CullableEntity, public Renderable
        {
        protected:
            /**
             * \brief The field of view angle, in degrees, in the Z direction.
             *
             * \c m_fov is the angle between the top and bottom of the view frustum.
             */
            float m_fov;

            /**
             * \brief The FOV in radians, see \c m_fov.
             */
            float m_fov_rad;

            /**
             * \brief Denotes whether or not the camera is used in perspective
             * or orthographic rendering mode.
             *
             * <em>Initial value:</em> false.
             */
            bool m_is_ortho;

            /**
             * \brief The position of the camera's near clip plane, 
             * along the cameras view direction.
             *
             * Any object closer to the camera will be clipped.
             */
            float m_near;

            /**
             * \brief The position of the camera's far clip plane, 
             * along the cameras view direction.
             *
             * Any object further away from the camera will be clipped.
             */        
            float m_far;

            float m_zoom_distance;

            /**
             * \brief X coordinate of the viewport rectangle.
             */
            int32_t m_x;

            /**
             * \brief Y coordinate of the viewport rectangle.
             */
            int32_t m_y;

            /**
             * \brief The width of the viewport rectangle.
             */
            int32_t m_w;

            /**
             * \brief The height of the viewport rectangle.
             */
            int32_t m_h;

            typedef std::map<std::pair<float, float>, 
                             boost::shared_ptr<Frustum> > FrustumMap;

            mutable FrustumMap m_frustums;

            /**
             * \brief Cached viewer-transformation, must be recalculated when
             * the viewer moves.
             *
             * \sa calculate_viewer_mtow()
             */
            Eigen::Transform3f m_viewer_mtow;

        private:
            void calculate_bvs() throw();

            void invalidate_cached_frustums() const throw();

            void calculate_viewer_mtow();


        protected:
            virtual void on_transformation_set();

        public:
            Camera(bool is_ortho = false,
                   float fov = 45,
                   float near = 1,
                   float far = 100,
                   float zoom_distance = 10) throw();

            virtual ~Camera() throw();

            virtual Camera *clone() const throw();



            /**
             * \brief Specify the viewport used in conjunction with rendering using
             * the camera.
             *
             * In order for projection and frustum volume calculations to be 
             * performed correctly, the viewport must be set prior to initating
             * a render using the camera.
             *
             * \pre \f$x,y \geq 0, \; w,h>0\f$
             *
             * \param x The x-coordinate of the rectangle which to render to.
             * \param y The y-coordinate of the rectangle which to render to.
             * \param w The width of the rendered rectangle.
             * \param h The height of the rendered rectangle.
             */
            void set_viewport(int32_t x, int32_t y, int32_t w, int32_t h) throw();


            void get_viewport(int32_t &x, int32_t &y, int32_t &w, int32_t &h) const throw();
        

            /**
             * \brief Set the camera's field of view.
             *
             * To enable smooth transitions between perspective and orthographic
             * rendering modes, the camera's FOV parameter is used not only by
             * a perspective cameras as its field of view, but also by orthographic
             * cameras. Ortho cameras use the FOV parameter as a zoom parameter
             * used to calculate the extent of the view frustum, resulting in an
             * effect much like that of the FOV's for perspective cameras.
             *
             * \pre \f$0 < fov < 180\f$
             *
             * \param fov The ("whole") field of view angle, in the Z direction. 
             * Specified in degrees.
             *
             * \remarks If a FOV of 180 degrees or wider is specified, the resulting
             * behaviour is undefined but will likely result in broken frustum 
             * culling.
             */
            void set_fov(float fov) throw();

            /**
             * \brief Get the camera's (top-to-bottom) field of view angle in the 
             * Z direction, in degrees.
             *
             * Note that this is the full FOV angle, from top to bottom!
             *
             * \sa \c set_fov(float)
             */
            float get_fov() const throw();

            /**
             * \brief The same as get_fov(), but the FOV angle is returned in 
             * radians rather than degrees.
             */
            inline float get_fov_rad() const throw()
            {
                return m_fov_rad;
            }

            /**
             * \brief Get the camera's (left-to-right) horizontal FOV.
             *
             *
             * Note that this is the full FOV angle, from side to side of the 
             * frustum!
             *
             * The horizontal FOV is not cached, and thus you should avoid 
             * calling this method repeatedly in time code.
             */
            inline float get_horiz_fov_rad() const throw()
            {
                return 2*atanf(tanf(m_fov_rad/2) * m_w/float(m_h));
            }

            /**
             * \brief Toggle orthogonal/perspective rendering.
             *
             * \param is_ortho If true, the camera will be set to orthographic mode.
             * Otherwise it will be set to perspective mode.
             *
             * \sa \c set_fov(float) (for the meaning of FOV for orthographic 
             * cameras).
             */
            void set_orthographic(bool is_ortho) throw();

            /**
             * \brief Returns true if the camera is in orthographic mode.
             */
            bool is_orthographic() const throw();

            /**
             * \brief Get the camera's view direction in world coordinates.
             *
             * \return The camera's positive view direction, as a homogenous vector.
             *
             * \remarks Returns the y-axis of the camera's model to world space
             * transform. It is always of unit length.
             */
            inline const Eigen::Vector4f get_view_vector() const throw()
            {
                return get_transformation().matrix().col(0);
            }

            /**
             * \brief Get the camera's up direction.
             *
             * \return The camera's positive view direction, as a homogenous vector.
             *
             * \remarks Returns the z-axis of the camera's model to world space
             * transform.
             */
            inline const Eigen::Vector4f get_up_vector() const throw()
            {
                return get_transformation().matrix().col(2);
            }

            inline const Eigen::Transform3f &get_viewer_mtow() const throw()
            {
                return m_viewer_mtow;
            }

            inline const Eigen::Vector4f get_viewer_world_pos() const throw()
            {
                return m_viewer_mtow.matrix().col(3);
            }

            /**
             * \brief Set the camera's near plane.
             *
             * Objects (or the parts of an objects) closer to the camera than 
             * \c y_distance units in along the camera's view vector will not 
             * be drawn.
             *
             * \param y_distance The near plane's y-distance from the camera, in 
             * local coordinates.
             */
            void set_near_plane(float y_distance) throw();

            /**
             * \brief Returns the camera's near clip plane distance, along the 
             * y axis.
             *
             * \sa \c set_near_plane(float)
             */
            inline float get_near_plane() const throw()
            {
                return m_near;
            }

            /**
             * \brief Set the camera's far plane.
             *
             * Objects (or the parts of an objects) further away from the camera 
             * than \a y_distance units in along the camera's view vector will not 
             * be drawn.
             *
             * \param y_distance The far plane's y-distance from the camera, in
             * local coordinates.
             */
            void set_far_plane(float y_distance) throw();

            /**
             * \brief Returns the camera's far clip plane distance, along the 
             * y axis.
             *
             * \sa set_far_plane(float)
             */
            inline float get_far_plane() const throw()
            {
                return m_far;
            }

            /**
             * \brief Get the aspect ratio (width/height) of the viewport.
             */
            inline float get_aspect_ratio() const throw()
            {
                return (float)m_w/m_h;
            }

            inline int32_t get_viewport_height() const throw()
            {
                return m_h;
            }

            inline int32_t get_viewport_width() const throw()
            {
                return m_w;
            }

            /**
             * \brief Returns the height of the viewport in meters.
             *
             * The height of the viewport in meter is the same as the height 
             * of the near plane.
             */
            inline float get_viewport_height_meter() const throw()
            {
                return 2 * m_near * tanf(m_fov_rad/2);
            }

            /**
             * \brief Returns the width of the viewport in meters.
             *
             * The width of the viewport in meter is the same as the width 
             * of the near plane.
             */
            inline float get_viewport_width_meter() const throw()
            {
                return get_aspect_ratio() * get_viewport_height_meter();
            }

            void set_zoom_distance(float zoom_distance) throw();

            float get_zoom_distance() const throw();

            const boost::shared_ptr<Frustum> get_frustum(
                float w_slack = 0, float h_slack = 0) const throw();

            void apply_projection_matrix() throw();

            virtual void render(RenderContext &context) const;

            virtual void get_renderables(PrepareRenderContext &context) const;

            /**
             * \brief Return the relationship between pixels and meters on the 
             * camera's near plane.
             *
             * E.g. if the factor is 100, drawing a 1 meter high primitive on x=near
             * would yield a 100 pixel high blob in the camera.
             */
            inline float get_pixels_per_meter() const throw()
            {
                return m_h / get_viewport_height_meter();
            }
        };
    }
}


#endif // PEEKABOT_RENDERER_CAMERA_HH_INCLUDED
