/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_PRIMITIVES_LABEL_HH_INCLUDED
#define PEEKABOT_RENDERER_PRIMITIVES_LABEL_HH_INCLUDED


#include <cmath>
#include <boost/shared_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <Eigen/Core>

#include "../CullableEntity.hh"
#include "../Renderable.hh"
#include "../Font.hh"

#include "../Statelet.hh"
#include "../PrepareRenderContext.hh"
#include "../../Config.hh"


namespace peekabot
{
    namespace renderer
    {
        class Label : public CullableEntity, 
                      public Renderable
        {
        public:
            explicit Label(const Config &config)
                : m_alignment(ALIGN_LEFT),
                  m_font(
                      new Font(
                          (config.get<boost::filesystem::path>(
                              "resources.share_path") / "fonts" /
                              "FreeSerif.ttf").string()))
            {
                get_state().set(new statelets::Culling(false));
                get_state().set(new statelets::Lighting(false));
            }

            Label(const Label &x)
                : m_text(x.m_text), m_alignment(x.m_alignment), m_font(x.m_font) {}

            /*virtual Label *clone() const throw()
            {
                return new Label(*this);
            }*/

            virtual void get_renderables(PrepareRenderContext &context) const
            {
                context.prepare(this);
            }

            virtual void render(RenderContext &context) const
            {
                m_font->render_string(m_text, m_offset);
            }

            void set_text(const std::string &text) throw()
            {
                if( m_text != text )
                {
                    m_text = text;
                    calculate_bvs();
                }
            }

            void set_alignment(TextAlignment alignment) throw()
            {
                if( m_alignment != alignment )
                {
                    m_alignment = alignment;
                    calculate_bvs();
                }
            }

            TextAlignment get_alignment() const throw()
            {
                return m_alignment;
            }

        protected:
            void calculate_bvs() throw()
            {
                std::pair<float, float> bb = m_font->measure_string(m_text);

                if( m_alignment == ALIGN_LEFT )
                    m_offset = 0;
                else if( m_alignment == ALIGN_RIGHT )
                    m_offset = -bb.first;
                else // m_alignment == ALIGN_CENTER
                    m_offset = -bb.first / 2.0f;

                float x_h = bb.first/2;
                float y_h = bb.second/2;

                set_bounding_sphere(
                    BoundingSphere(
                        Eigen::Vector3f(m_offset + x_h, y_h, 0),
                        sqrtf(x_h*x_h + y_h*y_h)));
            }

        private:
            std::string m_text;

            TextAlignment m_alignment;

            boost::shared_ptr<Font> m_font;

            /**
             * \brief The offset along the X-axis (in meters) relative the 
             * origin where the beginning of the label should be rendered.
             *
             * Used to center, left-, and right-align text.
             */
            float m_offset;
        };
    }
}


#endif // PEEKABOT_RENDERER_PRIMITIVES_LABEL_HH_INCLUDED
