/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cmath>
#include <cassert>
#include <GL/glew.h>
#include <boost/scoped_ptr.hpp>

#include "OccupancyGrid2D.hh"
#include "Camera.hh"
#include "../RenderContext.hh"
#include "../PrepareRenderContext.hh"
#include "../VertexBuffer.hh"
#include "../Frustum.hh"
#include "../Mesh.hh"


using namespace peekabot;
using namespace peekabot::renderer;


namespace
{
    const uint8_t UNKNOWN = 255;
}


//
// ------------- OccupancyGrid2D implementation ----------------
//

OccupancyGrid2D::OccupancyGrid2D(float cell_size, size_t max_cache_size)
    : m_root_node(new Node(0)),
      m_cell_size(cell_size),
      m_root_cell_size(cell_size),
      m_root_cell_center(Eigen::Vector2f(0,0)),
      m_unoccupied_color(0.9, 0.9, 0.9),
      m_occupied_color(0, 0, 0),
      m_max_cache_size(max_cache_size)
{
}


OccupancyGrid2D::~OccupancyGrid2D() throw()
{
    delete m_root_node;
    m_root_node = 0;
}


void OccupancyGrid2D::set_belief(const Eigen::Vector2f &x, float belief)
    throw(std::domain_error)
{
    if( belief < 0 || belief > 1 )
        throw std::domain_error("Belief must be in the range [0,1]");

    if( !m_root_node->encloses(x, m_root_cell_center, m_root_cell_size) )
        grow_to_accomodate(x);

    // Discretize the belief (so we have a higher chance of
    // merging them later on, and for reduced storage)
    uint8_t bel = (uint8_t)roundf(belief*254);

    m_root_node->set_belief(
        x, bel, m_cell_size, m_root_cell_center, m_root_cell_size);
}


void OccupancyGrid2D::set_cells(
    const std::vector<std::pair<Eigen::Vector2f, float> > &cells)
    throw(std::domain_error)
{
    for( std::size_t i = 0; i < cells.size(); ++i )
    {
        set_belief(cells[i].first, cells[i].second);
    }
}


float OccupancyGrid2D::get_belief(const Eigen::Vector2f &x)
    throw(std::runtime_error)
{
    uint8_t bel = m_root_node->get_belief(
        x, m_root_cell_center, m_root_cell_size);

    return (bel == UNKNOWN) ? -1 : bel/254.0f;
}


void OccupancyGrid2D::clear_cell(const Eigen::Vector2f &x) throw()
{
    if( !m_root_node->encloses(x, m_root_cell_center, m_root_cell_size) )
        return;

    m_root_node->set_belief(
        x, UNKNOWN, m_cell_size,
        m_root_cell_center, m_root_cell_size);
}


void OccupancyGrid2D::clear() throw()
{
    delete m_root_node;
    m_root_node = 0;
    m_root_node = new Node(0);
}


void OccupancyGrid2D::set_unoccupied_color(const RGBColor &color)
{
    m_unoccupied_color = color;
}


void OccupancyGrid2D::set_occupied_color(const RGBColor &color)
{
    m_occupied_color = color;
}


void OccupancyGrid2D::get_renderables(PrepareRenderContext &context) const
{
    //
    // Translate the camera's frustum bsphere and frustum planes
    // to local coordinates (for culling in local space!)
    //

    boost::scoped_ptr<Camera> cam(context.get_camera()->clone());
    cam->set_transformation(
        Eigen::Transform3f(get_transformation().inverse(Eigen::Isometry)) *
        context.get_camera()->get_transformation());

    RGBColor color_slope(
        m_occupied_color.r - m_unoccupied_color.r,
        m_occupied_color.g - m_unoccupied_color.g,
        m_occupied_color.b - m_unoccupied_color.b);

    Batch batch(context, 1024);

    // Render/cull the tree recursively
    m_root_node->render(
        batch, *context.get_camera(),
        m_unoccupied_color, color_slope,
        m_root_cell_center, m_root_cell_size,
        Frustum(*cam, 0.0f, 0.0f), 0x3F);
}


void OccupancyGrid2D::calculate_bvs() throw()
{
    const Eigen::Vector2f &cp = m_root_cell_center;

    set_bounding_sphere(
        BoundingSphere(
            Eigen::Vector3f(cp(0), cp(1), 0), M_SQRT2 * m_root_cell_size));
}


OccupancyGrid2D::Node *OccupancyGrid2D::grow_to_accomodate(
    const Eigen::Vector2f &x) throw()
{
    // (x, y) outside the octree!
    // deal with special case... by extending the tree outwards,
    // that is, by create a new big node and use the current
    // root node as a child quadrant in the new node.
    while( !m_root_node->encloses(x, m_root_cell_center, m_root_cell_size) )
    {
        Node *old_root = m_root_node;
        float old_root_cell_size = m_root_cell_size;
        Eigen::Vector2f old_root_center = m_root_cell_center;

        Eigen::Vector2f configuration[] =
        {
            Eigen::Vector2f(-1, -1),
            Eigen::Vector2f( 1, -1),
            Eigen::Vector2f(-1,  1),
            Eigen::Vector2f( 1,  1)
        };

        float min_dist = 0;
        int min_conf = -1;

        for( int i = 0; i < 4; i++ )
        {
            Eigen::Vector2f confpos = ( old_root_center +
                                 old_root_cell_size/2*configuration[i] );

            float d = (confpos-x).norm();

            if( min_conf == -1 || d < min_dist )
            {
                min_conf = i;
                min_dist = d;
            }
        }

        // Now we just have to figure out in which directions
        // to extend the octree.
        // This can easily be done by find the configuration p
        // which minimizes the distance dist(<x,y>,p). There
        // are 4 possible configurations.
        //
        // The different possible configurations are the
        // permutations of <x,y>, x,y \in {-1,1}. I.e.
        // all the ways the new square can be offset from the
        // old root node.
        //
        // Having found the best configuration, we implicitly
        // also have the index of the new root node to put the
        // old one in.
        // index = (x/2+0.5)+2*(y/2+0.5)

        m_root_node = new Node(0);
        m_root_cell_size *= 2;
        m_root_cell_center += old_root_cell_size/2*configuration[min_conf];

        m_root_node->m_children = new Node*[4];
        for( int i = 0; i < 4; ++i )
            m_root_node->m_children[i] = 0;

        int index = -1;
        for( int i = 0; i < 4; ++i )
        {
            if( (m_root_node->quadrant_offset(i, m_root_cell_size) +
                 m_root_cell_center).isApprox(
                     old_root_center,
                     m_root_cell_size * 1e-6) )
            {
                index = i;
                break;
            }
        }

        assert( index != -1 );

        m_root_node->m_children[index] = old_root;
        old_root->m_parent = m_root_node;

        // Prune old root?
        if( old_root->is_leaf() && old_root->m_belief == UNKNOWN )
        {
            // If the old root is a leaf, and unknown, we can safely prune it
            delete old_root;
            delete[] m_root_node->m_children;
            m_root_node->m_children = 0;
        }
    }

    // Update BVs to contain the new root
    calculate_bvs();

    return m_root_node;
}


//
// ------------- OccupancyGrid2D::Node implementation ----------------
//


OccupancyGrid2D::Node::Node(Node *parent)
    : m_parent(parent),
      m_children(0),
      m_belief(UNKNOWN)
{
}


OccupancyGrid2D::Node::~Node()
{
    if( m_children )
    {
        for( int i = 0; i < 4; i++ )
            if( m_children[i] != 0 )
            {
                delete m_children[i];
                m_children[i] = 0;
            }

        delete[] m_children;
        m_children = 0;
    }
}


uint8_t OccupancyGrid2D::Node::get_belief(
    const Eigen::Vector2f &x,
    const Eigen::Vector2f &center,
    float node_size) throw()
{
    //assert(encloses(x));

    if( is_leaf() )
        return m_belief;
    else
    {
        for( int i = 0; i < 4; ++i )
            if( enclosed_in_quadrant(x, i, center) )
            {
                if( m_children[i] )
                    return m_children[i]->get_belief(
                        x, center+quadrant_offset(i, node_size), node_size/2);
                else
                    return UNKNOWN;
            }

        assert( false );
    }
}


void OccupancyGrid2D::Node::set_belief(
    const Eigen::Vector2f &x, uint8_t belief, const float cell_size,
    const Eigen::Vector2f &center, float node_size)
    throw(std::domain_error)
{
    //assert(encloses(x));

    if( is_leaf() ) // Base case
    {
        if( m_belief == belief ) // TODO: use a tolerance here!
        {
            return;
            // Note: there's no need to check for merge potential here
            // since we haven't changed the tree in any way
        }
        // Is this a true leaf node? (m_node_side == cell_size)
        else if( node_size <= 1.1f*cell_size ) // mul by 1.1 for robustness
        {
            m_belief = belief;

            // Check and merge sibling cells if possible
            optimize(this);
        }
        else
        {
            // It's a merged cell - split it up

            m_children = new Node*[4];
            for( int i = 0; i < 4; ++i )
                m_children[i] = 0;


            for( int i = 0; i < 4; ++i )
            {
                bool in_quadrant = enclosed_in_quadrant(x, i, center);

                // Don't create split nodes for UNKNOWN cells
                if( in_quadrant || m_belief != UNKNOWN )
                {
                    m_children[i] = new Node(this);
                    m_children[i]->m_belief = m_belief;

                    // Propagate new value down the tree, splitting
                    // cells as/if needed
                    if( in_quadrant )
                        m_children[i]->set_belief(
                            x, belief, cell_size,
                            center+quadrant_offset(i, node_size),
                            node_size/2);
                }
            }
        }
    }
    else // Recursion case
    {
        assert( m_children != 0 );

        // Note: in reality it's 2, but we use 1.9 for robustness
        assert( node_size >= 1.9f*cell_size );

        // Find the containing child cell and recurse
        for( int i = 0; i < 4; ++i )
            if( enclosed_in_quadrant(x, i, center) )
            {
                if( !m_children[i] )
                    m_children[i] = new Node(this); // Create if needed

                return m_children[i]->set_belief(
                        x, belief, cell_size,
                        center+quadrant_offset(i, node_size),
                        node_size/2);
            }

        assert( false ); // Should never happen!
    }
}


bool OccupancyGrid2D::Node::encloses(
    const Eigen::Vector2f &x,
    const Eigen::Vector2f &center,
    float node_size) const throw()
{
    return ( x(0) >= center(0) - node_size/2 &&
             x(0) < center(0) + node_size/2 &&
             x(1) >= center(1) - node_size/2 &&
             x(1) < center(1) + node_size/2 );
}


void OccupancyGrid2D::Node::render(
    Batch &batch,
    const Camera &camera,
    const RGBColor &color_base,
    const RGBColor &color_slope,
    const Eigen::Vector2f &center,
    float node_size,
    const Frustum &frustum,
    int plane_mask) throw()
{
    // If it's a leaf, draw it - don't care about culling, rendering
    // an extra quad is cheaper by far
    if( is_leaf() )
        render_no_cull(batch, color_base, color_slope,
                       center, node_size);
    else // Check to see if we can render/cull the subtree in its entirety
    {
        BoundingSphere bsphere = BoundingSphere(
            Eigen::Vector3f(center(0), center(1), 0), node_size*M_SQRT2);

        // Cull the node?
        IntersectionTestResult culltest = bsphere.contained_in(
            frustum.get_bounding_sphere());


        if( culltest != DISJOINT ) // Refine cull test?
            culltest = frustum.is_in_frustum(bsphere, plane_mask);

        if( culltest == DISJOINT )
            return; // Cull all child nodes - yay!
        else if( culltest == INSIDE )//|| node_size < 0.1f ) // skip culling on small cells
            render_no_cull(batch, color_base, color_slope, center, node_size);
        else // culltest == INTERSECT
        {
            // Partial intersection - render/cull subnodes
            for( int i = 0; i < 4; i++ )
                if( m_children[i] )
                {
                    m_children[i]->render(
                        batch, camera,
                        color_base, color_slope,
                        center+quadrant_offset(i, node_size), node_size/2,
                        frustum, plane_mask);
                }
        }
    }
}


void OccupancyGrid2D::Node::render_no_cull(
    Batch &batch,
    const RGBColor &color_base,
    const RGBColor &color_slope,
    const Eigen::Vector2f &center,
    float node_size) throw()
{
    if( is_leaf() )
    {
        if( m_belief != UNKNOWN )
        {
            batch.add_quad(
                center(0), center(1), node_size,
                std::min(255.0f, 255*color_base.r + color_slope.r * m_belief),
                std::min(255.0f, 255*color_base.g + color_slope.g * m_belief),
                std::min(255.0f, 255*color_base.b + color_slope.b * m_belief));
        }
    }
    else
    {
        for( int i = 0; i < 4; ++i )
            if( m_children[i] )
                m_children[i]->render_no_cull(
                    batch, color_base, color_slope,
                    center+quadrant_offset(i, node_size), node_size/2);
    }
}

// opt. idea: delay cell merging until "all" updates have been done,
// so we avoid merging only to split in the next update
void OccupancyGrid2D::Node::optimize(Node *node)
{
    while( node->m_parent )
    {
        assert( node->is_leaf() );

        Node *parent = node->m_parent;

        if( parent == 0 )
            return; // We reached the root node

        assert( parent->m_children != 0 ); // parent can't be a leaf

        //
        // We can merge four siblings iff:
        //  1) all siblings are non-null (e.g. not UNKNOWN)
        //  2) they have the same value (belief)
        //  3) they are all leaf nodes
        //
        for( int i = 0; i < 4; ++i )
            if( !parent->m_children[i] || // 1)
                parent->m_children[i]->m_belief != node->m_belief || // 2)
                !parent->m_children[i]->is_leaf() ) // 3)
                return; // Can't merge, sibling cells have different configurations

        // Merging is ok! Commence merge!

        parent->m_belief = node->m_belief;

        for( int i = 0; i < 4; ++i )
        {
            // Invariant: parent->m_children[i] != 0, due to 1)
            delete parent->m_children[i];
            parent->m_children[i] = 0;
        }

        delete[] parent->m_children;
        parent->m_children = 0;

        node = parent;
    }
}


//
// -------------------- OccupancyGrid2D::Batch implementation -----------------
//

OccupancyGrid2D::Batch::Batch(PrepareRenderContext &context, size_t batch_size)
    : m_context(context),
      m_batch_size(batch_size),
      m_vertices(0), m_colors(0)
{
    allocate_buffers();
}


OccupancyGrid2D::Batch::~Batch()
{
    flush();
}


void OccupancyGrid2D::Batch::add_quad(
    double x_c, double y_c,
    double size,
    boost::uint8_t r, boost::uint8_t g, boost::uint8_t b)
{
    if( m_used == m_batch_size )
    {
        flush();
        allocate_buffers();
    }

    assert( m_vertices != 0 );
    assert( m_colors != 0 );

    for( size_t l = 0; l < 2*3; ++l )
    {
        m_colors[2*3*3*m_used + 3*l + 0] = r;
        m_colors[2*3*3*m_used + 3*l + 1] = g;
        m_colors[2*3*3*m_used + 3*l + 2] = b;
    }

    size_t k = 2 * 3*3 * m_used;

    // First tri
    m_vertices[k++] = x_c - size/2;
    m_vertices[k++] = y_c - size/2;
    m_vertices[k++] = 0;

    m_vertices[k++] = x_c + size/2;
    m_vertices[k++] = y_c - size/2;
    m_vertices[k++] = 0;

    m_vertices[k++] = x_c - size/2;
    m_vertices[k++] = y_c + size/2;
    m_vertices[k++] = 0;

    // Second tri
    m_vertices[k++] = x_c + size/2;
    m_vertices[k++] = y_c - size/2;
    m_vertices[k++] = 0;

    m_vertices[k++] = x_c + size/2;
    m_vertices[k++] = y_c + size/2;
    m_vertices[k++] = 0;

    m_vertices[k++] = x_c - size/2;
    m_vertices[k++] = y_c + size/2;
    m_vertices[k++] = 0;

    ++m_used;
}


void OccupancyGrid2D::Batch::flush()
{
    if( m_used<1 )
        return;

    assert( m_vertices );
    assert( m_colors );

    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer(true));
    vb->set_usage_hint(VertexBuffer::STREAM_DRAW);
    vb->set_data(m_vertices, m_used * sizeof(float) * 2*3*3);

    boost::shared_ptr<VertexBuffer> cb(new VertexBuffer(true));
    cb->set_usage_hint(VertexBuffer::STREAM_DRAW);
    cb->set_data(m_colors, m_used * 2*3*3);

    TriMesh *mesh = new TriMesh(
        vb,
        boost::shared_ptr<VertexBuffer>(),
        cb,
        boost::shared_ptr<VertexBuffer>(),
        boost::shared_ptr<IndexBuffer>(),
        0, 2*m_used);

    mesh->get_state().set(new statelets::Lighting(false));
    mesh->get_state().set(new statelets::Culling(false));

    m_context.prepare(mesh, true);

    delete[] m_vertices;
    m_vertices = 0;

    delete[] m_colors;
    m_colors = 0;
}


void OccupancyGrid2D::Batch::allocate_buffers()
{
    const size_t N = 2 * 3*3 * m_batch_size;

    m_vertices = new float[N];
    m_colors = new boost::uint8_t[N];

    m_used = 0;
}
