/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "OccupancyGrid3D.hh"
#include "Camera.hh"
#include "../RenderContext.hh"
#include "../PrepareRenderContext.hh"
#include "../VertexBuffer.hh"
#include "../Frustum.hh"
#include "../Mesh.hh"
#include "../Statelet.hh"
#include "Og3dNode.hh"
#include "Og3dCache.hh"
#include "Og3dTraversalContext.hh"
#include "Og3dRenderContext.hh"

#include <cmath>
#include <cassert>
#include <GL/glew.h>
#include <boost/scoped_ptr.hpp>
#include <boost/bimap.hpp>
#include <Eigen/LU>


using namespace peekabot;
using namespace peekabot::renderer;


OccupancyGrid3D::OccupancyGrid3D(float cell_xy_size, float cell_z_size)
    : m_root_node(0),
      m_cell_size(cell_xy_size),
      m_height_factor(cell_z_size/cell_xy_size),
      m_color_mapping_enabled(false),
      m_z_min(0), m_z_max(0)
{
}


OccupancyGrid3D::~OccupancyGrid3D()
{
    clear();
}


void OccupancyGrid3D::set_belief(const Eigen::Vector3f &x, float belief)
{
    if( belief < 0 || belief > 1 )
        throw std::domain_error("Belief must be in the range [0,1]");

    // If the tree is empty, create the root node at the position x
    if( !m_root_node )
    {
        m_root_cell_size = m_cell_size;
        m_root_cell_center = x;
        m_root_node = new Og3dNode;
    }
    else
    {
        if( !m_root_node->encloses(
                x, m_root_cell_center, m_root_cell_size, m_height_factor) )
            grow_to_accomodate(x, m_height_factor);
    }

    // Discretisize the belief (so we have a higher chance of
    // merging them later on, and for reduced storage)
    boost::uint8_t bel = (boost::uint8_t)roundf(belief*255);

    Og3dTraversalContext tc(m_cell_size, m_height_factor);
    tc.push_parent(0);

    m_root_node->set_belief(
        tc, x, bel,
        m_root_cell_center, m_root_cell_size);
}


void OccupancyGrid3D::set_cells(
    const std::vector<std::pair<Eigen::Vector3f, float> > &cells)
{
    for( std::size_t i = 0; i < cells.size(); ++i )
    {
        set_belief(cells[i].first, cells[i].second);
    }
}


void OccupancyGrid3D::clear()
{
    if( m_root_node )
    {
        delete m_root_node;
        m_root_node = 0;
    }
}


void OccupancyGrid3D::get_renderables(PrepareRenderContext &context) const
{
    TheOg3dCache::instance().begin_frame();

    if( !m_root_node )
    {
        // The occupancy grid is empty
        return;
    }

    //
    // Translate the camera's frustum bsphere and frustum planes
    // to local coordinates (for culling in local space!)
    //

    boost::scoped_ptr<Camera> cam(context.get_camera()->clone());
    cam->set_transformation(
        Eigen::Transform3f(get_transformation().inverse(Eigen::Isometry)) *
        context.get_camera()->get_transformation());

    // Compute the traversal order
    Eigen::Vector3f view_dir(
        cam->get_transformation()(0,0),
        cam->get_transformation()(1,0),
        cam->get_transformation()(2,0));
    std::vector<std::pair<float, int> > tmp(8);

    float z_sign = get_template_state().is_set<statelets::Alpha>() ? 1 : -1;
    for( int i = 0; i < 8; i++ )
    {
        float z_dist = view_dir.dot(
            view_dir + m_root_node->octant_offset(i, 1, m_height_factor));
        tmp[i] = std::make_pair(z_sign*z_dist, i);
    }

    std::sort(tmp.begin(), tmp.end());

    int order[8];
    for( int i = 0; i < 8; i++ )
    {
        order[i] = tmp[i].second;
    }

    Og3dTraversalContext tc(m_cell_size, m_height_factor);

    Og3dRenderContext rc(
        context, *cam, order,
        m_height_factor,
        m_color_mapping_enabled, m_z_min, m_z_max);

    // Render/cull the tree recursively
    m_root_node->render(tc, rc, m_root_cell_center, m_root_cell_size, 0x3F);
}


void OccupancyGrid3D::set_color_mapping(bool enable, float z_min, float z_max)
{
    m_color_mapping_enabled = enable;
    m_z_min = z_min;
    m_z_max = z_max;

    TheOg3dCache::instance().clear();
}


void OccupancyGrid3D::calculate_bvs() throw()
{
    const Eigen::Vector3f &cp = m_root_cell_center;

    set_bounding_sphere(
        BoundingSphere(
            Eigen::Vector3f(cp(0), cp(1), cp(2)),
            sqrtf(2+m_height_factor*m_height_factor) * m_root_cell_size));
}


Og3dNode *OccupancyGrid3D::grow_to_accomodate(
    const Eigen::Vector3f &x, float hf) throw()
{
    // (x, y) outside the octree!
    // deal with special case... by extending the tree outwards,
    // that is, by create a new big node and use the current
    // root node as a child quadrant in the new node.
    while( !m_root_node->encloses(x, m_root_cell_center, m_root_cell_size, hf) )
    {
        Og3dNode *old_root = m_root_node;
        float old_root_cell_size = m_root_cell_size;
        Eigen::Vector3f old_root_center = m_root_cell_center;

        Eigen::Vector3f configuration[] =
        {
            Eigen::Vector3f(-1, -1, -hf),
            Eigen::Vector3f( 1, -1,  hf),
            Eigen::Vector3f(-1,  1, -hf),
            Eigen::Vector3f( 1,  1,  hf),
            Eigen::Vector3f(-1, -1, -hf),
            Eigen::Vector3f( 1, -1,  hf),
            Eigen::Vector3f(-1,  1, -hf),
            Eigen::Vector3f( 1,  1,  hf)
        };

        float min_dist = 0;
        int min_conf = -1;

        for( int i = 0; i < 8; i++ )
        {
            Eigen::Vector3f confpos = ( old_root_center +
                                 old_root_cell_size/2*configuration[i] );

            float d = (confpos-x).norm();

            if( min_conf == -1 || d < min_dist )
            {
                min_conf = i;
                min_dist = d;
            }
        }

        // Now we just have to figure out in which directions
        // to extend the octree.
        // This can easily be done by find the configuration p
        // which minimizes the distance dist(<x,y>,p). There
        // are 4 possible configurations.
        //
        // The different possible configurations are the
        // permutations of <x,y>, x,y \in {-1,1}. I.e.
        // all the ways the new square can be offset from the
        // old root node.
        //
        // Having found the best configuration, we implicitly
        // also have the index of the new root node to put the
        // old one in.
        // index = (x/2+0.5)+2*(y/2+0.5)

        m_root_node = new Og3dNode;
        m_root_node->m_belief = old_root->m_belief;
        m_root_cell_size *= 2;
        m_root_cell_center += old_root_cell_size/2*configuration[min_conf];

        m_root_node->m_children = new Og3dNode*[8];
        for( int i = 0; i < 8; ++i )
            m_root_node->m_children[i] = 0;

        int index = -1;
        for( int i = 0; i < 8; ++i )
        {
            if( (m_root_node->octant_offset(
                     i, m_root_cell_size, m_height_factor) +
                 m_root_cell_center).isApprox(
                     old_root_center,
                     m_root_cell_size * 1e-6) )
            {
                index = i;
                break;
            }
        }

        assert( index != -1 );

        m_root_node->m_children[index] = old_root;
    }

    // Update BVs to contain the new root
    calculate_bvs();

    return m_root_node;
}
