/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_3D_HH_INCLUDED
#define PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_3D_HH_INCLUDED


#include <Eigen/Core>
#include <boost/cstdint.hpp>

#include "../../Types.hh"
#include "../CullableEntity.hh"


namespace peekabot
{
    namespace renderer
    {
        class Og3dNode;

        /**
         * \brief An object representing a 3D occupancy grid.
         *
         * The functionality to implement the object is split up into a few
         * different classes to make it easier to read and maintain:
         *
         * \li OccupancyGrid3D - provides the public interface along with some
         * internal functionality such as growing the occupancy grid when
         * setting the belief of a cell outside the current tree bounds and
         * calculating bounding volumes for the object.
         * \li Og3dNode - implements the Octree which actually stores the map.
         * \li Og3dTraversalContext - provides various information needed
         * when traversing the tree. We don't want to store parent pointers
         * etc. in the nodes themselves and thus need to pass needed
         * information along when traversing the tree. Consolidating all this
         * data in one object makes the code cleaner, and reduces the amount of
         * parameters the need to be passed on the stack.
         * \li Og3dRenderContext - contains code to do most of the actual
         * rendering.
         * \li Og3dCache - provides a caching mechanism to speed up rendering.
         */
        class OccupancyGrid3D : public CullableEntity
        {
        public:
            OccupancyGrid3D(float cell_xy_size, float cell_z_size);

            ~OccupancyGrid3D();

            void set_belief(const Eigen::Vector3f &x, float belief);

            void set_cells(
                const std::vector<std::pair<Eigen::Vector3f, float> > &cells);

            void clear();

            virtual void get_renderables(PrepareRenderContext &context) const;

            void set_color_mapping(bool enable, float z_min, float z_max);

        private:
            Og3dNode *grow_to_accomodate(const Eigen::Vector3f &x, float hf) throw();

            void calculate_bvs() throw();

        private:
            /**
             * \brief The root node of the octree, or null iff the octree is
             * empty.
             */
            Og3dNode *m_root_node;

            float m_cell_size;

            /**
             * \brief m_height_factor multiplied with the cell size determines
             * the height of the OG-map cells.
             */
            float m_height_factor;

            float m_root_cell_size;

            Eigen::Vector3f m_root_cell_center;

            bool m_color_mapping_enabled;

            float m_z_min, m_z_max;
        };
    }
}


#endif // PEEKABOT_RENDERER_PRIMITIVES_OCCUPANCY_GRID_3D_HH_INCLUDED
