/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Og3dCache.hh"
#include "Og3dNode.hh"
#include "Og3dRenderContext.hh"
#include "../VertexBuffer.hh"
#include "../Mesh.hh"

#include <cassert>


using namespace peekabot;
using namespace peekabot::renderer;


Og3dCache::Og3dCache()
    : m_max_size(128*1024*1024),
      m_cached_voxels(0),
      m_frame_no(0)
{
}


Og3dCache::~Og3dCache()
{
}


bool Og3dCache::cache(
    Og3dRenderContext &rc,
    const Og3dNode *subtree, const std::size_t n,
    const Eigen::Vector3f &center, float node_size)
{
    assert( n > 0 );
    assert( subtree );

    // If the cache is full, replace the last recently used entry, provide
    // that it has not been used to render the last X frames, in which case
    // we'll deny the caching request instead
    while( size() + 1008*n > get_max_size() )
    {
        if( m_entries.empty() )
            return false;

        CacheEntry *lru = m_lru_pq.back();
        if( m_frame_no - lru->m_last_used < 3 )
            return false;

        CacheEntries::right_map::iterator it = m_entries.right.find(lru);
        assert( it != m_entries.right.end() );
        evict(it->second);
    }

    //
    // Construct the data to cache
    //
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    vb->set_usage_hint(VertexBuffer::STATIC_DRAW);

    try
    {
        vb->resize(1008*n);

        boost::uint8_t *p = (boost::uint8_t *)vb->map(true);

        rc.render_subtree_to_buffer(subtree, center, node_size, p);

        vb->unmap();
    }
    catch(...)
    {
        // Probably an "out of memory" error
        return false;
    }

    CacheEntry *entry = new CacheEntry;
    m_lru_pq.push_front(entry);

    entry->m_last_used = m_frame_no;
    entry->m_size = n;
    entry->m_buf = vb;
    entry->m_lru_it = m_lru_pq.begin();
    entry->m_color_mapping_enabled = rc.color_mapping_enabled();
    m_cached_voxels += n;

    m_entries.insert(CacheEntries::value_type(subtree, entry));

    assert( m_entries.size() == m_lru_pq.size() );

    return true;
}


void Og3dCache::evict(const Og3dNode *subtree)
{
    CacheEntries::left_map::iterator it = m_entries.left.find(subtree);
    if( it == m_entries.left.end() )
        return;

    m_cached_voxels -= it->second->m_size;

    // Update the LRU priority queue
    m_lru_pq.erase(it->second->m_lru_it);
    delete it->second;

    m_entries.left.erase(it);

    assert( m_entries.size() == m_lru_pq.size() );
}


bool Og3dCache::is_cached(const Og3dNode *subtree) const
{
    CacheEntries::left_map::const_iterator it =
        m_entries.left.find(subtree);
    return it != m_entries.left.end();
}


TriMesh *Og3dCache::get_cached_mesh(const Og3dNode *subtree)
{
    CacheEntries::left_map::iterator it = m_entries.left.find(subtree);
    if( it == m_entries.left.end() )
        return 0;

    CacheEntry *entry = it->second;

    TriMesh *mesh = new TriMesh;
    // The stride is 3*4 (coordinates) + 3*4 (normal) + 4 (color) bytes
    int stride = 12 + 12 + 4;
    mesh->set_vertices(entry->m_buf, stride, 0);
    mesh->set_normals(entry->m_buf, stride, 12);
    if( entry->m_color_mapping_enabled )
        mesh->set_colors(entry->m_buf, stride, 24);
    mesh->set_element_count(6*2*entry->m_size);

    // Update the LRU priority queue
    m_lru_pq.erase(entry->m_lru_it);
    m_lru_pq.push_front(entry);
    entry->m_lru_it = m_lru_pq.begin();

    assert( m_entries.size() == m_lru_pq.size() );

    return mesh;
}


void Og3dCache::set_max_size(std::size_t n)
{
    m_max_size = n;

    while( size() > get_max_size() )
    {
        assert( !m_entries.empty() );

        CacheEntry *lru = m_lru_pq.back();
        CacheEntries::right_map::iterator it = m_entries.right.find(lru);
        assert( it != m_entries.right.end() );
        evict(it->second);
    }
}


std::size_t Og3dCache::get_max_size() const
{
    return m_max_size;
}


std::size_t Og3dCache::size() const
{
    // Note: see the comment in cache() for where 1008 comes from
    return 1008 * m_cached_voxels;
}


void Og3dCache::clear()
{
    while( !m_entries.empty() )
        evict(m_entries.left.begin()->first);

    assert( size() == 0 );
    assert( m_entries.empty() );
    assert( m_lru_pq.empty() );
}


void Og3dCache::begin_frame()
{
    if( ++m_frame_no == 0 )
    {
        // Overflow! We could be smart about this, or we can just clear the
        // cache. When sizeof(size_t)=8 this is something that will happen
        // every 19498080578:th year or so, so there's little incentive to
        // be smart about it.
        clear();
    }
}


bool Og3dCache::will_cache(std::size_t n) const
{
    if( size() + 1008*n > get_max_size() )
    {
        if( m_entries.empty() )
            return false;
        else
            return m_frame_no - m_lru_pq.back()->m_last_used >= 3;
    }
    else
    {
        return true;
    }
}
