/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_OG3D_CACHE_HH_INCLUDED
#define PEEKABOT_RENDERER_OG3D_CACHE_HH_INCLUDED


#include "../../Singleton.hh"

#include <list>
#include <cstddef>
#include <Eigen/Core>
#include <boost/cstdint.hpp>
#include <boost/bimap.hpp>
#include <boost/shared_ptr.hpp>


namespace peekabot
{
    namespace renderer
    {
        class TriMesh;
        class VertexBuffer;
        class Og3dNode;
        class Og3dRenderContext;

        /**
         * \brief A subtree cache for OccupancyGrid3D, used to speed up
         * rendering.
         *
         * The Og3dCache class caches the visual representation (vertices,
         * normals, colors) of Og3dNode subtrees in video memory for fast
         * rendering upon subsequent use.
         *
         * Logic for determining which subtrees to cache, and how the cached
         * results are used, is implemented in Og3dŃode.
         */
        class Og3dCache
        {
        public:
            Og3dCache();

            ~Og3dCache();

            /**
             * \brief Request for the given \c subtree to be added to the cache.
             *
             * \pre n > 0
             *
             * \return \c true is returned iff the subtree was added to the cache.
             */
            bool cache(
                Og3dRenderContext &rc,
                const Og3dNode *subtree, const std::size_t n,
                const Eigen::Vector3f &center, float node_size);

            /**
             * \brief Evict \c subtree from the cache. The operation is a
             * no-op if the given subtree is not cached.
             */
            void evict(const Og3dNode *subtree);

            /**
             * \brief Returns true iff the given subtree is cached.
             */
            bool is_cached(const Og3dNode *subtree) const;

            /**
             * \brief Returns true iff the cache will accept a caching request
             * for a subtree of size \c n.
             */
            bool will_cache(std::size_t n) const;

            /**
             * \brief Get the cached mesh for the given subtree, if cached.
             *
             * \return If the \c subtree is cached, it's cached mesh is
             * returned. If the \c subtree is not cached, null is returned.
             */
            TriMesh *get_cached_mesh(const Og3dNode *subtree);

            /**
             * \brief Set the max cache size, in bytes.
             *
             * \post <tt>size() <= get_max_size()</tt>
             */
            void set_max_size(std::size_t n);

            /**
             * \brief Get the cache's maximum size, in bytes.
             */
            std::size_t get_max_size() const;

            /**
             * \brief Get the cache's current size, in bytes.
             */
            std::size_t size() const;

            /**
             * \brief Clear the cache.
             *
             * \post <tt>size() == 0</tt>
             */
            void clear();

            void begin_frame();

        private:
            struct CacheEntry;

            /**
             * \brief We use a sorted list (MRU first, LRU last) as our LRU
             * priority queue, as it provides decent complexity guarantees for
             * use case.
             *
             * To find the iterators efficiently (naively a linear time
             * operation) we store the iterators associated with each
             * CacheEntry in our \c m_entries bimap.
             * This gives us O(log n) look-up (with the potential for O(1)
             * look-up when using an unordered map), O(1) insertion and erase
             * complexity.
             */
            typedef std::list<CacheEntry *> LruPq;

            LruPq m_lru_pq;

            struct CacheEntry
            {
                boost::shared_ptr<VertexBuffer> m_buf;

                /**
                 * \brief Frame number when the cache entry was last used.
                 */
                std::size_t m_last_used;

                /**
                 * \brief The number of voxels in the cache.
                 */
                std::size_t m_size;

                /**
                 * \brief The iterator to the CacheEntry's LruPq entry.
                 */
                LruPq::iterator m_lru_it;

                bool m_color_mapping_enabled;
            };

            typedef boost::bimap<const Og3dNode *, CacheEntry *> CacheEntries;

            CacheEntries m_entries;

            std::size_t m_max_size;

            std::size_t m_cached_voxels;

            std::size_t m_frame_no;
        };

        typedef Singleton<Og3dCache> TheOg3dCache;
    }
}


#endif // PEEKABOT_RENDERER_OG3D_CACHE_HH_INCLUDED
