/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Og3dNode.hh"
#include "Og3dCache.hh"
#include "Og3dTraversalContext.hh"
#include "Og3dRenderContext.hh"
#include "../PrepareRenderContext.hh"
#include "../Mesh.hh"

#include <cassert>


using namespace peekabot;
using namespace peekabot::renderer;


namespace
{
    const std::size_t MIN_CACHE_SIZE = 100; // ~100 KiB
    const std::size_t MAX_CACHE_SIZE = 20000; // ~20 MiB
}


Og3dNode::Og3dNode()
    : m_children(0),
      m_belief(0),
      m_counter(0)
{
}


Og3dNode::~Og3dNode()
{
    TheOg3dCache::instance().evict(this);

    if( m_children )
    {
        for( int i = 0; i < 8; i++ )
        {
            if( m_children[i] != 0 )
            {
                delete m_children[i];
                m_children[i] = 0;
            }
        }

        delete[] m_children;
        m_children = 0;
    }
}


void Og3dNode::set_belief(
    Og3dTraversalContext &tc,
    const Eigen::Vector3f &x,
    boost::uint8_t belief,
    const Eigen::Vector3f &center, float node_size)
{
    //assert(encloses(x));

    TheOg3dCache::instance().evict(this);
    m_counter = 0;

    // TODO: push/pop_parent() stuff

    if( is_leaf() ) // Base case
    {
        if( m_belief == belief ) // TODO: use a tolerance here!
        {
            return;
            // Note: there's no need to check for merge potential here
            // since we haven't changed the tree in any way
        }
        // Is this a true leaf node? (m_node_side == cell_size)
        else if( node_size <= 1.1f*tc.get_cell_size() ) // mul by 1.1 for robustness
        {
            m_belief = belief;

            update_subtree_beliefs(tc);

            // Check and merge sibling cells if possible
            optimize(tc, this);
        }
        else
        {
            // It's a merged cell - split it up

            m_children = new Og3dNode*[8];
            for( int i = 0; i < 8; ++i )
                m_children[i] = 0;

            tc.push_parent(this);

            for( int i = 0; i < 8; ++i )
            {
                bool in_octant = enclosed_in_octant(x, i, center);

                if( in_octant )
                {
                    m_children[i] = new Og3dNode;
                    m_children[i]->m_belief = m_belief;

                    // Propagate new value down the tree, splitting
                    // cells as/if needed
                    if( in_octant )
                    {
                        m_children[i]->set_belief(
                            tc, x, belief, center+octant_offset(
                                i, node_size, tc.get_height_factor()),
                            node_size/2);
                    }
                }
            }

            tc.pop_parent();
        }
    }
    else // Recursion case
    {
        assert( m_children != 0 );

        // Note: in reality it's 2, but we use 1.9 for robustness
        assert( node_size >= 1.9f*tc.get_cell_size() );

        tc.push_parent(this);

        // Find the containing child cell and recurse
        for( int i = 0; i < 8; ++i )
        {
            if( enclosed_in_octant(x, i, center) )
            {
                if( !m_children[i] )
                    m_children[i] = new Og3dNode;

                m_children[i]->set_belief(
                    tc, x, belief, center+octant_offset(
                        i, node_size, tc.get_height_factor()), node_size/2);

                tc.pop_parent();
                return;
            }
        }

        assert( false ); // Should never happen!
    }
}


bool Og3dNode::encloses(
    const Eigen::Vector3f &x,
    const Eigen::Vector3f &center, float node_size,
    float hf) const throw()
{
    return ( x(0) >= center(0) - node_size/2 &&
             x(0) < center(0) + node_size/2 &&
             x(1) >= center(1) - node_size/2 &&
             x(1) < center(1) + node_size/2 &&
             x(2) >= center(2) - hf*node_size/2 &&
             x(2) < center(2) + hf*node_size/2 );
}


void Og3dNode::render(
    Og3dTraversalContext &tc, Og3dRenderContext &rc,
    const Eigen::Vector3f &center, float node_size,
    int plane_mask)
{
    if( get_belief() <= 127 )
    {
        return;
    }
    else if( is_leaf() )
    {
        // If it's a leaf, draw it - don't care about culling, rendering
        // a few extra tris is probably cheaper
        //
        // Also don't care about trying to use the cache, or trying to cache
        // it, since it's only a single node
        render_no_cull(tc, rc, center, node_size);
    }
    else // Check to see if we can render/cull the subtree in its entirety
    {
        BoundingSphere bsphere = BoundingSphere(
            Eigen::Vector3f(center(0), center(1), center(2)),
            node_size*rc.get_size_to_bsphere_radius_factor());

        // Cull the node?
        IntersectionTestResult culltest = bsphere.contained_in(
            rc.get_frustum().get_bounding_sphere());


        if( culltest != DISJOINT ) // Refine cull test?
            culltest = rc.get_frustum().is_in_frustum(bsphere, plane_mask);

        if( culltest == DISJOINT )
        {
            return; // Cull all child nodes - yay!
        }
        else if( culltest == INSIDE ) //|| node_size < 0.1f )
        {
            TriMesh *cached_mesh =
                TheOg3dCache::instance().get_cached_mesh(this);
            if( cached_mesh )
            {
                rc.get_render_context().prepare(cached_mesh, true);
            }
            else
            {
                if( m_counter >= 5 &&
                    TheOg3dCache::instance().will_cache(MIN_CACHE_SIZE) )
                {
                    // The tree is stable (5 or more renders with no updates)
                    // and the cache might accept new cache entries, so there's a
                    // good cache we can cache the subtree.
                    std::size_t n_drawn = 0;
                    render_no_cull(tc, rc, center, node_size, &n_drawn);
                    if( n_drawn >= MIN_CACHE_SIZE && n_drawn <= MAX_CACHE_SIZE )
                        TheOg3dCache::instance().cache(
                            rc, this, n_drawn, center, node_size);
                }
                else
                {
                    render_no_cull(tc, rc, center, node_size);
                }
            }
        }
        else // culltest == INTERSECT
        {
            // Partial intersection - render the entire subtree if the node is
            // small and the subtree is cached, otherwise cull and render all
            // the subtree
            if( node_size <= 2 && TheOg3dCache::instance().is_cached(this) )
            {
                rc.get_render_context().prepare(
                    TheOg3dCache::instance().get_cached_mesh(this), true);
            }
            else
            {
                for( int j = 0; j < 8; j++ )
                {
                    int i = rc.get_render_order(j);
                    if( m_children[i] )
                    {
                        m_children[i]->render(
                            tc, rc, center+octant_offset(
                                i, node_size, tc.get_height_factor()), node_size/2,
                            plane_mask);
                    }
                }
            }
        }
    }
}


void Og3dNode::update_subtree_beliefs(Og3dTraversalContext &tc)
{
    std::size_t i = 0;
    Og3dNode *ancestor = tc.get_ancestor(i++);
    while( ancestor )
    {
        assert( !ancestor->is_leaf() );
        boost::uint8_t max_belief = 0;
        for( int j = 0; j < 8; ++j )
        {
            if( ancestor->get_child(j) )
            {
                max_belief = std::max(
                    max_belief, ancestor->get_child(j)->m_belief);
            }
        }

        ancestor->m_belief = max_belief;

        ancestor = tc.get_ancestor(i++);
    }
}


void Og3dNode::render_no_cull(
    Og3dTraversalContext &tc, Og3dRenderContext &rc,
    const Eigen::Vector3f &center, float node_size,
    std::size_t *n_drawn)
{
    m_counter = std::min<boost::int8_t>(m_counter+1, 10);

    if( get_belief() <= 127 )
    {
        return;
    }
    else if( is_leaf() )
    {
        rc.render_voxel(center, node_size);
        if( n_drawn )
            ++(*n_drawn);
    }
    else if( !n_drawn && TheOg3dCache::instance().is_cached(this) )
    {
        rc.get_render_context().prepare(
            TheOg3dCache::instance().get_cached_mesh(this), true);
    }
    else
    {
        for( int j = 0; j < 8; ++j )
        {
            int i = rc.get_render_order(j);
            if( m_children[i] )
            {
                m_children[i]->render_no_cull(
                    tc, rc, center+octant_offset(
                        i, node_size, tc.get_height_factor()), node_size/2,
                    n_drawn);
            }
        }
    }
}


// opt. idea: delay cell merging until "all" updates have been done,
// so we avoid merging only to split in the next update
void Og3dNode::optimize(Og3dTraversalContext &tc, Og3dNode *node)
{
    std::size_t i = 0;
    while( tc.get_ancestor(i) )
    {
        Og3dNode *parent = tc.get_ancestor(i);

        assert( node->is_leaf() && !parent->is_leaf() );

        //
        // We can merge four siblings iff:
        //  1) all siblings are non-null (i.e. not unknown)
        //  2) they all have the same occupancy value (belief)
        //  3) they are all leaf nodes
        //
        for( int j = 0; j < 8; ++j )
            if( !parent->m_children[j] || // 1)
                parent->m_children[j]->m_belief != node->m_belief || // 2)
                !parent->m_children[j]->is_leaf() ) // 3)
                return; // Can't merge, sibling cells have different configurations

        // Merging is ok! Commence merge!

        parent->m_belief = node->m_belief;

        for( int j = 0; j < 8; ++j )
        {
            // Invariant: parent->m_children[j] != 0, due to 1)
            delete parent->m_children[j];
            parent->m_children[j] = 0;
        }

        delete[] parent->m_children;
        parent->m_children = 0;

        node = parent;
        ++i;
    }
}
