/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Og3dRenderContext.hh"
#include "Og3dNode.hh"
#include "Camera.hh"
#include "../PrepareRenderContext.hh"
#include "../VertexBuffer.hh"
#include "../Mesh.hh"

#include <cassert>


using namespace peekabot;
using namespace peekabot::renderer;


namespace
{
    const std::size_t VOXEL_BUFFER_SIZE = 2048;
}


Og3dRenderContext::Og3dRenderContext(
    PrepareRenderContext &context,
    const Camera &cam,
    const int *order,
    float height_factor,
    bool color_mapping_enabled,
    float z_min, float z_max)
    : m_render_context(context),
      m_frustum(cam, 0.0f, 0.0f),
      m_order(order),
      m_color_mapping_enabled(color_mapping_enabled),
      m_z_min(z_min), m_z_max(z_max),
      m_hf(height_factor),
      m_bsphere_factor(sqrtf(2 + height_factor*height_factor)),
      m_voxels_buffered(0)
{
    // Pre-compute LUTs for mapping z coordinate to color
    compute_color_tabs();

    allocate_voxel_buffers();
}


Og3dRenderContext::~Og3dRenderContext()
{
    flush_voxels();
}


void Og3dRenderContext::get_color_from_z(
    float z,
    boost::uint8_t &r, boost::uint8_t &g, boost::uint8_t &b) const
{
    float z_norm = (z - m_z_min)/m_z_max;
    if( z_norm < 0 )
        z_norm = 0;
    if( z_norm > 1 )
        z_norm = 1;

    int color_tab_idx = z_norm*511;

    r = m_color_tab_r[color_tab_idx];
    g = m_color_tab_g[color_tab_idx];
    b = m_color_tab_b[color_tab_idx];
}


void Og3dRenderContext::z_to_color(
    float z,
    boost::uint8_t &r, boost::uint8_t &g, boost::uint8_t &b) const
{
    // Normalize z to the range [0, 1]
    float z_norm = (z-m_z_min)/m_z_max;

    if( z_norm < 0 )
        z_norm = 0;
    if( z_norm > 1 )
        z_norm = 1;

    // Compute the red component
    if( z_norm >= 0.2 && z_norm <= 0.6 )
        r = 0;
    else if( z_norm >= 0.8 )
        r = 255;
    else if( z_norm < 0.2 )
        r = -5 * (z_norm - 0.2) * 255;
    else // 0.6 < z_norm < 0.8
        r = 5 * (z_norm - 0.6) * 255;

    // Green
    if( z_norm <= 0.2 )
        g = 0;
    else if( z_norm >= 0.4 && z_norm <= 0.8 )
        g = 255;
    else if( z_norm < 0.4 )
        g = 5 * (z_norm - 0.2) * 255;
    else // z_norm > 0.8
        g = -5 * (z_norm - 1.0) * 255;

    // Blue
    if( z_norm <= 0.4 )
        b = 255;
    else if( z_norm >= 0.6 )
        b = 0;
    else // 0.4 < z_norm < 0.6
        b = -5 * (z_norm - 0.6) * 255;
}


void Og3dRenderContext::compute_color_tabs()
{
    for( int i = 0; i < 512; ++i )
    {
        float z = (m_z_max-m_z_min)*i/(512-1) + m_z_min;
        z_to_color(z, m_color_tab_r[i], m_color_tab_g[i], m_color_tab_b[i]);
    }
}


void Og3dRenderContext::render_voxel(
    const Eigen::Vector3f &center, double node_size)
{
    if( m_voxels_buffered == VOXEL_BUFFER_SIZE )
    {
        flush_voxels();
        allocate_voxel_buffers();
    }

    render_voxel_to_buffer(m_voxel_vp, center, node_size);

    ++m_voxels_buffered;
}


void Og3dRenderContext::render_subtree_to_buffer(
    const Og3dNode *node,
    const Eigen::Vector3f &center, float node_size,
    boost::uint8_t * &buf) const
{
    // If no nodes in tree are occupied, bail out early
    if( node->get_belief() <= 127 )
    {
        return;
    }
    else if( node->is_leaf() )
    {
        render_voxel_to_buffer(buf, center, node_size);
    }
    else
    {
        for( int i = 0; i < 8; ++i )
        {
            if( node->get_child(i) )
            {
                render_subtree_to_buffer(
                    node->get_child(i),
                    center+node->octant_offset(i, node_size, m_hf),
                    node_size/2, buf);
            }
        }
    }
}


void Og3dRenderContext::render_voxel_to_buffer(
    boost::uint8_t * &buf,
    const Eigen::Vector3f &center, float node_size) const
{
    const float hs = node_size/2;
    const float hsz = hs*m_hf;
    const Eigen::Vector3f vertices[] = {
        Eigen::Vector3f(center(0) + hs, center(1) + hs, center(2) + hsz),
        Eigen::Vector3f(center(0) + hs, center(1) + hs, center(2) - hsz),
        Eigen::Vector3f(center(0) + hs, center(1) - hs, center(2) + hsz),
        Eigen::Vector3f(center(0) + hs, center(1) - hs, center(2) - hsz),
        Eigen::Vector3f(center(0) - hs, center(1) + hs, center(2) + hsz),
        Eigen::Vector3f(center(0) - hs, center(1) + hs, center(2) - hsz),
        Eigen::Vector3f(center(0) - hs, center(1) - hs, center(2) + hsz),
        Eigen::Vector3f(center(0) - hs, center(1) - hs, center(2) - hsz)
    };

    boost::uint8_t r, g, b;
    get_color_from_z(center(2), r, g, b);

    // Front
    render_voxel_face(
        buf, vertices,
        0, 2, 3, 1,
        Eigen::Vector3f(1,0,0),
        r, g, b);

    // Back
    render_voxel_face(
        buf, vertices,
        6, 4, 5, 7,
        Eigen::Vector3f(-1,0,0),
        r, g, b);

    // Right
    render_voxel_face(
        buf, vertices,
        4, 0, 1, 5,
        Eigen::Vector3f(0,1,0),
        r, g, b);

    // Left
    render_voxel_face(
        buf, vertices,
        2, 6, 7, 3,
        Eigen::Vector3f(0,-1,0),
        r, g, b);

    // Top
    render_voxel_face(
        buf, vertices,
        0, 4, 6, 2,
        Eigen::Vector3f(0,0,1),
        r, g, b);

    // Bottom
    render_voxel_face(
        buf, vertices,
        3, 7, 5, 1,
        Eigen::Vector3f(0,0,-1),
        r, g, b);
}


void Og3dRenderContext::render_voxel_face(
    boost::uint8_t * &buf,
    const Eigen::Vector3f *vertices,
    int i0, int i1, int i2, int i3,
    const Eigen::Vector3f &normal,
    float r, float g, float b)
{
    const int idx[] = { i0, i1, i2, i2, i3, i0 };

    for( int j = 0; j < 6; ++j )
    {
        const Eigen::Vector3f &v = vertices[idx[j]];

        /*Eigen::Vector3f *pp = (Eigen::Vector3f *)buf;
        *pp = v;
        *(pp+1) = normal;*/

        float *fp = (float *)buf;

        // Coordinates
        for( int i = 0; i < 3; ++i )
            *(fp++) = v(i);

        // Normal
        for( int i = 0; i < 3; ++i )
            *(fp++) = normal(i);

        buf += 24;

        *(buf++) = r;
        *(buf++) = g;
        *(buf++) = b;
        *(buf++) = 0;
    }
}


void Og3dRenderContext::flush_voxels()
{
    if( m_voxel_vb->is_mapped() )
        m_voxel_vb->unmap();

    if( m_voxels_buffered < 1 )
        return;

    TriMesh *mesh = new TriMesh();
    // The stride is 3*4 (coordinates) + 3*4 (normal) + 4 (color) bytes
    int stride = 12 + 12 + 4;
    mesh->set_vertices(m_voxel_vb, stride, 0);
    mesh->set_normals(m_voxel_vb, stride, 12);
    if( m_color_mapping_enabled )
        mesh->set_colors(m_voxel_vb, stride, 24);

    // We're rendering 6*2 tris per voxel (6 faces and 2 triangles per face)
    mesh->set_element_count(6*2*m_voxels_buffered);

    m_render_context.prepare(mesh, true);
}


void Og3dRenderContext::allocate_voxel_buffers()
{
    // Allocate a new system memory buffer, not a VBO!
    // We want to avoid VBOs since we can only map one VBO at the time, and if
    // we were to map a VBO here it would interfere with the caching mechanism,
    // which uses VBO mapping
    m_voxel_vb.reset(new VertexBuffer(true));
    m_voxel_vb->set_usage_hint(VertexBuffer::STREAM_DRAW);

    // Each cube has 6 faces, each face consists of 2 triangles with 2
    // vertices/colors/normals per triangle. In total we have 6*2*3*3
    // triplets, where 2 triplets are of type float and one of type uint8
    // (for colors).
    m_voxel_vb->resize(1008*VOXEL_BUFFER_SIZE);
    m_voxel_vp = (boost::uint8_t *)m_voxel_vb->map(true);

    m_voxels_buffered = 0;
}
