/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_RENDERER_OG3D_RENDER_CONTEXT_HH_INCLUDED
#define PEEKABOT_RENDERER_OG3D_RENDER_CONTEXT_HH_INCLUDED


#include "../Frustum.hh"

#include <Eigen/Core>
#include <boost/cstdint.hpp>
#include <boost/shared_ptr.hpp>


namespace peekabot
{
    namespace renderer
    {
        class PrepareRenderContext;
        class Frustum;
        class Camera;
        class VertexBuffer;
        class Og3dNode;

        /**
         *
         * \par Vertex data format
         *
         * All rendering (render_subtree_to_buffer(), render_voxel()) uses the
         * same vertex data format, which contains of coordinates, normals and
         * colors. Coordinates and normals are stored in floats, colors are
         * stored in unsigned bytes. The vertex data is stored in an
         * interleaved fashion:
         *
         * \verbatim
           +-----------+------------+-------------+
           |   x,y,z   |  nx,ny,nz  |   r,g,b,a   |
           +-----------+------------+-------------+
              12 bytes    12 bytes      4 bytes
           \endverbatim
         *
         * Even if we don't use the alpha component we want to keep it around
         * to get 32-bit alignment -- which might improve performance depending
         * on the driver. It also makes the code more portable since not all
         * CPUs can access 32-bit values at byte aligned addresses (e.g ARM).
         *
         * Each rendered voxel has 6 faces and each face consists of 2
         * triangles, giving us a total of 6*2 triangles per voxel. In turn,
         * each triangle consists of 3 vertices, where each vertex consists of
         * the data outlined above. I.e. we use 84 bytes per triangle and 1008
         * bytes per voxel.
         */
        class Og3dRenderContext
        {
        public:
            Og3dRenderContext(
                PrepareRenderContext &context,
                const Camera &cam,
                const int *order,
                float height_factor,
                bool color_mapping_enabled,
                float z_min, float z_max);

            ~Og3dRenderContext();

            inline PrepareRenderContext &get_render_context()
            {
                return m_render_context;
            }

            inline const Frustum &get_frustum() const { return m_frustum; }

            /**
             * \brief \c get_render_order(n) returns the index of the n:th
             * child to render.
             *
             * The render order is specified through the constructor. If the
             * occupancy grid is rendered non-opaque, the render order is
             * typically set as to produce an approximate back-to-front render
             * of the entire occupancy grid given the camera used to render it.
             */
            inline const int get_render_order(int i) const
            {
                return m_order[i];
            }

            inline const bool color_mapping_enabled() const
            {
                return m_color_mapping_enabled;
            }

            void get_color_from_z(
                float z,
                boost::uint8_t &r, boost::uint8_t &g, boost::uint8_t &b) const;

            inline float get_size_to_bsphere_radius_factor() const
            {
                return m_bsphere_factor;
            }

            void render_voxel(const Eigen::Vector3f &center, double node_size);

            void render_subtree_to_buffer(
                const Og3dNode *node,
                const Eigen::Vector3f &center, float node_size,
                boost::uint8_t * &buf) const;

        private:
            void z_to_color(
                float z,
                boost::uint8_t &r, boost::uint8_t &g, boost::uint8_t &b) const;

            void compute_color_tabs();

            void flush_voxels();

            void allocate_voxel_buffers();

            void render_voxel_to_buffer(
                boost::uint8_t * &buf,
                const Eigen::Vector3f &center, float node_size) const;

            static void render_voxel_face(
                boost::uint8_t * &buf,
                const Eigen::Vector3f *vertices,
                int i0, int i1, int i2, int i3,
                const Eigen::Vector3f &normal,
                float r, float g, float b);

        private:
            PrepareRenderContext &m_render_context;

            const Frustum m_frustum;

            const int *m_order;

            bool m_color_mapping_enabled;
            float m_z_min, m_z_max;

            boost::uint8_t m_color_tab_r[512];
            boost::uint8_t m_color_tab_g[512];
            boost::uint8_t m_color_tab_b[512];

            const float m_hf;

            // The factor to multiply the node size with to get its bounding sphere
            // radius
            const float m_bsphere_factor;

            std::size_t m_voxels_buffered;
            boost::shared_ptr<VertexBuffer> m_voxel_vb;
            boost::uint8_t *m_voxel_vp;
        };
    }
}


#endif // PEEKABOT_RENDERER_OG3D_RENDER_CONTEXT_HH_INCLUDED
