/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <GL/glew.h>
#include <boost/scoped_array.hpp>
#include <boost/bind.hpp>
#include <boost/ref.hpp>
#include <Eigen/Core>

#include "Polygon.hh"
#include "../State.hh"
#include "../../GeometryToolbox.hh"
#include "../Tessellator.hh"
#include "../PrepareRenderContext.hh"


using namespace peekabot;
using namespace peekabot::renderer;


Polygon::Polygon()
{
    // Cull front faces on the back-face mesh
    m_bf_mesh.get_state().set(new statelets::CullFrontFaces(true));
}


Polygon::Polygon(const Polygon &x)
    : VertexBased(x),
      m_ff_mesh(x.m_ff_mesh),
      m_bf_mesh(x.m_bf_mesh)
{
}


void Polygon::get_renderables(PrepareRenderContext &context) const
{
    context.prepare(&m_ff_mesh);
    context.prepare(&m_bf_mesh);
}

void Polygon::set_vertices(const Vertices &vertices) throw()
{
    // Reset the element count, we need this if there were previous vertices
    // but vertices is empty, otherwise the old data will be rendered
    m_ff_mesh.set_element_count(0);
    m_bf_mesh.set_element_count(0);

    // Reset color pointers
    m_ff_mesh.set_colors(boost::shared_ptr<VertexBuffer>());
    m_bf_mesh.set_colors(boost::shared_ptr<VertexBuffer>());

    if( vertices.empty() || vertices.size() < 3 )
        return;

    Indices indices;
    // Copy of vertices which will include eventual new vertices added
    // through the combine callback (e.g. if the polygon is self-intersecting)
    Vertices verts(vertices);

    Tessellator tess(
        boost::bind(
            &Polygon::emit_face_callback,
            boost::ref(indices), _1, _2, _3),
        boost::bind(
            &Polygon::combine_callback,
            boost::ref(verts), (VertexColors *)0,
            _1, _2, _3, _4, _5, _6));

    tess.begin_polygon();
    tess.begin_contour();
        for( std::size_t i = 0; i < vertices.size(); ++i )
        tess.vertex( vertices[i](0), vertices[i](1), vertices[i](2) );
    tess.end_polygon();

    // Calculate normals
    Vertices ff_normals, bf_normals;
    calculate_normals(indices, verts, ff_normals, bf_normals);

    boost::shared_ptr<IndexBuffer> ib(new IndexBuffer);
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> ff_nb(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> bf_nb(new VertexBuffer);

    ib->set_data(&indices[0], sizeof(boost::uint32_t)*indices.size());
    vb->set_data(&verts[0], 3*sizeof(float)*verts.size());
    ff_nb->set_data(&ff_normals[0], 3*sizeof(float)*ff_normals.size());
    bf_nb->set_data(&bf_normals[0], 3*sizeof(float)*bf_normals.size());

    m_ff_mesh.set_indices(ib);
    m_bf_mesh.set_indices(ib);
    m_ff_mesh.set_vertices(vb);
    m_bf_mesh.set_vertices(vb);
    m_ff_mesh.set_normals(ff_nb);
    m_bf_mesh.set_normals(bf_nb);

    m_ff_mesh.set_element_count(
        indices.size()/m_ff_mesh.vertices_per_element());
    m_bf_mesh.set_element_count(
        indices.size()/m_bf_mesh.vertices_per_element());

    calculate_bvs(vertices);
}

void Polygon::add_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices) throw()
{
    set_vertices(all_vertices);
}

void Polygon::get_vertices(Vertices &vertices) const throw()
{
    vertices.clear();

    if( !m_ff_mesh.has_vertices() )
        return;

    const boost::shared_ptr<VertexBuffer> verts = m_ff_mesh.get_vertices();

    VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> p(*verts);

    std::size_t N = m_ff_mesh.get_element_count();
    for( std::size_t i = 0; i < N; ++i )
        vertices.push_back(p[i]);
}


void Polygon::set_colored_vertices(
    const Vertices &vertices,
    const VertexColors &colors)
{
    // Reset the element count, we need this if there were previous vertices
    // but vertices is empty, otherwise the old data will be rendered
    m_ff_mesh.set_element_count(0);
    m_bf_mesh.set_element_count(0);

    if( vertices.empty() || vertices.size() < 3 )
        return;

    Indices indices;
    // Copy of vertices which will include eventual new vertices added
    // through the combine callback (e.g. if the polygon is self-intersecting)
    Vertices verts(vertices);
    // Copy of colors which will include eventual new colors added
    // through the combine callback (e.g. if the polygon is self-intersecting)
    VertexColors colors_(colors);

    Tessellator tess(
        boost::bind(
            &Polygon::emit_face_callback,
            boost::ref(indices), _1, _2, _3),
        boost::bind(
            &Polygon::combine_callback,
            boost::ref(verts), &colors_,
            _1, _2, _3, _4, _5, _6));

    tess.begin_polygon();
    tess.begin_contour();
        for( std::size_t i = 0; i < vertices.size(); ++i )
        tess.vertex( vertices[i](0), vertices[i](1), vertices[i](2) );
    tess.end_polygon();

    assert( 3*verts.size() == colors_.size() );

    // Calculate normals
    Vertices ff_normals, bf_normals;
    calculate_normals(indices, verts, ff_normals, bf_normals);

    boost::shared_ptr<IndexBuffer> ib(new IndexBuffer);
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> ff_nb(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> bf_nb(new VertexBuffer);
    boost::shared_ptr<VertexBuffer> cb(new VertexBuffer);

    ib->set_data(&indices[0], sizeof(boost::uint32_t)*indices.size());
    vb->set_data(&verts[0], 3*sizeof(float)*verts.size());
    ff_nb->set_data(&ff_normals[0], 3*sizeof(float)*ff_normals.size());
    bf_nb->set_data(&bf_normals[0], 3*sizeof(float)*bf_normals.size());
    cb->set_data(&colors_[0], colors_.size());

    m_ff_mesh.set_indices(ib);
    m_bf_mesh.set_indices(ib);
    m_ff_mesh.set_vertices(vb);
    m_bf_mesh.set_vertices(vb);
    m_ff_mesh.set_normals(ff_nb);
    m_bf_mesh.set_normals(bf_nb);
    m_ff_mesh.set_colors(cb);
    m_bf_mesh.set_colors(cb);

    m_ff_mesh.set_element_count(
        indices.size()/m_ff_mesh.vertices_per_element());
    m_bf_mesh.set_element_count(
        indices.size()/m_bf_mesh.vertices_per_element());

    calculate_bvs(vertices);
}


void Polygon::add_colored_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices,
    const VertexColors &new_colors,
    const VertexColors &all_colors)
{
    set_colored_vertices(all_vertices, all_colors);
}


void Polygon::calculate_bvs(const Vertices &vertices) throw()
{
    Eigen::Vector3f midpoint;
    float r;

    geom::calc_approx_bounding_sphere(
        vertices, midpoint, r,
        1, 1, 1);

    set_bounding_sphere(BoundingSphere(midpoint, r));
}


void Polygon::calculate_normals(
    const Indices &indices, const Vertices &vertices,
    std::vector<Eigen::Vector3f> &ff_normals,
    std::vector<Eigen::Vector3f> &bf_normals)
{
    ff_normals.clear();
    bf_normals.clear();
    assert( indices.size()%3 == 0 );

    // Initialize ff_normals to all zeros
    for( std::size_t i = 0; i < vertices.size(); ++i )
        ff_normals.push_back(Eigen::Vector3f(0,0,0));

    for( std::size_t i = 0; i < indices.size(); i += 3 )
    {
        Indices::value_type i1 = indices[i+0];
        Indices::value_type i2 = indices[i+1];
        Indices::value_type i3 = indices[i+2];

        if( i1 >= vertices.size() ||
            i2 >= vertices.size() ||
            i3 >= vertices.size() )
            continue;

        const Eigen::Vector3f &p1 = vertices[i1];
        const Eigen::Vector3f &p2 = vertices[i2];
        const Eigen::Vector3f &p3 = vertices[i3];

        Eigen::Vector3f face_normal = (p1-p3).cross(p2-p3);
        face_normal.normalize();

        // NOTE: We don't need to normalize the normals since we have
        // GL_NORMALIZE enabled to do this for us (needed for scaling).
        ff_normals[i1] += face_normal;
        ff_normals[i2] += face_normal;
        ff_normals[i3] += face_normal;
    }

    for( std::size_t i = 0; i < ff_normals.size(); ++i )
    {
        bf_normals.push_back(-ff_normals[i]);
    }

    assert( ff_normals.size() == vertices.size() );
    assert( bf_normals.size() == vertices.size() );
}


void Polygon::emit_face_callback(
    Indices &indices,
    std::size_t idx1, std::size_t idx2, std::size_t idx3)
{
    indices.push_back(idx1);
    indices.push_back(idx2);
    indices.push_back(idx3);
}


std::size_t Polygon::combine_callback(
    Vertices &vertices,
    VertexColors *colors,
    const Eigen::Vector3f &v,
    std::size_t idx1, std::size_t idx2,
    std::size_t idx3, std::size_t idx4,
    const float w[4])
{
    if( colors )
    {
        float r = std::max(
            0.0f, std::min(
                w[0]*(*colors)[3*idx1+0] + w[1]*(*colors)[3*idx2+0] +
                w[2]*(*colors)[3*idx3+0] + w[3]*(*colors)[3*idx4+0], 255.0f));

        float g = std::max(
            0.0f, std::min(
                w[0]*(*colors)[3*idx1+1] + w[1]*(*colors)[3*idx2+1] +
                w[2]*(*colors)[3*idx3+1] + w[3]*(*colors)[3*idx4+1], 255.0f));

        float b = std::max(
            0.0f, std::min(
                w[0]*(*colors)[3*idx1+2] + w[1]*(*colors)[3*idx2+2] +
                w[2]*(*colors)[3*idx3+2] + w[3]*(*colors)[3*idx4+2], 255.0f));

        colors->push_back(r);
        colors->push_back(g);
        colors->push_back(b);
    }

    vertices.push_back(v);
    return vertices.size()-1;
}
