/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Polyline.hh"
#include "../Mesh.hh"
#include "../PrepareRenderContext.hh"
#include "../../GeometryToolbox.hh"

#include <GL/glew.h>


using namespace peekabot;
using namespace peekabot::renderer;


Polyline::Polyline()
    : m_mesh(new LineStripMesh)
{
    /*boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    vb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    m_mesh->set_vertices(vb);*/
}


Polyline::Polyline(const Polyline &other)
    : m_mesh(other.m_mesh->clone())
{
}


Polyline::~Polyline()
{
}


Polyline *Polyline::clone() const
{
    return new Polyline(*this);
}


void Polyline::get_renderables(PrepareRenderContext &context) const
{
    if( m_mesh->has_vertices() )
        context.prepare(m_mesh.get());
}


void Polyline::set_vertices_helper(
    const VertexBased::Vertices &vertices)
{
    boost::shared_ptr<VertexBuffer> vb(new VertexBuffer);
    vb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    vb->set_data(&vertices[0], 3*sizeof(float)*vertices.size());
    m_mesh->set_vertices(vb, 0, 0);

    // Update element count
    if( vertex_count() < 2 )
        m_mesh->set_element_count(0);
    else
        m_mesh->set_element_count(vertex_count());

    // Calculate bounding sphere
    Eigen::Vector3f midpoint;
    float r;
    geom::calc_approx_bounding_sphere(
        vertices, midpoint, r,
        1, 1, 1);

    set_bounding_sphere(BoundingSphere(midpoint, r));
}


void Polyline::add_vertices_helper(
    const VertexBased::Vertices &new_vertices,
    const VertexBased::Vertices &all_vertices)
{
    // Update the vertex buffer
    boost::shared_ptr<VertexBuffer> vb = m_mesh->get_vertices();
    std::size_t old_size = vb->size();
    std::size_t n = 3*sizeof(float)*new_vertices.size();
    vb->grow_back(n);
    vb->set_sub_data(&new_vertices[0], n, old_size);

    // Update element count
    if( vertex_count() < 2 )
        m_mesh->set_element_count(0);
    else
        m_mesh->set_element_count(vertex_count());

    // Update the bounding sphere
    Eigen::Vector3f midp = get_bounding_sphere_lc().get_pos();
    float r = get_bounding_sphere_lc().get_radius();
    if( geom::extend_bounding_sphere(new_vertices, midp, r, 1.5) )
    {
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> verts(*vb);
        geom::calc_approx_bounding_sphere(
            verts.get(), verts.get()+vertex_count(),
            midp, r,
            1, 1, 1);
    }

    set_bounding_sphere(BoundingSphere(midp, r));
}


void Polyline::set_vertices(const VertexBased::Vertices &vertices)
{
    // Clear color pointers
    m_mesh->set_colors(boost::shared_ptr<VertexBuffer>());

    set_vertices_helper(vertices);
}


void Polyline::add_vertices(
    const VertexBased::Vertices &new_vertices,
    const VertexBased::Vertices &all_vertices)
{
    assert( !m_mesh->has_colors() );

    if( new_vertices.empty() )
        return;

    if( !m_mesh->has_vertices() )
    {
        set_vertices(new_vertices);
        return;
    }

    add_vertices_helper(new_vertices, all_vertices);
}


void Polyline::get_vertices(VertexBased::Vertices &vertices) const
{
    vertices.clear();

    if( m_mesh->has_vertices() )
    {
        VertexBuffer::ReadOnlyMapping<Eigen::Vector3f> verts(
            *m_mesh->get_vertices());

        for( std::size_t i = 0; i < vertex_count(); ++i )
            vertices.push_back(verts[i]);
    }
}


void Polyline::set_colored_vertices(
    const Vertices &vertices,
    const VertexColors &colors)
{
    assert( 3*vertices.size() == colors.size() );

    // Set color pointers
    boost::shared_ptr<VertexBuffer> cb(new VertexBuffer);
    cb->set_usage_hint(VertexBuffer::DYNAMIC_DRAW);
    cb->set_data(&colors[0], colors.size());
    m_mesh->set_colors(cb);

    set_vertices_helper(vertices);
}


void Polyline::add_colored_vertices(
    const Vertices &new_vertices,
    const Vertices &all_vertices,
    const VertexColors &new_colors,
    const VertexColors &all_colors)
{
    assert( 3*new_vertices.size() == new_colors.size() );
    assert( 3*all_vertices.size() == all_colors.size() );

    if( new_vertices.empty() )
        return;

    if( !m_mesh->has_vertices() )
    {
        assert( new_vertices.size() == all_vertices.size() );
        assert( new_colors.size() == all_colors.size() );
        set_colored_vertices(new_vertices, new_colors);
        return;
    }

    // Update the color buffer
    assert( m_mesh->has_colors() );
    boost::shared_ptr<VertexBuffer> cb = m_mesh->get_colors();
    assert( cb );
    std::size_t old_size = cb->size();
    std::size_t n = new_colors.size();
    cb->grow_back(n);
    cb->set_sub_data(&new_colors[0], n, old_size);

    add_vertices_helper(new_vertices, all_vertices);
}


std::size_t Polyline::vertex_count() const
{
    if( m_mesh->has_vertices() )
        return m_mesh->get_vertices()->size()/3/sizeof(float);
    else
        return 0;
}


void Polyline::set_line_style(
    float line_width,
    boost::uint16_t stipple_pattern,
    int stipple_factor)
{
    m_mesh->get_state().set(
        new statelets::LineParams(
            line_width, stipple_pattern != 0xFFFF,
            stipple_factor, stipple_pattern));
}
