/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_SERIALIZATION_CONTAINERS_HH_INCLUDED
#define PEEKABOT_SERIALIZATION_CONTAINERS_HH_INCLUDED


#include <map>
#include <set>
#include <vector>
#include <list>
#include <boost/cstdint.hpp>

#include "SerializationInterface.hh"
#include "DeserializationInterface.hh"


namespace peekabot
{
    // ------------- std::vector -----------

    template<typename T> inline
    SerializationInterface &operator<<(
        SerializationInterface &ar, const std::vector<T> &x)
    {
        ar << (boost::uint32_t)x.size();
        // Note: the storage in std::vector is guaranteed to be contiguous, so
        // this is safe
        ar.save_array(&x[0], x.size());
        return ar;
    }

    template<typename T> inline
    DeserializationInterface &operator>>(
        DeserializationInterface &ar, std::vector<T> &x)
    {
        boost::uint32_t len;
        ar >> len;
        x.resize(len);
        // Note: the storage in std::vector is guaranteed to be contiguous, so
        // this is safe
        ar.load_array(&x[0], len);
        return ar;
    }

    // ----------- std::list ----------

    template<typename T> inline
    SerializationInterface &operator<<(
        SerializationInterface &ar, const std::list<T> &x)
    {
        ar << (boost::uint32_t)x.size();
        for( typename std::list<T>::const_iterator it = x.begin();
             it != x.end(); ++it )
            ar << *it;
        return ar;
    }

    template<typename T> inline
    DeserializationInterface &operator>>(
        DeserializationInterface &ar, std::list<T> &x)
    {
        boost::uint32_t len;
        ar >> len;
        x.clear();
        for( boost::uint32_t i = 0; i < len; ++i )
        {
            T tmp;
            ar >> tmp;
            x.push_back(tmp);
        }
        return ar;
    }

    // -------------- std::set --------------

    template<typename T> inline
    SerializationInterface &operator<<(
        SerializationInterface &ar, const std::set<T> &x)
    {
        ar << (boost::uint32_t)x.size();
        for( typename std::set<T>::const_iterator it = x.begin();
             it != x.end(); ++it )
            ar << *it;
        return ar;
    }

    template<typename T> inline
    DeserializationInterface &operator>>(
        DeserializationInterface &ar, std::set<T> &x)
    {
        boost::uint32_t len;
        ar >> len;
        x.clear();
        for( boost::uint32_t i = 0; i < len; ++i )
        {
            T tmp;
            ar >> tmp;
            x.insert(tmp);
        }
        return ar;
    }

    // ------------- std::map -------------

    template<typename K, typename V> inline
    SerializationInterface &operator<<(
        SerializationInterface &ar, const std::map<K, V> &x)
    {
        ar << (boost::uint32_t)x.size();
        for( typename std::map<K, V>::const_iterator it = x.begin();
             it != x.end(); ++it )
            ar << it->first << it->second;
        return ar;
    }

    template<typename K, typename V> inline
    DeserializationInterface &operator>>(
        DeserializationInterface &ar, std::map<K, V> &x)
    {
        boost::uint32_t len;
        ar >> len;
        x.clear();
        for( boost::uint32_t i = 0; i < len; ++i )
        {
            K key; V val;
            ar >> key >> val;
            x.insert(std::make_pair(key, val));
        }
        return ar;
    }
}


#endif // PEEKABOT_SERIALIZATION_CONTAINERS_HH_INCLUDED
