/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_MEM_ADAPTERS_HH_INCLUDED
#define PEEKABOT_MEM_ADAPTERS_HH_INCLUDED


#include <cstring>
#include <cassert>
#include <stdexcept>
#include <algorithm>
#include <boost/cstdint.hpp>
#include <boost/scoped_array.hpp>

#include "DeserializationBuffer.hh"
#include "SerializationBuffer.hh"


namespace peekabot
{
    class MemDeserializationBuffer : public serialization::DeserializationBuffer
    {
    public:
        MemDeserializationBuffer(const void *buf, std::size_t size)
            : m_cur((const boost::uint8_t *)buf), m_end(m_cur+size)
        {
        }

        MemDeserializationBuffer(const MemDeserializationBuffer &other)
            : m_cur(other.m_cur), m_end(other.m_end)
        {
        }

        virtual void read(void *x, std::size_t n)
        {
            std::size_t m = std::min<std::size_t>(n, m_end-m_cur);
            std::memcpy(x, m_cur, m);
            m_cur += m;

            if( m < n )
                throw std::runtime_error("Memory buffer exhausted");
        }

    private:
        const boost::uint8_t *m_cur, *m_end;
    };


    class MemSerializationBuffer : public serialization::SerializationBuffer
    {
    public:
        MemSerializationBuffer()
            : m_size(0),
              m_capacity(0)
        {
        }

        MemSerializationBuffer(const MemSerializationBuffer &other)
            : m_size(other.m_size),
              m_capacity(other.m_capacity)
        {
            assert( m_size <= m_capacity );
            if( m_size > 0 )
            {
                m_buf.reset(new boost::uint8_t[m_capacity]);
                std::memcpy(m_buf.get(), other.m_buf.get(), m_size);
            }
        }

        virtual void write(const void *x, std::size_t n)
        {
            if( n < 1 )
                return;

            if( m_size+n > m_capacity )
            {
                std::size_t new_capacity =
                    std::max<std::size_t>(m_capacity, 1024);
                while( m_size+n > new_capacity )
                    new_capacity *= 2;

                boost::scoped_array<boost::uint8_t> old;
                if( m_size > 0 )
                {
                    old.reset(new boost::uint8_t[m_size]);
                    std::memcpy(old.get(), m_buf.get(), m_size);
                }

                m_capacity = new_capacity;
                m_buf.reset(new boost::uint8_t[new_capacity]);

                std::memcpy(m_buf.get(), old.get(), m_size);
            }

            std::memcpy(m_buf.get()+m_size, x, n);
            m_size += n;
        }

        inline void *get()
        {
            return m_buf.get();
        }

        inline const void *get() const
        {
            return m_buf.get();
        }

        inline std::size_t size() const
        {
            return m_size;
        }

    private:
        std::size_t m_size;

        std::size_t m_capacity;

        boost::scoped_array<boost::uint8_t> m_buf;
    };
}


#endif // PEEKABOT_MEM_ADAPTERS_HH_INCLUDED
