/*
 * Copyright Staffan Gimåker 2008-2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_SERIALIZATION_SERIALIZABLE_FACTORY_HH_INCLUDED
#define PEEKABOT_SERIALIZATION_SERIALIZABLE_FACTORY_HH_INCLUDED


#include "../Singleton.hh"
#include "Access.hh"

#include <typeinfo>
#include <cassert>
#include <stdexcept>
#include <boost/cstdint.hpp>
#include <boost/foreach.hpp>
#include <boost/type_traits.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/unordered_map.hpp>


namespace peekabot
{
    namespace serialization
    {
        // Forward declarations
        class SerializationInterface;
        class DeserializationInterface;
        typedef boost::uint16_t IdType;
        struct SerializableInfoBase;
        template<typename T> struct SerializableInfo;
        struct TypeNotRegistered;
        class SerializableFactory;
    }


    struct serialization::SerializableInfoBase
    {
        IdType m_id;

        virtual void *create() const = 0;

        virtual void save(SerializationInterface &, const void *) const = 0;

        virtual void load(DeserializationInterface &, void *, int) const = 0;

        virtual int version() const = 0;
    };


    template<typename T>
    struct serialization::SerializableInfo : public SerializableInfoBase
    {
        virtual void *create() const
        {
            return new T;
        }

        virtual void save(SerializationInterface &ar, const void *x) const
        {
            reinterpret_cast<const T *>(x)->save(ar);
        }

        virtual void load(
            DeserializationInterface &ar, void *x, int version) const
        {
            dispatch_load(ar, reinterpret_cast<T *>(x), version);
        }

        virtual int version() const
        {
            return typename detail::VersionTag<T>::value();
        }


        // Meta function for checking for whether load() takes a version
        // argument or not
        typedef char (&no_tag)[1];
        typedef char (&yes_tag)[2];

        template<typename U, void (U::*)(
            DeserializationInterface &, int)>
        struct ptmf_versioned_load_helper {};

        template<typename U> static no_tag has_versioned_load_helper(...);

        template<typename U> static yes_tag
        has_versioned_load_helper(
            ptmf_versioned_load_helper<U, &U::load>* p);

        template<typename U> struct has_versioned_load
        {
            BOOST_STATIC_CONSTANT(
                bool, value = sizeof(has_versioned_load_helper<U>(0)) ==
                sizeof(yes_tag));

            typedef boost::mpl::bool_<value> type;
        };


        // Helper function to call either the versioned or the unversioned load
        // method
        template<typename U> inline typename boost::enable_if_c<
            has_versioned_load<U>::value, void>::type
        dispatch_load(
            DeserializationInterface &ar, U *x, int version) const
        {
            x->load(ar, version);
        }

        template<typename U> inline typename boost::enable_if_c<
            boost::mpl::not_<has_versioned_load<U> >::value, void>::type
        dispatch_load(
            DeserializationInterface &ar, U *x, int version) const
        {
            x->load(ar);
        }
    };


    struct serialization::TypeNotRegistered : public std::runtime_error
    {
        TypeNotRegistered(const std::string &msg)
            : std::runtime_error(msg) {}
    };


    class serialization::SerializableFactory
    {
        friend class SerializationInterface;
        friend class DeserializationInterface;

    public:
        template<class T>
        void register_type(IdType id) throw(std::runtime_error)
        {
            if( is_registered<T>() )
                throw std::runtime_error("Type already registered");
            else if( is_registered(id) )
                throw std::runtime_error("Id already registered");

            SerializableInfo<T> *info = new SerializableInfo<T>;
            info->m_id = id;
            m_type_to_info.insert(std::make_pair(typeid(T).name(), info));
            m_id_to_info.insert(std::make_pair(id, info));
        }

        void deregister_all() throw()
        {
            BOOST_FOREACH( IdToInfo::value_type &x, m_id_to_info )
            {
                delete x.second;
            }

            m_type_to_info.clear();
            m_id_to_info.clear();
        }

        template<typename T> inline bool is_registered() const
        {
            return m_type_to_info.find(typeid(T).name()) != m_type_to_info.end();
        }

        inline bool is_registered(IdType id) const
        {
            return m_id_to_info.find(id) != m_id_to_info.end();
        }

        template<typename PolymorphicBase> inline
        typename boost::enable_if_c<
            boost::is_polymorphic<PolymorphicBase>::value,
            const SerializableInfoBase &>::type
        get_serializable_info(const PolymorphicBase *x) const
        {
            TypeToInfo::const_iterator it =
                m_type_to_info.find(typeid(*x).name());
            if( it == m_type_to_info.end() )
                throw TypeNotRegistered("Type not registered");
            return *(it->second);
        }

        inline const SerializableInfoBase &get_serializable_info(int id) const
        {
            IdToInfo::const_iterator it = m_id_to_info.find(id);
            if( it == m_id_to_info.end() )
                throw TypeNotRegistered("Type not registered");
            return *(it->second);
        }

    private:
        typedef boost::unordered_map<
            const char *, SerializableInfoBase *> TypeToInfo;
        TypeToInfo m_type_to_info;

        typedef boost::unordered_map<IdType, SerializableInfoBase *> IdToInfo;
        IdToInfo m_id_to_info;
    };

    typedef Singleton<serialization::SerializableFactory> TheSerializableFactory;
}


#endif // PEEKABOT_SERIALIZATION_SERIALIZABLE_FACTORY_HH_INCLUDED
