/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <fstream>
#include <boost/test/unit_test.hpp>

#include "Fixture.hh"


using namespace peekabot;


BOOST_AUTO_TEST_SUITE( Serialization_BinaryIO );


BOOST_FIXTURE_TEST_CASE( load_binary, Fixture )
{
    int x = 0xdeadbeef;

    // Check that deserializing from an empty buffer throws
    BOOST_CHECK_THROW( m_deser.load_binary(&x, sizeof(int)), std::exception );
    BOOST_CHECK_THROW( 
        m_deser_foreign.load_binary(&x, sizeof(int)), std::exception );
   
    // Check that deserializing from sufficiently filled buffer does not throw
    // and that loading data from the buffer decreases the size of the data
    // store accordingly
    m_buf.write(&x, sizeof(int));
    m_buf.write(&x, sizeof(int));
    BOOST_CHECK_NO_THROW( m_deser.load_binary(&x, sizeof(int)) );
    BOOST_REQUIRE_EQUAL( m_buf.get_size(), sizeof(int) );
    BOOST_CHECK_NO_THROW( m_deser.load_binary(&x, sizeof(int)) );
    BOOST_REQUIRE_EQUAL( m_buf.get_size(), 0u );

    // Check that loading a value returns the expected value, independent of
    // the buffer's byte order setting
    m_buf.clear();
    uint8_t value1 = 77, expected1 = 0;
    uint32_t value2 = 88188233, expected2 = 0;
    double value3 = 8.111, expected3 = 0;
    m_buf.write(&value1, sizeof(value1));
    m_buf.write(&value2, sizeof(value2));
    m_buf.write(&value3, sizeof(value3));
    BOOST_CHECK_NO_THROW( m_deser.load_binary(&expected1, sizeof(expected1)) );
    BOOST_CHECK_EQUAL( value1, expected1 );
    BOOST_CHECK_NO_THROW( m_deser.load_binary(&expected2, sizeof(expected2)) );
    BOOST_CHECK_EQUAL( value2, expected2 );
    BOOST_CHECK_NO_THROW( m_deser.load_binary(&expected3, sizeof(expected3)) );
    BOOST_CHECK_EQUAL( value3, expected3 );
    BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    expected1 = 0; expected2 = 0; expected3 = 0;
    m_buf.write(&value1, sizeof(value1));
    m_buf.write(&value2, sizeof(value2));
    m_buf.write(&value3, sizeof(value3));
    BOOST_CHECK_NO_THROW( 
        m_deser_foreign.load_binary(&expected1, sizeof(expected1)) );
    BOOST_CHECK_EQUAL( value1, expected1 );
    BOOST_CHECK_NO_THROW( 
        m_deser_foreign.load_binary(&expected2, sizeof(expected2)) );
    BOOST_CHECK_EQUAL( value2, expected2 );
    BOOST_CHECK_NO_THROW( 
        m_deser_foreign.load_binary(&expected3, sizeof(expected3)) );
    BOOST_CHECK_EQUAL( value3, expected3 );
    BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
}

BOOST_FIXTURE_TEST_CASE( save_binary, Fixture )
{
    int x = 0xdeafd00d; char y = 56;

    // Check that writing to a "full" (eof) data store throws
    {
        std::filebuf fb;
        SerializationInterface sb(fb);
        BOOST_CHECK_THROW( sb.save_binary(&x, sizeof(int)), std::exception );
    }

    // Test that save_binary increases the size of the stored data accordingly, 
    // and that it actually saves the right data
    BOOST_CHECK_NO_THROW( m_ser.save_binary(&x, sizeof(int)) );
    BOOST_CHECK_EQUAL( m_buf.get_size(), sizeof(int) );
    BOOST_CHECK_NO_THROW( m_ser.save_binary(&x, sizeof(int)) );
    BOOST_CHECK_EQUAL( m_buf.get_size(), 2*sizeof(int) );
    BOOST_CHECK_NO_THROW( m_ser.save_binary(&y, sizeof(char)) );
    BOOST_CHECK_EQUAL( m_buf.get_size(), 2*sizeof(int)+sizeof(char) );
    int r1 = 0, r2 = 0; char r3 = 0;
    BOOST_REQUIRE_NO_THROW( m_buf.read(&r1, sizeof(int)) );
    BOOST_REQUIRE_NO_THROW( m_buf.read(&r2, sizeof(int)) );
    BOOST_REQUIRE_NO_THROW( m_buf.read(&r3, sizeof(char)) );
    BOOST_CHECK_EQUAL( x, r1 );
    BOOST_CHECK_EQUAL( x, r2 );
    BOOST_CHECK_EQUAL( y, r3 );
    BOOST_REQUIRE( m_buf.is_empty() );
}


BOOST_AUTO_TEST_SUITE_END();
