/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <boost/test/unit_test.hpp>

#include "../Containers.hh"
#include "Fixture.hh"


using namespace peekabot;



BOOST_AUTO_TEST_SUITE( Serialization_Containers );


//
// Vector
//

BOOST_FIXTURE_TEST_CASE( save_load_vector, Fixture )
{
    const size_t n = 5;
    std::vector<float> data(n);
    
    data[0] = 1.5556;
    data[1] = -9.1e-7;
    data[2] = 3.14159;
    data[3] = 9992.3;
    data[4] = 183838;

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( m_buf.get_size(), n*sizeof(float)+sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::vector<float> read_back;

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_REQUIRE_EQUAL( read_back.size(), data.size() );

        for( size_t i = 0; i < n; ++i )
            BOOST_CHECK_EQUAL( read_back[i], data[i] );

        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

BOOST_FIXTURE_TEST_CASE( save_load_empty_vector, Fixture )
{
    std::vector<float> data;
    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( m_buf.get_size(), sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::vector<float> read_back(10);

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_CHECK_EQUAL( read_back.size(), 0u );
        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

BOOST_FIXTURE_TEST_CASE( save_load_vector_array, Fixture )
{
    const size_t n = 2, m = 2;
    std::vector<short> data[n] = {
        std::vector<short>(m),
        std::vector<short>(m)
    };

    data[0][0] = 74;
    data[0][1] = -555;
    data[1][0] = 923;
    data[1][1] = 0;

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );

        BOOST_CHECK_EQUAL( 
            m_buf.get_size(), n*m*sizeof(short)+n*sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::vector<short> read_back[n];

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );

        for( size_t i = 0; i < n; ++i )
            for( size_t j = 0; j < m; ++j )
                BOOST_CHECK_EQUAL( read_back[i][j], data[i][j] );

        BOOST_CHECK( m_buf.is_empty() );
    }
}

//
// List
//

BOOST_FIXTURE_TEST_CASE( save_load_list, Fixture )
{
    std::list<float> data;
    data.push_back(1.5556f);
    data.push_back(-3.14159f);
    data.push_back(1.5501e16f);
    data.push_back(0.000001e-15f);
    data.push_back(12.0f);

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( 
            m_buf.get_size(), data.size()*sizeof(float)+sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::list<float> read_back;

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );

        BOOST_CHECK_EQUAL( data.size(), read_back.size() );
        BOOST_REQUIRE( read_back.size() <= data.size() );

        std::list<float>::const_iterator read_back_it = read_back.begin();
        std::list<float>::const_iterator expected_it = data.begin();
        for( ; read_back_it != read_back.end(); ++read_back_it, ++expected_it )
            BOOST_CHECK_EQUAL( *read_back_it, *expected_it );

        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

BOOST_FIXTURE_TEST_CASE( save_load_empty_list, Fixture )
{
    std::list<float> data;

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( m_buf.get_size(), sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::list<float> read_back;
        read_back.push_back(7.5f);

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_CHECK_EQUAL( read_back.size(), 0u );
        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

//
// Set
//

BOOST_FIXTURE_TEST_CASE( save_load_set, Fixture )
{
    std::set<int> data;
    data.insert( 42 );
    data.insert( 0 );
    data.insert( -42 );
    data.insert( 719143 );

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( 
            m_buf.get_size(), data.size()*sizeof(int)+sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::set<int> read_back;

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );

        BOOST_CHECK_EQUAL( data.size(), read_back.size() );
        BOOST_REQUIRE( read_back.size() <= data.size() );

        std::set<int>::const_iterator read_back_it = read_back.begin();
        std::set<int>::const_iterator expected_it = data.begin();
        for( ; read_back_it != read_back.end(); ++read_back_it, ++expected_it )
            BOOST_CHECK_EQUAL( *read_back_it, *expected_it );

        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

BOOST_FIXTURE_TEST_CASE( save_load_empty_set, Fixture )
{
    std::set<float> data;

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( m_buf.get_size(), sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::set<float> read_back;
        read_back.insert(7.5f);

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_CHECK_EQUAL( read_back.size(), 0u );
        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

//
// Map
//

BOOST_FIXTURE_TEST_CASE( save_load_map, Fixture )
{
    std::map<int, double> data;
    data.insert( std::make_pair(8, 9.2) );
    data.insert( std::make_pair(18, 0.0011) );
    data.insert( std::make_pair(-28, 9.21123) );
    data.insert( std::make_pair(0, 239.2e-5) );

    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( 
            m_buf.get_size(), 
            data.size()*(sizeof(int)+sizeof(double))+sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::map<int, double> read_back;

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );

        BOOST_CHECK_EQUAL( data.size(), read_back.size() );
        BOOST_REQUIRE( read_back.size() <= data.size() );

        std::map<int, double>::const_iterator read_back_it = read_back.begin();
        std::map<int, double>::const_iterator expected_it = data.begin();
        for( ; read_back_it != read_back.end(); ++read_back_it, ++expected_it )
        {
            BOOST_CHECK_EQUAL( read_back_it->first, expected_it->first );
            BOOST_CHECK_EQUAL( read_back_it->second, expected_it->second );
        }

        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}

BOOST_FIXTURE_TEST_CASE( save_load_empty_map, Fixture )
{
    std::map<int, double> data;
    {
        SerializationInterface *ret;

        BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
        // Check returned serialization interface, should be the same as
        // the one we passed along (or m_ser << foo << bar; won't work)
        BOOST_CHECK_EQUAL( ret, &m_ser );
        // Check to see that the buffer has grown accordingly
        // after we've saved a value in it
        BOOST_CHECK_EQUAL( m_buf.get_size(), sizeof(uint32_t) );
    }
    {
        DeserializationInterface *ret;
        std::map<int, double> read_back;
        read_back.insert(std::make_pair(2, 3.0));

        BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back) );
        BOOST_CHECK_EQUAL( ret, &m_deser );
        BOOST_CHECK_EQUAL( read_back.size(), 0u );
        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );
    }
}





BOOST_AUTO_TEST_SUITE_END();
