/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <limits>
#include <boost/test/unit_test.hpp>

#include "Fixture.hh"
#include "../../Types.hh"


using namespace peekabot;


namespace
{
    template<class T, size_t S>
    struct PodFixture : public Fixture
    {
        void _test_save(const T data)
        {
            SerializationInterface *ret;
            size_t buf_size = m_buf.get_size();

            BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
            // Check returned serialization interface, should be the same as
            // the one we passed along (or m_ser << foo << bar; won't work)
            BOOST_CHECK_EQUAL( ret, &m_ser );
            // Check to see that the buffer has grown accordingly
            // after we've saved a value in it
            BOOST_CHECK_EQUAL( m_buf.get_size()-buf_size, S );
            T tmp = (T)0;
            BOOST_REQUIRE_NO_THROW( m_buf.read(&tmp, S) );
            BOOST_CHECK_EQUAL( tmp, data );
            BOOST_REQUIRE_EQUAL( m_buf.get_size(), buf_size );
        }

        template<size_t n>
        void _test_save_array(const T (&data)[n])
        {
            SerializationInterface *ret;
            size_t buf_size = m_buf.get_size();

            BOOST_CHECK_NO_THROW( ret = &(m_ser << data) );
            // Check returned serialization interface, should be the same as
            // the one we passed along (or m_ser << foo << bar; won't work)
            BOOST_CHECK_EQUAL( ret, &m_ser );
            // Check to see that the buffer has grown accordingly
            // after we've saved a value in it
            BOOST_CHECK_EQUAL( m_buf.get_size()-buf_size, n*S );

            for( size_t i = 0; i < n; ++i )
            {
                T tmp = (T)0;
                BOOST_REQUIRE_NO_THROW( m_buf.read(&tmp, S) );
                BOOST_CHECK_EQUAL( tmp, data[i] );
                BOOST_REQUIRE_EQUAL( 
                    m_buf.get_size() - (n-i-1)*S, buf_size );
            }
        }

        void _test_load(const T data, bool foreign)
        {
            DeserializationInterface *ret;
            size_t buf_size = m_buf.get_size();
            T val = (T)0;

            // Prepare buffer, write the test data to it
            T swapped = data;
            if( foreign )
                switch_byte_order(&swapped, 1);
            m_buf.write(&swapped, S);
            BOOST_REQUIRE_EQUAL( buf_size, m_buf.get_size()-S );

            if( !foreign )
            {
                BOOST_CHECK_NO_THROW( ret = &(m_deser >> val) );
                // Check returned serialization interface, should be the same as
                // the one we passed along (or m_ser >> foo >> bar; won't work)
                BOOST_CHECK_EQUAL( ret, &m_deser );
                // and check to see that the buffer has shrunk accordingly
                // after we've loaded from it
            }
            else
            {
                BOOST_CHECK_NO_THROW( ret = &(m_deser_foreign >> val) );
                BOOST_CHECK_EQUAL( ret, &m_deser_foreign );
            }

            BOOST_CHECK_EQUAL( m_buf.get_size(), buf_size );
            BOOST_CHECK_EQUAL( val, data );
        }

        template<size_t n>
        void _test_load_array(const T (&data)[n], bool foreign)
        {
            DeserializationInterface *ret;
            size_t buf_size = m_buf.get_size();

            // Prepare buffer, write the test data to it
            if( foreign )
            {
                T swapped[n];
                for( size_t i = 0; i < n; ++i )
                    swapped[i] = data[i];
                switch_byte_order(swapped, n);
                m_buf.write(swapped, n*S);
            }
            else
                m_buf.write(data, n*S);
            BOOST_REQUIRE_EQUAL( buf_size, m_buf.get_size()-n*S );

            T val[n];

            if( !foreign )
            {
                BOOST_CHECK_NO_THROW( ret = &(m_deser >> val) );
                // Check returned serialization interface, should be the same as
                // the one we passed along (or m_ser >> foo >> bar; won't work)
                BOOST_CHECK_EQUAL( ret, &m_deser );
                // and check to see that the buffer has shrunk accordingly
                // after we've loaded from it
            }
            else
            {
                BOOST_CHECK_NO_THROW( ret = &(m_deser_foreign >> val) );
                BOOST_CHECK_EQUAL( ret, &m_deser_foreign );
            }

            BOOST_CHECK_EQUAL( m_buf.get_size(), buf_size );

            for( size_t i = 0; i < n; ++i )
                BOOST_CHECK_EQUAL( val[i], data[i] );
        }
    };


    template<class T, size_t S = sizeof(T)>
    struct SignedIntFixture : public PodFixture<T, S>
    {
        void test_save()
        {
            PodFixture<T, S>::_test_save(0);
            PodFixture<T, S>::_test_save(1);
            PodFixture<T, S>::_test_save(-1);
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::min());
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::max());
            PodFixture<T, S>::_test_save(-std::numeric_limits<T>::max());
        }

        void test_load(bool foreign)
        {
            PodFixture<T, S>::_test_load(0, foreign);
            PodFixture<T, S>::_test_load(1, foreign);
            PodFixture<T, S>::_test_load(-1, foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::min(), foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::max(), foreign);
            PodFixture<T, S>::_test_load(-std::numeric_limits<T>::max(), foreign);
        }

        void test_save_array()
        {
            T array[] = { 
                0, 1, -1,
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max(),
                -std::numeric_limits<T>::max()
            };
            PodFixture<T, S>::_test_save_array(array);
        }

        void test_load_array(bool foreign)
        {
            T array[] = { 
                0, 1, -1,
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max(),
                -std::numeric_limits<T>::max()
            };
            PodFixture<T, S>::_test_load_array(array, foreign);
        }
    };


    template<class T, size_t S = sizeof(T)>
    struct UnsignedIntFixture : public PodFixture<T, S>
    {
        void test_save()
        {
            PodFixture<T, S>::_test_save(0);
            PodFixture<T, S>::_test_save(1);
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::min());
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::max());
        }

        void test_load(bool foreign)
        {
            PodFixture<T, S>::_test_load(0, foreign);
            PodFixture<T, S>::_test_load(1, foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::min(), foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::max(), foreign);
        }

        void test_save_array()
        {
            T array[] = { 
                0, 1, 
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max()
            };
            PodFixture<T, S>::_test_save_array(array);
        }

        void test_load_array(bool foreign)
        {
            T array[] = { 
                0, 1,
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max()
            };
            PodFixture<T, S>::_test_load_array(array, foreign);
        }
    };

    template<class T, size_t S = sizeof(T)>
    struct FloatFixture : public PodFixture<T, S>
    {
        void test_save()
        {
            PodFixture<T, S>::_test_save(0);
            PodFixture<T, S>::_test_save(1);
            PodFixture<T, S>::_test_save(-1);
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::min());
            PodFixture<T, S>::_test_save(std::numeric_limits<T>::max());
            PodFixture<T, S>::_test_save(-std::numeric_limits<T>::max());

            PodFixture<T, S>::_test_save(std::numeric_limits<T>::infinity());
            //PodFixture<T, S>::_test_save(std::numeric_limits<T>::quiet_NaN());
        }

        void test_load(bool foreign)
        {
            PodFixture<T, S>::_test_load(0, foreign);
            PodFixture<T, S>::_test_load(1, foreign);
            PodFixture<T, S>::_test_load(-1, foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::min(), foreign);
            PodFixture<T, S>::_test_load(std::numeric_limits<T>::max(), foreign);
            PodFixture<T, S>::_test_load(-std::numeric_limits<T>::max(), foreign);

            PodFixture<T, S>::_test_load(std::numeric_limits<T>::infinity(), foreign);
            //PodFixture<T, S>::_test_load(std::numeric_limits<T>::quiet_NaN(), foreign);
        }

        void test_save_array()
        {
            T array[] = { 
                0, 1, -1,
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max(),
                -std::numeric_limits<T>::max(),
                std::numeric_limits<T>::infinity()
                //std::numeric_limits<T>::quiet_NaN());
            };
            PodFixture<T, S>::_test_save_array(array);
        }

        void test_load_array(bool foreign)
        {
            T array[] = { 
                0, 1, -1,
                std::numeric_limits<T>::min(), 
                std::numeric_limits<T>::max(),
                -std::numeric_limits<T>::max(),
                std::numeric_limits<T>::infinity()
                //std::numeric_limits<T>::quiet_NaN());
            };
            PodFixture<T, S>::_test_load_array(array, foreign);
        }
    };
}





BOOST_AUTO_TEST_SUITE( Serialization_FundamentalTypes );


#define DEF_TEST_CASES( type, fixture )                             \
    BOOST_FIXTURE_TEST_CASE( save_##type, fixture )                 \
    {                                                               \
        test_save();                                                \
    }                                                               \
    BOOST_FIXTURE_TEST_CASE( save_##type##_array, fixture )         \
    {                                                               \
        test_save_array();                                          \
    }                                                               \
    BOOST_FIXTURE_TEST_CASE( load_##type##_native, fixture )        \
    {                                                               \
        test_load(false);                                           \
    }                                                               \
    BOOST_FIXTURE_TEST_CASE( load_##type##_foreign, fixture )       \
    {                                                               \
        test_load(true);                                            \
    }                                                               \
    BOOST_FIXTURE_TEST_CASE( load_##type##_array_native, fixture )  \
    {                                                               \
        test_load_array(false);                                     \
    }                                                               \
    BOOST_FIXTURE_TEST_CASE( load_##type##_array_foreign, fixture ) \
    {                                                               \
        test_load_array(true);                                      \
    }


//
// Int types
//

DEF_TEST_CASES( int8_t, SignedIntFixture<int8_t> )
DEF_TEST_CASES( uint8_t, UnsignedIntFixture<uint8_t> )

DEF_TEST_CASES( int16_t, SignedIntFixture<int16_t> )
DEF_TEST_CASES( uint16_t, UnsignedIntFixture<uint16_t> )

DEF_TEST_CASES( int32_t, SignedIntFixture<int32_t> )
DEF_TEST_CASES( uint32_t, UnsignedIntFixture<uint32_t> )

DEF_TEST_CASES( int64_t, SignedIntFixture<int64_t> )
DEF_TEST_CASES( uint64_t, UnsignedIntFixture<uint64_t> )


//
// Float types
//

DEF_TEST_CASES( float, FloatFixture<float> )

DEF_TEST_CASES( double, FloatFixture<double> )

//
// Enums
//

namespace
{
    enum TestEnum
    {
        ENUM1 = -10000000,
        ENUM2 = -1,
        ENUM3 = 0,
        ENUM4 = 1,
        ENUM5,
        ENUM6,
        ENUM7 = 0x7FFFFFFF
    };

    typedef PodFixture<TestEnum, sizeof(int32_t)> TestEnumFixture;

    TestEnum enum_array[] = {
        ENUM1,
        ENUM2,
        ENUM3,
        ENUM4,
        ENUM5,
        ENUM6,
        ENUM7
    };
}

BOOST_FIXTURE_TEST_CASE( save_enum, TestEnumFixture )
{
    TestEnumFixture::_test_save(ENUM1);
    TestEnumFixture::_test_save(ENUM2);
    TestEnumFixture::_test_save(ENUM3);
    TestEnumFixture::_test_save(ENUM4);
    TestEnumFixture::_test_save(ENUM5);
    TestEnumFixture::_test_save(ENUM6);
    TestEnumFixture::_test_save(ENUM7);
}

BOOST_FIXTURE_TEST_CASE( save_enum_array, TestEnumFixture )
{
    TestEnumFixture::_test_save_array(enum_array);
}

BOOST_FIXTURE_TEST_CASE( load_enum_native, TestEnumFixture )
{
    TestEnumFixture::_test_load(ENUM1, false);
    TestEnumFixture::_test_load(ENUM2, false);
    TestEnumFixture::_test_load(ENUM3, false);
    TestEnumFixture::_test_load(ENUM4, false);
    TestEnumFixture::_test_load(ENUM5, false);
    TestEnumFixture::_test_load(ENUM6, false);
    TestEnumFixture::_test_load(ENUM7, false);
}

BOOST_FIXTURE_TEST_CASE( load_enum_foreign, TestEnumFixture )
{
    TestEnumFixture::_test_load(ENUM1, true);
    TestEnumFixture::_test_load(ENUM2, true);
    TestEnumFixture::_test_load(ENUM3, true);
    TestEnumFixture::_test_load(ENUM4, true);
    TestEnumFixture::_test_load(ENUM5, true);
    TestEnumFixture::_test_load(ENUM6, true);
    TestEnumFixture::_test_load(ENUM7, true);
}

BOOST_FIXTURE_TEST_CASE( load_enum_array_native, TestEnumFixture )
{
    TestEnumFixture::_test_load_array(enum_array, false);
}

BOOST_FIXTURE_TEST_CASE( load_enum_array_foreign, TestEnumFixture )
{
    TestEnumFixture::_test_load_array(enum_array, true);
}


//
// Bool
//

namespace
{
    typedef PodFixture<bool, sizeof(uint8_t)> BoolFixture;

    bool bool_array[] = {
        false,
        true,
        false,
        true,
        true,
        true,
        false
    };
}

BOOST_FIXTURE_TEST_CASE( save_bool, BoolFixture )
{
    BoolFixture::_test_save(false);
    BoolFixture::_test_save(true);
}

BOOST_FIXTURE_TEST_CASE( save_bool_array, BoolFixture )
{
    BoolFixture::_test_save_array(bool_array);
}

BOOST_FIXTURE_TEST_CASE( load_bool_native, BoolFixture )
{
    BoolFixture::_test_load(false, false);
    BoolFixture::_test_load(true, false);
}

BOOST_FIXTURE_TEST_CASE( load_bool_foreign, BoolFixture )
{
    BoolFixture::_test_load(false, true);
    BoolFixture::_test_load(true, true);
}

BOOST_FIXTURE_TEST_CASE( load_bool_array_native, BoolFixture )
{
    BoolFixture::_test_load_array(bool_array, false);
}

BOOST_FIXTURE_TEST_CASE( load_bool_array_foreign, BoolFixture )
{
    BoolFixture::_test_load_array(bool_array, true);
}

//
// Dynamic array of fundamental types
//

// Test saving/loading from/to dynamic arrays of fundamental types
BOOST_FIXTURE_TEST_CASE( load_save_dyn_array, Fixture )
{
    float *data = new float[2];
    data[0] = 42.0f;
    data[1] = -1991.0f;

    {
        SerializationInterface *ret;
        
        for( int i = 0; i < 2; ++i )
        {
            BOOST_CHECK_NO_THROW( ret = &(m_ser << data[i]) );
            BOOST_CHECK_EQUAL( ret, &m_ser );
        }

        BOOST_CHECK_EQUAL( m_buf.get_size(), 2*sizeof(float) );
    }
    {
        DeserializationInterface *ret;
        float *read_back = new float[2];
        
        for( int i = 0; i < 2; ++i )
        {
            BOOST_CHECK_NO_THROW( ret = &(m_deser >> read_back[i]) );
            BOOST_CHECK_EQUAL( ret, &m_deser );
            BOOST_CHECK_EQUAL( data[i], read_back[i] );
        }

        BOOST_CHECK_EQUAL( m_buf.get_size(), 0u );

        delete[] read_back;
    }

    delete[] data;
}



BOOST_AUTO_TEST_SUITE_END();
