=head1 NAME

iPE::State::AltSplice - Defines a special kind of state called an altsplice state.

=head1 DESCRIPTION

The alt splice state distinguishes itself from states in that since ASEs are generally not explicitly annotataed, they have to be inferred by the occurrence of overlapping transcripts.  This is done in the detect () function.  This is a base class; the subclass is determined by the altsplice_type tag in the XML input file.

=head1 FUNCTIONS

=over 8

=cut

package iPE::State::AltSplice;
use iPE;
use base("iPE::State");

use strict;

=item new (attributes)

Creates a new AltSplice state.

=cut

sub new {
    my $class = shift;
    my $this = $class->SUPER::new(@_);
    my ($tag, $m) = @_;

    $this->{altsplice_type_} = $m->{altsplice_type};
    $this->{in_transition_}  = $m->{in_transition};
    $this->{origin_state_}   = $m->{origin_state};
    $this->{alt_states_}     = [];
    $this->{alt_states_}     = [ split (" ", $m->{alt_states}) ]
      if(defined($m->{alt_states}));

    return $this;
}

=item getClassname(altspliceType)

This function gets the classname for a altsplice type after checking if it exists.  If the class does not exist, it dies.  This is not an object method.

=cut
sub getClassname {
    my $type = shift;
    my $classname = __PACKAGE__."::$type";
    
    eval "require $classname";
    die __PACKAGE__.": The altsplice type $type does not exist in iPE.\n$@" 
        if ($@); 

    return $classname;
}

=item inTransition (), altspliceType (), originState (), alternateStates ()

These are all defined in the originating xml document that described the altsplice state.  The altspliceType implies which class will be used to define the altsplice feature object, the inTransition is the expected transition preceding the altsplice state, and the originState is the state which will be examined for potential altsplicing, and thus altered.  (If found, the incoming transition will be altered to have the state of the type in the in transition field as well.)  

The list of alternateStates contains states which may be considered candidate states for the alternate isoforms, whereas the origin state usually implies the core constituent state.  For example, with MultiSite altsplice states, the frame of the innermost feature is the frame of the origin state.  However when a frameshifting alternate site is found, different states must be considered for the other splice sites.

=cut
sub inTransition    { shift->{in_transition_}   }
sub altspliceType   { shift->{altsplice_type_}  }
sub originState     { shift->{origin_state_}    }
sub alternateStates { shift->{alt_states_}      }

=item validAlternateState (state)

Returns 1 if the state is in the list of valid alternate states for this AltSplice state, 0 otherwise.

=cut
sub validAlternateState {
    my ($this, $state) = @_;
    my $stateName = $state->name;
    for my $altState(@{$this->alternateStates}) {
        return 1 if($altState eq $stateName);
    }
    return 0;
}

=item validState (state) 

Returns 1 if the state passed is the origin state or is in the list of alternate states, 0 otherwise.

=cut
sub validState {
    my ($this, $state) = @_;
    return 1 if($state->name eq $this->originState);
    return $this->validAlternateState($state);
}

=back

=head1 STUB FUNCTIONS

=over 8

=item detect (transcripts, seqLength)

This function must be defined in all altsplice states.  It takes in a group of overlapping transcripts and finds all altsplice cases of its kind in the annotation and alters the annotation as necessary.

=cut

sub detect {
    die __PACKAGE__.
        ":Class ".ref($_[0])." does not define the detect method.\n";
}

=back

=head1 SEE ALSO

L<iPE::gHMM> L<iPE::State>

=head1 AUTHOR

Bob Zimmermann (rpz@cse.wustl.edu)

=cut

1;
