=head1 NAME

iPE::State::AltSplice::MultiFivePrime - Multi 5' site state.

=head1 DESCRIPTION

This altsplice state looks for multiple 5' sites of any feature.  If features overlap and contain multiple 5' sites, then a state is defined for both of these transcripts as being the MultiFivePrime state.

=head1 FUNCTIONS

=over 8

=cut

package iPE::State::AltSplice::MultiFivePrime;
use iPE;
use iPE::Util::DNATools;
use base("iPE::State::AltSplice::MultiSite");
use strict;

=item detect (transcripts, seqLen)

Detect the multi 5' sites in the passed array of annotations, and alter the annotation to contain these states.

What we're looking for is this:

 =====|||||||||=====  1
 ====||||||||||=====  2
 ======||||||||=====  3

Layer 1 and 2 can happen any number of times in the gene.  We want to create two additional states, the first will be the space between 2's 5' end and 1's 5' end, and the second will be between 1's 5' end and 3's 5' end.

All features which are on opposing strands will not be candidates for grouping into multi 5' sites.

=cut
sub detect {
    my ($this, $transcripts, $seqLen) = @_;

    my $featGroups = $this->getGroups($transcripts, $seqLen);

    for my $featGroup (@$featGroups) {
        next if scalar(@$featGroup) == 1;
        my $coreFeat = shift @$featGroup;
        my $strand = $coreFeat->strand;
        for my $feat (@$featGroup) {
            if($strand eq '+' && $feat->strand eq '+' &&
                    $feat->start < $coreFeat->start) {
                $feat->transcript->addFeature(
                    $this, $feat->start, $coreFeat->start-1,
                    $feat->startFrame, 
                    getOpposingFrame($feat->start, $coreFeat->start-1,
                        $feat->startFrame));

                $feat->start($coreFeat->start);
                $feat->startFrame(
                    getOpposingFrame($feat->start, $feat->end, $feat->endFrame)
                    );
            }
            elsif($strand eq '-' && $feat->strand eq '-' &&
                    $feat->end > $coreFeat->end) {
                $feat->transcript->addFeature(
                    $this, $coreFeat->end+1, $feat->end,
                    getOpposingFrame($coreFeat->end+1, $feat->end,
                        $feat->endFrame), $feat->endFrame);

                $feat->end($coreFeat->end);
                $feat->endFrame(
                    getOpposingFrame($feat->start, $feat->end, 
                        $feat->startFrame));
            }
        }
    }
}

=back

=head1 SEE ALSO

L<iPE::State::AltSplice> L<iPE::State>

=head1 AUTHOR

Bob Zimmermann (rpz@cse.wustl.edu)

=cut

1;
