/* 
 * ppmtomd
 * Convert from PPM to the PCL dialect used by the Alps MicroDry
 * series of printers.
 * (c) J.C. Bradfield 2000-2004.
 * Some parts based on ppmtopcl by Angus Duggan.
 * This program is copyright, and is distributed under the terms
 * of the GNU General Public License, version 2 or any later version.
 * Please rename significantly modified versions.
 * This program is not warranted for any purpose.
 * It may be possible to damage your printer using this program.
 */

#include "version.h"
static const char rcsid[] = "$Header: /home/jcb/piva/ppmtomd/RCS/ppmtomd.c,v 2.18 2009/09/09 16:23:27 jcb Exp $";

#if defined( __GNUC__) && !defined(__GNUG__)
# define UNUSED __attribute ((unused))
#else
# define UNUSED
#endif

#define STDOUT 1

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <math.h>
#include <assert.h>
#include <pam.h>

#include "mddata.h"

static void expand_lut(unsigned char inlut[][16][16][4],
  unsigned char outlut[][64][64][4]);

/* lookup tables in external files */
extern unsigned char photo_colcor[16*16*16*4];
extern unsigned char vphoto_colcor[16*16*16*4];
extern unsigned char dyesub_colcor[16*16*16*4];

/* The default printer model when called as ppmtomd */
#ifndef DEFAULT_MODEL
#define DEFAULT_MODEL modelDP5000
#endif

/* logical page length, width; top and left unprintable areas,
   in 600dpi dots, for each paper size.
   Two of these: one for pre-5000 models, one for post */
static const struct {
  int width, length, left, top; 
} papersize_info_pre5000[] = {
  /* Actually, the left margin for the custom size varies (for
     Printiva models, at any rate) on where the printer was 
     destined for. It's 80 in Europe and Japan, and 150 in the U.S.
     Don't know what happens in the pre5000 alps models.
     But we'll just use 80, 'cos who cares... */
  { 0, 0, 80, 284 }, /* custom */
  { 4190, 5660, 80, 284 }, /* executive */
  { 4800, 5960, 150, 284 }, /* letter */
  { 4800, 7760, 150, 284 }, /* legal */
  { 4800, 6372, 80, 284 }, /* a4 */
  { 4138, 5430, 80, 284 }, /* b5 */
  { 2202, 2870, 80, 284 }, /* postcard */
  { 2202, 3114, 80, 284 } /* dyesublabel */
};

/* and the 5000 models */
static const struct {
  int width, length, left, top; 
} papersize_info_5000[] = {
  { 0, 0, 80, 284 }, /* custom */
  { 4190, 5660, 80, 284 }, /* executive */
  { 4940, 5960, 80, 284 }, /* letter */
  { 4940, 7760, 80, 284 }, /* legal */
  { 4800, 6372, 80, 284 }, /* a4 */
  { 4138, 5430, 80, 284 }, /* b5 */
  { 2202, 3082, 80, 71 }, /* postcard */
  { 2202, 3353, 80, 71 } /* dyesublabel */
};

/* index of the components that may be rasterized, in printing order */
typedef enum { compU1, compU2, compU3, compU4,
	       compC, compM, compY, compK, 
	       compS1, compS2, compS3, compS4, numComps } components;

static const OptEntry component_table[] = {
  /* four under colours */
  { "U1", compU1 },
  { "U2", compU2 },
  { "U3", compU3 },
  { "U4", compU4 },
  /* the main colours */
  { "C", compC },
  { "M", compM },
  { "Y", compY },
  { "K", compK },
  /* four spot colours */
  { "S1", compS1 },
  { "S2", compS2 },
  { "S3", compS3 },
  { "S4", compS4 },
  { "", -1 }, /* remember to put new components before here! */
  { "Cyan", compC },
  { "Magenta", compM },
  { "Yellow", compY },
  { "Black", compK },
  { NULL, -1 }
};

/* mapping from component to default colour, if any */
static const int comp_default_col[numComps] = {
  -1, -1, -1, -1,
  colCyan, colMagenta, colYellow, colBlack,
  -1, -1, -1, -1
};

/* type specifying how a component's value is to be computed from
   the input pixels */
typedef enum { 
  /* a default value. The natural meaning (with colour correction)
     for the CMYK components, and no meaning for other components */
  compInputDefault,
  /* the component value is the cyan component. This is taken to
     be the cyan component in a CMYK model with full black gen
     and undercolour removal, so that one can have four spot
     colours indicated by the four primary ink colours */
  compInputC, 
  compInputM, 
  compInputY, 
  compInputK,
  /* The component is ON if the input pixel lies within a given
     RGB range, OFF otherwise */
  compInputRGBRange,
  /* printed if the input pixel is not white */
  compInputNotWhite,
  /* always printed (in the image area) */
  compInputAlways
} compInput;

typedef struct {
  compInput type;
  /* data for the range type */
  int r_start;
  int r_end;
  int g_start;
  int g_end;
  int b_start;
  int b_end;
} compInputType;

/* auxiliary function: parse a string as either decimal or hexadecimal
   but NOT octal */
static int parse_num(const char *num)
{
  if ( strncasecmp(num,"0x",2) == 0 )
    return strtol(num,NULL,16);
  else 
    return strtol(num,NULL,10);
}

/* parse function for above */
int parse_comp_input(const char *spec,compInputType *res)
{
  /* just for cleanliness */
  res->type = compInputDefault;
  res->r_start = res->r_end =
    res->g_start = res->g_end =
    res->b_start = res->b_end = 0;
  if ( strncasecmp(spec,"c",1) == 0 )
    res->type = compInputC;
  else if ( strncasecmp(spec,"m",1) == 0 )
    res->type = compInputM;
  else if ( strncasecmp(spec,"y",1) == 0 )
    res->type = compInputY;
  else if ( strncasecmp(spec,"k",1) == 0 )
    res->type = compInputK;
  else if ( strncasecmp(spec,"n",1) == 0 )
    res->type = compInputNotWhite;
  else if ( strncasecmp(spec,"a",1) == 0 )
    res->type = compInputAlways;
  else {
    char rr[32],gr[32],br[32];
    char *p;
    *rr = *gr = *br = 0;
    /* rgb range */
    res->type = compInputRGBRange;
    /* first get the rgb ranges */
    sscanf(spec,"%31[^:]:%31[^:]:%31[^:]",rr,gr,br);
    if ( !*rr ) pm_error("Bad input spec %s",spec);
    p = strchr(rr,'-');
    res->r_start = parse_num(rr);
    if ( p++ )
      res->r_end = parse_num(p);
    else 
      res->r_end = res->r_start;
    if ( !*gr ) pm_error("Bad input spec %s",spec);
    p = strchr(gr,'-');
    res->g_start = parse_num(gr);
    if ( p++ )
      res->g_end = parse_num(p);
    else 
      res->g_end = res->g_start;
    if ( !*br ) pm_error("Bad input spec %s",spec);
    p = strchr(br,'-');
    res->b_start = parse_num(br);
    if ( p++ )
      res->b_end = parse_num(p);
    else 
      res->b_end = res->b_start;
    /* sanity checking */
    if ( res->r_start < 0 || res->r_start > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->r_start);
    if ( res->r_end < 0 || res->r_end > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->r_end);
    if ( res->g_start < 0 || res->g_start > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->g_start);
    if ( res->g_end < 0 || res->g_end > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->g_end);
    if ( res->b_start < 0 || res->b_start > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->b_start);
    if ( res->b_end < 0 || res->b_end > 255 )
      pm_error("number %d out of range (0-255) in input spec", res->b_end);
    if ( res->r_end < res->r_start )
      pm_error("bad range %s in input spec",rr);
    if ( res->g_end < res->g_start )
      pm_error("bad range %s in input spec",gr);
    if ( res->b_end < res->b_start )
      pm_error("bad range %s in input spec",br);
  }
  return 1;
}
/* how to interpret the input file */
typedef enum { defaultInput, monoInput, diagInput } inputType;
static const OptEntry input_table[] = {
  { "Colour", defaultInput },
  { "Monochrome", monoInput },
  { "Diag", diagInput },
  { NULL, -1 }
};

/* bits per pixel (if known) for the modes */
int print_mode_bpps[] = {
  0,
  1,1,
  4,4,4,4,
  4,4,
  1,1,
  4,4
};

/* We don't at present export all the modes, so here are user options */
typedef enum {
  defaultPrintMode = 0,
  standardPrintMode,
  vphotoPrintMode,
  dyesubPrintMode,
} userPrintMode;

const OptEntry user_print_mode_table[] = {
  { "Default", defaultPrintMode },
  { "Standard", standardPrintMode },
  { "VPhoto", vphotoPrintMode },
  { "DyeSub", dyesubPrintMode },
  { NULL, -1 }
};

/* what sort of output to produce.
   It is defined that ppmOut is the first non RGL mode type,
   so we can test for < ppmOut
*/
typedef enum {
  rglOut=0, packetRglOut, ppmOut, ppmDiag, ppmColourDiag
} outputType;

static const OptEntry output_table[] = {
  { "RGL", rglOut },
  { "packetRGL", packetRglOut },
  { "PPM", ppmOut },
  { "Diag", ppmDiag },
  { "ColourDiag", ppmColourDiag },
  { NULL, -1 }
};

/* output resolutions (as in RGL) */
typedef enum {
  res300 = 0x02, res600 = 0x03, res1200 = 0x04
} outRes;

/* halftoning options */
typedef enum {
  ditherNone, /* no dither; snap to primary */
  ditherFS, /* Floyd-Steinberg */
  ditherSquare, /* my special square cell for raster mode */
  ditherHT, /* normal halftoning */
  ditherHTcoarse, /* coarser dot halftoning */
} ditherType;

static const OptEntry dither_table[] = {
  { "None", ditherNone },
  { "FloydSteinberg", ditherFS },
  { "Square", ditherSquare },
  { "Halftone", ditherHT },
  { "CoarseHalftone", ditherHTcoarse },
  { "fs", ditherFS },
  { "ht", ditherHT },
  { "cht", ditherHTcoarse },
  { NULL, -1 }
};

/* colour correction options */
typedef enum {
  colcorNone, /* no correction at all: RGB -> CMY, and no gamma in mono */
  colcorPlain, /* just do RGB -> CMYK, with black gen and undercolour removal */
  colcorPhoto, /* my custom correction */
  colcorDyeSub = colcorPhoto, /* colour correction suitable for dye sub photo printing */
} colcorType;

static const OptEntry colcor_table[] = {
  { "None", colcorNone },
  { "Plain", colcorPlain },
  { "Photo", colcorPhoto },
  { "DyeSub", colcorDyeSub },
  { NULL, -1 }
};

/* structure containing all information about the page to be
   printed. */
typedef struct {
  /* First the details of the ppm file */
  FILE *in_file; /* the file itself */
  FILE *spot_file; /* the input file for spot colours */
  int in_rows,in_cols,in_maxval, in_format; /* dimensions of the input ppm file*/
  /* function to call to process printer commands.
     First argument is tag saying which component (0,1,2,3);
     second is buffer with command, third is length.
     NOTE: in printing modes, the length is in bytes;
     in the ppm output modes, it's in pixels. */
  void (*out_function)(int,unsigned char *,int);
  /* Components to be rasterized and output.
     Up to four components may be produced.
     The interpretation depends on the mode:
     In the monochrome modes (black, econoblack), only one
     component is produced. The input PPM file should be
     monochrome; colour will be converted to grey.
     In the older multi-plane modes, the input file will be used to
     generate CMYK components, according to the colour correction
     options, and the result will be rasterized. In CMYK modes,
     CMYK are interpreted as the colours given in the following
     field of this structure; in the 4-color modes, CMYK will
     be interpreted as the cassette numbers given in the following
     field of this structure. N.B. Decision: we will adopt
     KCMY as the natural order of cassettes!!!
     In the MD-5000 n-color mode, the colours given in following
     field will be used.
     Note that when the prn_out function is called, it is given
     an argument of the component number; thus it is the caller's
     responsibility to separate and store different components
     in the multi-plane modes.
  */
  /* the colours (for select color plane) or
     cassette numbers (for select cassette holder)
     to be used for each component.
     Note: value of -1 in this field means that the component is
     not to be printed.
  */
  int comp_colours[numComps];
  /* fds to send output to */
  int comp_fds[numComps];
  /* this tells how the value of the component is determined.
     See description above of the type */
  compInputType comp_inputs[numComps];
  /* this lists the components in the order in which they are
     to be printed. (This is for dyesub, which has a different order). */
  int comp_print_order[numComps];
  /* This determines how the input file is interpreted.
     The default value means that it is a grey level file
     (for mono modes), or a colour file (for multi-plane modes) with the 
     colours interpreted as explained above.
     Other possibilities are:
     monoInput:  the file is read as a pgm file (in fact only
     the cyan component is read) and all output components are
     given by this value. Gamma correction will be applied.
     diagInput: the input is a file in the format used by
     the diagnostic ppm output; it therefore specifies each
     pixel in four colours, and no colour correction or dithering
     is applied.
  */
  inputType input_type;
  /* the data transfer mode. Each component may have
     a different mode.
     blackRaster: monochrome black output
     econoblackRaster: monochrome draft black output (MD-5000 only)
     colourRaster: CMYK raster output (not for MD-5000)
     cassetteRaster: raster by cassette holder number (not for MD-5000)
     colourPlane: CMYK etc plane output
     cassettePlane: plane by cassette holder (not for MD-5000)
     multiPlane: multicolour plane output (MD-5000 only)
  */
  transferMode transfer_mode[numComps];
  /* print mode requested by user */
  userPrintMode user_print_mode;
  /* the print mode for each component, in the 5000 series */
  printMode print_mode[numComps];
  /* the type of the raster data to be sent. 0 for bits, 1 for nybbles.
     This is determined by the media type and/or the print mode. */
  int nybble_mode[numComps];
  /* Output type:
     rglOut: to printer
     ppmOut: produce PPM file showing CMY pixels that are set
     ppmDiag: produce PPM file showing which CMYK pixels are set
     ppmColourDiag: produce PPM file showing effect of colour correction
  */
  outputType output_type;
  /* the shift values for x and y. x_shift is in output resolution */
  int x_shift, y_shift;
  /* output resolution */
  outRes output_resolution;
  /* set if we need to double up the x dimension (i.e. we are
     inputting at 600dpi, but outputting at 1200 dpi) */
  int x_double;
  ditherType dither_type;
  mediaType media_type;
  paperSize paper_size;
  model model; /* printer model */
  colcorType colcor_type; /* colour correction */
  int lfadj; /* line-feed adjustment parameter */
} pageInfo;

/* is this pageInfo * one with dyesub media in it? */
#define IS_DYESUB(p) (((p)->media_type == mediaDyeSubPaper) || ((p)->media_type == mediaDyeSubLabel) || ((p)->user_print_mode == dyesubPrintMode))

static int parse_option(const OptEntry table[], char *input);

static void print_page(pageInfo *page_info);
static void rgl_init_page(pageInfo *page_info, int fd);

/* function for outputting the printer commands.
*/
static void print_out(int fd, unsigned char *buf, int len)
{
  if ( write(fd,buf,len) < 0 ) pm_error("Error writing output file\n");
}
/* function for outputting the command wrapped up in packet format */
static void packet_out(int fd, unsigned char *buf, int len)
{
  static char pbuf[9000] = "\002\001" ;
  if ( len > 8192 ) pm_error("Excessively long command for packet - this shouldn't happen\n");
  pbuf[2] = len & 0xFF;
  pbuf[3] = len >> 8;
  memcpy(pbuf+4,buf,len);
  pbuf[len+4] = 0x03;
  if ( write(fd,pbuf,len+5) < 0 ) pm_error("Error writing output file\n");
}

pixval maxval; /* set by main routine */
int format; /* ditto */
/* and for ppm output */
static void ppm_out(int fd UNUSED, unsigned char *buf, int len) 
{
  ppm_writeppmrow(stdout,(pixel *)buf,len,maxval,0);
}

static int opt_halftone = 0;
static int fsdith = 0; /* global used to pass info to jcb_cmyk */

/* This struct carries everything needed to do halftoning */
typedef struct {
  struct {
    int cellsize_r,cellsize_c; /* size of the dither matrix: rows,cols
				  For square matrices and algorithms 
				  assuming them, use just cellsize_c */
    int *dither; /* the dither matrix, as a simple array of length cellsize^2 */
    int x,y,z; /* pythagorean triplet for screen angle: x positive */
    int yneg; /* flag if y is to be negative. We don't actually
		 want to have y negative, to avoid complicating
		 the algorithm, but instead will just invert
		 the rows on input */
    /* temporary storage for the incremental halftoning */
    int S1xi, S1xf, /* stage 1, x, integer and fractional */
		 S1yi, S1yf, /* ... y, ... */
		 S2xi, S2xf, S2yi, S2yf,
	       S3xi, S3xf, S3yi, S3yf,
      hrow,hcol ; /* the indices into the dither matrix */
  } comps[numComps];
} htinfo;

/* this macro sets the screen angle values (with possibly negative y)
   for comp c of the struct htinfo hti */
#define htset(hti,c,X,Y,Z) \
  hti.comps[c].x = X;   hti.comps[c].y = abs(Y);   hti.comps[c].z = Z; \
  hti.comps[c].yneg = (Y < 0);

/* These are the dithering constructs for vphoto dithering, which
   uses a replicated and sheared rectangular matrix to generate
   lines */
#if 1
#define vph_ht_init(h,c,row) (h)->comps[c].hrow = (row)%((h)->comps[c].cellsize_r); (h)->comps[c].hcol = (h)->comps[c].x*(row)%(h)->comps[c].cellsize_c;
#define vph_ht_inc(h,c) if ( ++(h)->comps[c].hcol == (h)->comps[c].cellsize_c ) (h)->comps[c].hcol = 0
#else
#define vph_ht_init ht_init
#define vph_ht_inc ht_inc
#endif

/* These are for normal dithering, with rotated square cell */
/* This macro returns the dither value for comp c in ht struct h */
#define ht_elt(h,c) ((h)->comps[c].dither[(h)->comps[c].cellsize_c*((h)->comps[c].hrow)+((h)->comps[c].hcol)])
/* this is the same but it transpose row and col */
#define ht_telt(h,c) ((h)->comps[c].dither[(h)->comps[c].cellsize_c*((h)->comps[c].hcol)+((h)->comps[c].hrow)])

/* The following is a rotation function, the incremental version
   of which is given by the following macros. The function is
   left here for reference */
/* It is a minor miracle that the incremental versions are actually
   correct. But they really are! */
#if 0
/* Discrete 1-1 rotation by Paeth's 3-shear method */
void rotate(int row,int col,int *hrow,int *hcol,int x,int y,int z,int cellsize) {
  int i,j;
  /* first the x shear, with X' = (X-1)/Y = (x/z - 1)/(y/z) = (x-z)/y */
  i = 2*col*y + 2*row*(x-z); 
  if ( i >= 0 ) i = (i+y)/(2*y) ; else i = (i-y+1)/(2*y);
  j = row ;
  /* now the y shear, with Y' = Y = y/z */
  col = i;
  /* bloody rounding */
  row = 2*y*i + 2*j*z;
  if ( row >= 0 ) row = (row+z)/(2*z) ; else row = (row+1-z)/(2*z);
  /* now x shear again */
  i = 2*col*y + 2*row*(x-z); 
  if ( i >= 0 ) i = (i+y)/(2*y) ; else i = (i-y+1)/(2*y);
  j = row ;
  *hrow = j%cellsize; if ( *hrow < 0 ) (*hrow) += cellsize;
  *hcol = i%cellsize; if ( *hcol < 0 ) (*hcol) += cellsize;
}

/* and here are versions of ht_init and ht_inc that aren't incremental
   at all, but just invoke rotate */

#define ht_init(h,c,row) (h)->comps[c].S1yi = ((h)->comps[c].yneg ? 10000 - row : row) ; (h)->comps[c].S1xi = 0; \
 rotate((h)->comps[c].S1yi,0,&(h)->comps[c].hrow,&(h)->comps[c].hcol, \
   (h)->comps[c].x,(h)->comps[c].y,(h)->comps[c].z,(h)->comps[c].cellsize_c);

#define ht_inc(h,c) (h)->comps[c].S1xi++ ; \
 rotate((h)->comps[c].S1yi,(h)->comps[c].S1xi,&(h)->comps[c].hrow,&(h)->comps[c].hcol, \
   (h)->comps[c].x,(h)->comps[c].y,(h)->comps[c].z,(h)->comps[c].cellsize_c);

#else

/* This macro initializes the halftone structure for dithering
   component c
   at a given image row. The image col is assumed to be zero.
   If the yneg flag is set, we use 10000 - row, to invert things */
#define ht_init(h,c,row) \
 if ( (h)->comps[c].y == 0 ) { (h)->comps[c].hrow = row; (h)->comps[c].hcol = 0 ; } else { \
 (h)->comps[c].S1xf = 2*((h)->comps[c].yneg ? 10000 - row : row)*((h)->comps[c].x-(h)->comps[c].z); \
 if ( (h)->comps[c].y > 0 ) \
  (h)->comps[c].S1xi = ((h)->comps[c].S1xf-(h)->comps[c].y+1)/(2*(h)->comps[c].y); \
 else \
  (h)->comps[c].S1xi = ((h)->comps[c].S1xf+(h)->comps[c].y+1)/(2*-(h)->comps[c].y); \
 (h)->comps[c].S1xf -= (h)->comps[c].S1xi*2*(h)->comps[c].y; \
 (h)->comps[c].S1yi = ((h)->comps[c].yneg ? 10000 - row : row); (h)->comps[c].S1yf = 0; \
 (h)->comps[c].S2xi = (h)->comps[c].S1xi; (h)->comps[c].S2xf = 0; \
 (h)->comps[c].S2yf = 2*(h)->comps[c].y*(h)->comps[c].S1xi + 2*(h)->comps[c].z*(h)->comps[c].S1yi; \
 if ( (h)->comps[c].S2yf >= 0 ) (h)->comps[c].S2yi = ((h)->comps[c].S2yf+(h)->comps[c].z)/(2*(h)->comps[c].z) ; \
 else (h)->comps[c].S2yi = ((h)->comps[c].S2yf+1-(h)->comps[c].z)/(2*(h)->comps[c].z); \
 (h)->comps[c].S2yf -= 2*(h)->comps[c].z*(h)->comps[c].S2yi ; \
 (h)->comps[c].S3xf = 2*(h)->comps[c].y*(h)->comps[c].S2xi + 2*((h)->comps[c].x-(h)->comps[c].z)*(h)->comps[c].S2yi ; \
 if ( (h)->comps[c].S3xf >= 0 ) (h)->comps[c].hcol = ((h)->comps[c].S3xf+((h)->comps[c].y>0?1:-1)*(h)->comps[c].y)/(2*((h)->comps[c].y>0?1:-1)*(h)->comps[c].y); \
 else (h)->comps[c].hcol = ((h)->comps[c].S3xf-((h)->comps[c].y>0?1:-1)*(h)->comps[c].y+1)/(2*((h)->comps[c].y>0?1:-1)*(h)->comps[c].y); \
 (h)->comps[c].S3xf -= 2*(h)->comps[c].y*(h)->comps[c].hcol; \
 (h)->comps[c].hrow = (h)->comps[c].S2yi ; (h)->comps[c].S3yf = 0 ; \
 } /* end of y == 0 conditional */ \
 (h)->comps[c].hrow %= (h)->comps[c].cellsize_c ; if ( (h)->comps[c].hrow < 0 ) (h)->comps[c].hrow += (h)->comps[c].cellsize_c ; \
 (h)->comps[c].hcol %= (h)->comps[c].cellsize_c ; if ( (h)->comps[c].hcol < 0 ) (h)->comps[c].hcol += (h)->comps[c].cellsize_c ;

/* This macro increments the halftone calculation for component c
   by one column */
#define ht_inc(h,c) \
 if ( (h)->comps[c].y == 0 ) { (h)->comps[c].hcol++ ; } else { \
 /* stage 1: incrementing col means ... just */ \
 (h)->comps[c].S1xi += ((h)->comps[c].y>0?1:-1) ; \
 /* S1y is unchanged */ \
 /* Stage 2 : S1X has been incremented by sgn(y), so */ \
 (h)->comps[c].S2xi += ((h)->comps[c].y>0?1:-1) ; \
 (h)->comps[c].S2yf += 2*(h)->comps[c].y ; \
 if ( (h)->comps[c].S2yf >= (h)->comps[c].z ) { \
  /* increment S2y */ \
  (h)->comps[c].S2yi++ ; (h)->comps[c].S2yf -= 2*(h)->comps[c].z; \
  /* now do stage three: S2x and y have both been incremented */ \
  (h)->comps[c].hcol += ((h)->comps[c].y>0?1:-1) ; /* because of the 2*y */ \
  (h)->comps[c].S3xf += 2*((h)->comps[c].x-(h)->comps[c].z) ; \
  while ( (h)->comps[c].S3xf < -((h)->comps[c].y>0?1:-1)*(h)->comps[c].y ) { (h)->comps[c].hcol-- ; (h)->comps[c].S3xf += 2*((h)->comps[c].y>0?1:-1)*(h)->comps[c].y ; } \
  (h)->comps[c].hrow++; \
 } else if ((h)->comps[c].S2yf >= -(h)->comps[c].z ) { \
  /* no increment to S2y; do stage three with only S2x incd */ \
  (h)->comps[c].hcol += ((h)->comps[c].y>0?1:-1) ; \
 } else /* < -(h)->comps[c].z */ { \
  /* decrement S2y */ \
  (h)->comps[c].S2yi-- ; (h)->comps[c].S2yf += 2*(h)->comps[c].z; \
  /* now do stage three: S2x and y have both been decremented */ \
  (h)->comps[c].hcol-- ; /* because of the 2*y */ \
  (h)->comps[c].S3xf -= 2*((h)->comps[c].x-(h)->comps[c].z) ; \
  while ( (h)->comps[c].S3xf >= ((h)->comps[c].y>0?1:-1)*(h)->comps[c].y ) { (h)->comps[c].hcol++ ; (h)->comps[c].S3xf -= 2*((h)->comps[c].y>0?1:-1)*(h)->comps[c].y ; } \
  (h)->comps[c].hrow--; \
 } \
 } /* end of if y == 0 conditional */ \
 if ( (h)->comps[c].hrow >= (h)->comps[c].cellsize_c ) (h)->comps[c].hrow -= (h)->comps[c].cellsize_c; \
 if ( (h)->comps[c].hrow < 0 ) (h)->comps[c].hrow += (h)->comps[c].cellsize_c; \
 if ( (h)->comps[c].hcol >= (h)->comps[c].cellsize_c ) (h)->comps[c].hcol -= (h)->comps[c].cellsize_c; else \
 if ( (h)->comps[c].hcol < 0 ) (h)->comps[c].hcol += (h)->comps[c].cellsize_c ;

#endif /* normal routines */

/* This dithering algorithm is intended for use with the Printiva, which
   doesn't seem well-suited to simple Floyd-Steinberg.
   It is a clustered-dot ordered dither, with a few hacks.
   The standard monochrome dither takes a 4x4 cell, and replicates it into
   a 2x2 supercell: AB
                    CD
   where the cells A and D have black growing out from the centre, and
   the cells B and C have white growing out from the centre.
   In order to achieve the best (we hope) mix of black and colour, we
   do exactly that for CMY, but for black we swap the roles of AD and BC,
   so that black dots don't (usually...) coincide with colour dots.
   In addition, we offset the thresholds in D by one from A, out of a range
   of 64 instead of 32; so theoretically (at least) we double the resolution
   (and similarly for C and B).
   */


/* times 4, to get values from 256 */
static const int cluster8[8][8] = {
  { 24*4,12*4,20*4,26*4,39*4,51*4,43*4,37*4, },
  { 22*4, 0*4, 2*4, 4*4,41*4,63*4,61*4,59*4, },
  { 16*4, 8*4, 6*4,10*4,47*4,55*4,57*4,53*4, },
  { 30*4,18*4,14*4,28*4,33*4,45*4,49*4,35*4, },
  { 38*4,50*4,42*4,36*4,25*4,13*4,21*4,27*4, },
  { 40*4,62*4,60*4,58*4,23*4, 1*4, 3*4, 5*4, },
  { 46*4,54*4,56*4,52*4,17*4, 9*4, 7*4,11*4, },
  { 32*4,44*4,48*4,34*4,31*4,19*4,15*4,29*4 } } ;


/* here's an attempt at a highres table, 4 by 8 */
static const int hcluster[4][8] = {
  /* this is real */
  { 16*4, 0*4, 4*4,24*4,46*4,62*4,58*4,38*4, },
  { 28*4, 8*4,12*4,20*4,34*4,54*4,50*4,42*4, },
  { 44*4,60*4,56*4,36*4,18*4, 2*4, 6*4,26*4, },
  { 32*4,52*4,48*4,40*4,30*4,10*4,14*4,22*4  } };


/* try a line screen with 8x8 */
static const int dithmat8line[8*8] = {
    32*4,36*4,40*4,44*4,34*4,38*4,42*4,46*4,
     0*4, 4*4, 8*4,12*4, 2*4, 6*4,10*4,14*4,
    48*4,52*4,56*4,60*4,50*4,54*4,58*4,62*4,
    16*4,20*4,24*4,28*4,18*4,22*4,26*4,30*4,
    35*4,39*4,43*4,47*4,33*4,37*4,41*4,45*4,
     3*4, 7*4,11*4,15*4, 1*4, 5*4, 9*4,13*4,
    51*4,55*4,59*4,63*4,49*4,53*4,57*4,61*4,
    19*4,23*4,27*4,31*4,17*4,21*4,25*4,29*4 } ;

/* 6x6 line screen, single line */
static const int dithmat6line[6*6] = {
    56*4,48*4,52*4,58*4,50*4,54*4,
    32*4,24*4,28*4,34*4,26*4,30*4,
     8*4, 0*4, 4*4,10*4, 2*4, 6*4,
    20*4,12*4,16*4,22*4,14*4,18*4,
    44*4,36*4,40*4,46*4,38*4,42*4,
    63*4,59*4,61*4,63*4,60*4,62*4 
};

/* 4x4 line screen */
static const int dithmat4line[4*4] = {
  8*4,24*4,40*4,56*4,
  48*4, 0*4,16*4,32*4,
  44*4,60*4,12*4,28*4,
  20*4,36*4,52*4, 4*4
};

/* 5x5 horizontal screen */
static const int dithmat5line[5*5] = {
  53*4,49*4,45*4,47*4,51*4,
  27*4,21*4,15*4,18*4,24*4,
  12*4, 6*4, 0*4, 3*4, 9*4,
  42*4,36*4,30*4,33*4,39*4,
  63*4,59*4,55*4,57*4,61*4
};

/* dot growing outward */
#if 1
// compromise between the following two, squashing the high end
static const int dithmat6dot[6*6] = {
     100, 40,140,176,222,144,
      20,  0, 60,234,246,208,
     160, 80,120,128,192,160,
     184,228,152,110, 50,150,
     240,252,216, 30, 10, 70,
     136,200,168,170, 90,130  };
#else
#if 0
// reduced version to keep yellow and black intensity down 
static const int dithmat6dot[6*6] = {
    30*4,12*4,42*4,63*4,63*4,60*4,
     6*4, 0*4,18*4,63*4,63*4,63*4,
    43*4,24*4,36*4,54*4,63*4,63*4,
    63*4,63*4,63*4,33*4,15*4,45*4,
    63*4,63*4,63*4, 9*4, 3*4,21*4,
    57*4,63*4,63*4,51*4,27*4,39*4  };
#else
// normal version
static const int dithmat6dot[6*6] = {
    20*4, 8*4,28*4,48*4,58*4,40*4,
     4*4, 0*4,12*4,60*4,62*4,56*4,
    32*4,16*4,24*4,36*4,52*4,44*4,
    50*4,60*4,42*4,22*4,10*4,30*4,
    62*4,254,58*4, 6*4, 2*4,14*4,
    38*4,54*4,46*4,34*4,18*4,26*4  };
#endif
#endif

static const int dithmat10[10*10] = {
#if    0
    42*4,26*4,20*4,34*4,43*4,54*4,55*4,61*4,53*4,49*4,
    30*4,10*4, 4*4,14*4,40*4,56*4,58*4,63*4,59*4,57*4,
    18*4, 2*4, 0*4, 6*4,24*4,60*4,62*4,64*4,62*4,60*4,
    38*4,16*4, 8*4,12*4,32*4,57*4,59*4,63*4,58*4,56*4,
    44*4,36*4,22*4,28*4,45*4,48*4,52*4,61*4,55*4,47*4,
    54*4,55*4,61*4,53*4,49*4,42*4,27*4,21*4,35*4,43*4,
    56*4,58*4,63*4,59*4,57*4,31*4,11*4, 5*4,15*4,41*4,
    60*4,62*4,64*4,62*4,60*4,19*4, 3*4, 1*4, 7*4,25*4,
    57*4,59*4,63*4,58*4,56*4,39*4,17*4, 9*4,13*4,33*4,
    48*4,52*4,61*4,55*4,47*4,44*4,37*4,23*4,29*4,46*4
#else
    27*4,19*4,15*4,23*4,31*4,41*4,52*4,55*4,49*4,37*4,
    25*4,10*4, 4*4,12*4,21*4,43*4,58*4,62*4,60*4,48*4,
    17*4, 2*4, 0*4, 6*4,18*4,53*4,64*4,64*4,64*4,54*4,
    22*4,13*4, 8*4,14*4,26*4,47*4,61*4,63*4,59*4,45*4,
    33*4,24*4,16*4,20*4,29*4,35*4,50*4,56*4,51*4,39*4,
    42*4,52*4,55*4,49*4,38*4,28*4,19*4,15*4,23*4,32*4,
    44*4,58*4,62*4,60*4,48*4,25*4,11*4, 5*4,12*4,21*4,
    53*4,64*4,64*4,64*4,54*4,17*4, 3*4, 1*4, 7*4,18*4,
    47*4,61*4,63*4,59*4,46*4,22*4,13*4, 9*4,14*4,26*4,
    36*4,50*4,57*4,51*4,40*4,34*4,24*4,16*4,20*4,30*4,
#endif
} ;

/* These tiles are used in vphoto dithering.
   They're not rotated, but rather sheared. */

static const int tile112[11*2] = {
 0*4, 6*4,12*4,18*4,24*4,30*4,38*4,46*4,54*4,62*4,63*4,
 3*4, 9*4,15*4,21*4,27*4,34*4,42*4,50*4,58*4,63*4,63*4,
};

static const int tile73[7*3] = {
 0*4,12*4,21*4,30*4,39*4,51*4,63*4,
 3*4,15*4,24*4,33*4,43*4,55*4,63*4,
 9*4,18*4,27*4,36*4,47*4,59*4,63*4,
};

static const int tile74[7*4] = {
  0*4, 24*4, 48*4, 12*4, 36*4, 60*4, 63*4,
  3*4, 27*4, 51*4, 15*4, 39*4, 63*4, 63*4,
  6*4, 30*4, 54*4, 18*4, 42*4, 63*4, 63*4,
  9*4, 33*4, 57*4, 21*4, 45*4, 63*4, 63*4,
};

/* a 4 by 4 dot (for black) */
static const int tile44[4*4] = {
   0*4, 5*4,25*4,63*4,
  10*4,15*4,20*4,63*4,
  35*4,30*4,24*4,60*4,
  40*4,45*4,50*4,55*4,
};

/* a 6x6 tile for black */
static const int tile66[6*6] = {
  58*4,52*4,50*4,26*4,24*4,22*4,
  60*4,44*4,42*4,32*4,14*4,12*4,
  63*4,36*4,34*4,34*4,10*4, 8*4,
  20*4,18*4,16*4,62*4,56*4,54*4,
  28*4, 6*4, 4*4,63*4,48*4,46*4,
  30*4, 2*4, 0*4,63*4,40*4,38*4,
};

/* a tile for yellow */
static const int tile66y[6*6] = {
   0*4, 2*4, 4*4, 6*4, 8*4,10*4,
  38*4,40*4,42*4,44*4,46*4,12*4,
  36*4,62*4,63*4,63*4,48*4,14*4,
  34*4,60*4,63*4,63*4,50*4,16*4,
  32*4,58*4,56*4,54*4,52*4,18*4,
  30*4,28*4,26*4,24*4,22*4,20*4,
};

static unsigned char initgamma[256] ;
static unsigned char satgamma[256] ;

static double opt_kfactor = 1.0; /* k gen factor */

/* The following colour correction stuff is no longer used;
   instead, we've reverse-engineered the Windows driver values
*/

#if 0

/* new version for halftoning: start from no change except
   gamma and black adjustment */

#define NOHUE -1

/* Convert RGB to HSV, putting results in last three args;
   leaves hue unchanged if the result is no hue */
static void RGBtoHSV(pixval r, pixval g, pixval b,
	      double *hp, double *sp, double *vp)
{
  double rd, gd, bd, h, s, v, max, min, del, rc, gc, bc;

  /* convert RGB to HSV */
  rd = r / 255.0;            /* rd,gd,bd range 0-1 instead of 0-255 */
  gd = g / 255.0;
  bd = b / 255.0;

  /* compute maximum of rd,gd,bd */
  if (rd>=gd) { if (rd>=bd) max = rd;  else max = bd; }
  else { if (gd>=bd) max = gd;  else max = bd; }
  
  /* compute minimum of rd,gd,bd */
  if (rd<=gd) { if (rd<=bd) min = rd;  else min = bd; }
  else { if (gd<=bd) min = gd;  else min = bd; }

  del = max - min;
  v = max;
  if (max != 0.0) s = (del) / max;
  else s = 0.0;
  
  h = NOHUE;

  if (s != 0.0 && v != 0.0) {
    rc = (max - rd) / del;
    gc = (max - gd) / del;
    bc = (max - bd) / del;
    
    if      (rd==max) h = bc - gc;
    else if (gd==max) h = 2 + rc - bc;
    else if (bd==max) h = 4 + gc - rc;
    
    h = h / 6.0;
    if (h<0) h += 1;
  }
  
  /* map near-black to black to avoid weird effects */
  if (v <= .0625) s = 0.0;
  
  if ( h != NOHUE ) *hp = h ;
  *sp = s;
  *vp = v;
}

/* and vice versa */
static void HSVtoRGB(double hf, double sf, double vf,
	      pixval *rp, pixval *gp, pixval *bp)
{
  int    j;
  double h, s, v, rd, gd, bd;
  double f, p, q, t;

  h = hf ; s = sf ; v = vf ;

  if (s==0.0) { rd = v;  gd = v;  bd = v; }
  else {
    if (h==1.0) h = 0.0;
    h = h * 6.0;
    j = (int) floor(h);
    f = h - j;
    p = v * (1-s);
    q = v * (1 - (s*f));
    t = v * (1 - (s*(1 - f)));
    
    switch (j) {
      /* there isn't really a default, since we know that the value
	 will always be 0 to 5; this is just to shut up warnings 
	 about uninitialized variables */
    default:
      pm_error("This can't happen (default case in HSVtoRGB)");
    case 0:  rd = v;  gd = t;  bd = p;  break;
    case 1:  rd = q;  gd = v;  bd = p;  break;
    case 2:  rd = p;  gd = v;  bd = t;  break;
    case 3:  rd = p;  gd = q;  bd = v;  break;
    case 4:  rd = t;  gd = p;  bd = v;  break;
    case 5:  rd = v;  gd = p;  bd = q;  break;
    }
  }

  *rp = (pixval) floor(rd * 255);
  *gp = (pixval) floor(gd * 255);
  *bp = (pixval) floor(bd * 255);
}


static void jcb_cmyk_ht(int *cp, int *mp, int *yp, int *kp, int vphoto)
{
  double kfac;
  pixval c,m,y,k,hmax,knew;
  double h,s,v;
  c = *cp; m = *mp ; y = *yp ; k = *kp ;
  /* add the black back into the colours */ 
  c += k ; m += k ; y += k ;  
  /* first adjust the val by highly non-linear factor */
  RGBtoHSV(255-c,255-m,255-y,&h,&s,&v);
  v = 1.0-initgamma[(int)((1.0-v)*255.0)]/255.0;
  s = satgamma[(int)(s*255.0)]/255.0;
  HSVtoRGB(h,s,v,&c,&m,&y);
  c = 255 - c; m = 255 - m; y = 255 - y;
  m = 97*m/100;
  k = c ; if ( m < k ) k = m ; if ( y < k ) k = y ; 
  hmax = c; if ( m > hmax ) hmax = m ; if ( y > hmax ) hmax = y;
  /* set k = big for highly sat, k = low for low sat */
  /* rethinking: we want a lot of black left in colours
     that are dark *or* highly saturated */
  /*
    Let's say we want almost full black for colours darker
    than about 50%.
  */
  if ( vphoto ) {
    /* black gen for vphoto mode is only for really black bits */
    kfac = opt_kfactor;
    kfac = kfac*(1.0-v)*(1.0-v)*(1.0-v);
    knew = kfac*k/3; k = kfac*k ;
  } else {
    knew = k/2; k = 3*k/4 ;
  }
  c -= knew ; m -= knew ; y -= knew ;
  if ( fsdith ) { c = 2*c/4 ; m = 2*m/4; y = 2*y/4 ; k = 2*k/4 ; }
  *cp = c ; *mp = m ; *yp = y ; *kp = k ;
#if 0
  /* >=0 is redundant since pixvals are unsigned */
  assert(/* c >=0 && */ c <= 255);
  assert(/* m >=0 && */ m <= 255);
  assert(/* y >=0 && */ y <= 255);
  assert(/* k >=0 && */ k <= 255);
#endif
}

#endif /* end of unused colour correction stuff */

/* compare integers for sorting */
static int intcomp(const void *a,const void *b) {
  return *(int *)a - *(int *)b;
}

/* Function for building dither matrices.
   indith is an n by n dither matrix with values in 0-255.
   condith is an m by m matrix with values in 0..m*m-1
   The result is a malloc'ed mn by mn matrix, in which the [i,j]th
   element of the [p,q]th n by n block has a value which is
   condith[p,q]/m*m of the way between indith[i,j] and the 
   next higher value (ignoring duplicates) that occurs in indith.
   The highest value is interpolated to 256.
   To ensure that solid colour is always printed as such,
   values in the out matrix are clamped to 254. */
static int *build_dith(int n, const int *indith, int m, const int *condith) {
  int *sorted;
  int *result;
  int i,j,p,q;
  /* build a sorted copy of indith */
  sorted = (int *)malloc(sizeof(int)*n*n);
  memcpy(sorted,indith,sizeof(int)*n*n);
  qsort(sorted,n*n,sizeof(int),intcomp);

  result = (int *)malloc(sizeof(int)*m*n*m*n);
  for ( i = 0; i < n; i++ ) {
    for ( j = 0; j < n; j++ ) {
      int val, nval, k;
      /* find the value in sorted order */
      val = indith[i*n+j];
      for ( k = 0; sorted[k] != val ; k++ );
      /* get the next value */
      while ( sorted[k] == val && k < n*n ) k++;
      if ( k == n*n ) nval = 256 ; else nval = sorted[k];
      /* and interpolate */
      for ( p = 0; p < m; p++ ) {
	for ( q = 0; q < m; q++ ) {
	  int res = (int)rint(((m*m-condith[p*m+q])*val
		       +condith[p*m+q]*nval)/(1.0*m*m));
	  if ( res > 254 ) res = 254;
	  result[(n*p+i)*m*n + (n*q + j)] = res;
	}
      }
    }
  }
  free(sorted);
  return result;
}

/* unused */

#if 0

/* this function just scales a dither matrix (without antialiasing)*/
static int *scale_dith(int n, const int *indith, int scale)
{
  int *result;
  int i,j,k1,k2;

  result = (int *)malloc(sizeof(int)*n*n*scale*scale);
  for ( i = 0; i < n; i++ )
    for ( j = 0; j < n; j++ )
      for ( k1 = 0; k1 < scale; k1++ )
	for ( k2 = 0; k2 < scale; k2++ )
	  result[(scale*i+k1)*(n*scale)+(scale*j+k2)] = indith[n*i+j];
  return result;
}

#endif /* unused scale_dith */

#define MAXCOLORS 1024

/* dot weight */
static double dot[numComps] = { 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0 };
static double fthresh = 0.5 ;
static int idot[numComps], ithresh; /* integer versions */

char *usage="              [-autoshift]\n"
"              [-black]\n"
"              [-forcecurlcorrection|-nocurlcorrection]\n"
"              [-colourcorrection <colour correction mode>]\n"
"              [-colours <printing colours specification>]\n"
"              [-dither <mode>]\n"
"              [-draft]\n"
"              [-econoblack]\n"
"              [-firstpass] [-midpass] [-lastpass]\n"
"              [-gamma <gamma factor>]\n"
"              [-glossy]\n"
"              [-informat <input type>]\n"
"              [-inresolution <resolution>]\n"
"              [-keepblack]\n"
"              [-lfadjust <adjustment>]\n"
"              [-model <printer model>]\n"
"              [-media <media type>]\n"
"              [-monochrome]\n"
"              [-noglossy]\n"
"              [-nopack]\n"
"              [-noreset]\n"
"              [-outformat <output type>]\n"
"              [-overlay]\n"
"              [-pagelength <length>]\n"
"              [-pagewidth <width>]\n"
"              [-papersize <paper size>]\n"
"              [-phadjust <adjustment>]\n"
"              [-primer]\n"
"              [-printmode <mode>]\n"
"              [-resolution <resolution>]\n"
"              [-satgamma <gamma factor>]\n"
"              [-spotcolours <spot colour spec>]\n"
"              [-spotfile <file>]\n"
"              [-transfermode <mode>]\n"
"              [-undercolours <under colour spec>]\n"
"              [-usemulticolourribbon]\n"
"              [-version]\n"
"              [-xshift <offset>]\n"
"              [-yshift <offset>]\n"
"\n"
"       options for compatibility with previous versions of ppmtomd\n"
"              [-datamode <mode>]\n"
"\n"
"       ppmtocpva compatibility options:\n"
"              [-dummy <ignored argument>]\n"
"              [-halftone]\n"
"              [-pageC] [-pageK] [-pageM] [-pageY]\n"
"              [-ppmout]\n"
"              [-solidblack]\n"
"\n"
"       undocumented options (see source for usage):\n"
"              [-htscreen <screen spec>]\n"
"              [-testint  <integer>]\n"
;
              

/* The actual maximum is 4800, but since the printer discards excess data,
   there's no point in creating errors for small overruns */
#define PCL_MAXWIDTH 6000
#define PCL_MAXHEIGHT 10000 /* 6375 is the A4 value. */
#define PCL_MAXVAL 255

static int opt_nopack = 0;
static int opt_xshift = 0;
static int opt_yshift = 0;
static int opt_autoshift = 0;
static char *dummyarg;
static int opt_pagelength = 0;
static int opt_pagewidth = 0;
static int opt_resolution = 600;
static int opt_inresolution = 0;
static int opt_noreset = 0;
static int ffonly = 0; /* do formfeed at end of page, but not reset.
			  Used internally only */
static int expandx = 0; /* the incoming data is 600dpi, but we're in 1200dpi */
static int opt_overlay = 0;

static char *opt_colourcorrection = NULL;
static char *opt_transfermode = NULL;
static char *opt_dither = NULL;
static int opt_black = 0;
static int opt_econoblack = 0;
static int opt_nocurl = 0;
static int opt_keepblack = 0; /* map pure black to pure black */
static int opt_ppmout = 0;
static char *opt_model = NULL;
static int opt_mono = 0;
static double initgam = 0.0;
static double satgam = 0.0;
static int opt_curlcorrection = 0;
static int opt_glossy = 0;
static char *opt_printmode = NULL;
static int opt_noglossy = 0;
static int opt_lfadj = 0 ;
static int opt_phadj = 0;
static int testint = 0 ;
static char *opt_media = NULL;
static char *opt_papersize = NULL;
static int opt_primer = 0;
static char *opt_colours = NULL;
static char *opt_spotcolours = NULL;
static char *opt_spotfile = NULL;
static char *opt_undercolours = NULL;
static int version;
static char *opt_htscreen = NULL;
static char *opt_informat = NULL;
static char *opt_outformat = NULL;
static int opt_usemulti = 0;
static int opt_pageC = 0, opt_pageM = 0 , opt_pageY = 0 , opt_pageK = 0;
static int twopassesneeded = 0; /* two pass printing required */

/* By default, in pageplane mode C is assumed to be first and K last.
   This can be overridden by the firstpass, midpass and lastpass options
   */
static int lastpass = 0, firstpass = 0, midpass = 0;

/* argument types */
#define DIM 0
#define REAL 1
#define BOOL 2
#define INT  3
#define STRING 4
static struct options {
   char *name;
   int type;
   void *value;
} options[] = {
  {"autoshift", BOOL, &opt_autoshift },
  {"black",        BOOL, &opt_black },
  {"colourcorrection", STRING, &opt_colourcorrection }, 
  {"colours", STRING, &opt_colours },
  {"datamode", STRING, &opt_transfermode },
  {"dither", STRING, &opt_dither },
  {"draft", BOOL, &opt_econoblack },
  {"dummy", STRING, &dummyarg },
  {"econoblack", BOOL, &opt_econoblack },
  {"firstpass",   BOOL, &firstpass },
  {"forcecurlcorrection", BOOL, &opt_curlcorrection},
  {"gamma",    REAL, &initgam },
  {"glossy",  BOOL, &opt_glossy },
  {"halftone",     BOOL, &opt_halftone},
  {"htscreen", STRING, &opt_htscreen},
  {"informat", STRING, &opt_informat},
  {"inresolution",   INT, &opt_inresolution },
  {"keepblack",      BOOL, &opt_keepblack },
  {"kfactor",      REAL, &opt_kfactor },
  {"lastpass",   BOOL, &lastpass },
  {"lfadjust",           INT,  &opt_lfadj},
  {"media",      STRING, &opt_media },
  {"midpass",   BOOL, &midpass },
  {"model",      STRING, &opt_model },
  {"monochrome",          BOOL, &opt_mono},
  {"nocurlcorrection",      BOOL, &opt_nocurl },
  {"noglossy",  BOOL, &opt_noglossy },
  {"nopack",       BOOL, &opt_nopack },
  {"noreset",      BOOL, &opt_noreset },
  {"outformat",  STRING, &opt_outformat },
  {"overlay",      BOOL, &opt_overlay },
  {"pageC",        BOOL, &opt_pageC },
  {"pageK",        BOOL, &opt_pageK },
  {"pageM",        BOOL, &opt_pageM },
  {"pageY",        BOOL, &opt_pageY },
  {"pagelength",   DIM, &opt_pagelength },
  {"pagewidth",    DIM, &opt_pagewidth },
  {"papersize",    STRING, &opt_papersize },
  {"phadjust",           INT,  &opt_phadj},
  {"ppmout",             BOOL, &opt_ppmout},
  {"primer", BOOL, &opt_primer },
  {"printmode", STRING, &opt_printmode},
  {"resolution",   INT, &opt_resolution },
  {"satgamma",     REAL, &satgam },
  {"solidblack",      BOOL, &opt_keepblack },
  {"spotcolours", STRING, &opt_spotcolours },
  {"spotfile", STRING, &opt_spotfile },
  {"testint",     INT,  &testint}, /* random int for testing purposes */
  {"transfermode", STRING, &opt_transfermode },
  {"undercolours", STRING, &opt_undercolours },
  {"usemulticolourribbon", BOOL, &opt_usemulti },
  {"version", BOOL, &version },
  {"xshift",       DIM,  &opt_xshift },
  {"yshift",       DIM,  &opt_yshift },
};

#define putword(w) (putchar(((w)>>8) & 255), putchar((w) & 255))

static int rows ;

htinfo kht; /* initialized later */

unsigned char *colconv; /* huge look up table */

int main(int argc, char *argv[])
{
   FILE *ifd = NULL;
   register int i;
   int cols;
#define FS_BASIC_SCALE 16
#define FS_SCALE (FS_BASIC_SCALE*(int)maxval)
   int zero = 0;
   char buffer[80]; /* buffer for printing */
   int printer_initted = 0; /* is the printer initialized */
   pageInfo page_info;
   
   ppm_init( &zero, argv ); /* don't let pbmplus at the arguments */
   /*   ppm_init( &argc, argv );*/

   while (argc > 1 && argv[1][0] == '-') {
      char *c, *val=argv[1]+1;
      for (i = 0; i < (int)(sizeof(options)/sizeof(struct options)); i++) {
	 if (strlen(val) <= strlen(options[i].name) &&
	     !strncmp(val, options[i].name, strlen(val))) {
	    switch (options[i].type) {
	    case DIM:
	      /* the constants in here were for 300dpi. We work
		 at 600 dpi, hence the 2 everywhere */
	       if (++argv, --argc == 1)
		  pm_usage(usage);
	       for (c = argv[1]; isdigit(*c) || *c == '.' || *c == '-'; c++);
	       if (c[0] == 'p' && c[1] == 't') /* points */
		  *(int *)(options[i].value) = (int)(2*atof(argv[1])*4.16667);
	       else if (c[0] == 'd' && c[1] == 'p') /* decipoints */
		  *(int *)(options[i].value) = (int)(2*atof(argv[1])*0.41667);
	       else if (c[0] == 'i' && c[1] == 'n') /* inches */
		  *(int *)(options[i].value) = (int)(2*atof(argv[1])*300);
	       else if (c[0] == 'c' && c[1] == 'm') /* centimetres */
		  *(int *)(options[i].value) = (int)(2*atof(argv[1])*118.11024);
	       else if (c[0] == 'm' && c[1] == 'm') /* millimetres */
		  *(int *)(options[i].value) = (int)(2*atof(argv[1])*11.81102);
	       else if (!c[0]) /* dots */
		 /* NB here we assume the user knows what it's doing */
		  *(int *)(options[i].value) = atoi(argv[1]);
	       else
		  pm_error("illegal unit of measure %s", c);
	       break;
	    case REAL:
	       if (++argv, --argc == 1)
		  pm_usage(usage);
	       *(double *)(options[i].value) = atof(argv[1]);
	       break;
	    case BOOL:
	       *(int *)(options[i].value) = 1;
	       break;
	     case INT:
	       if (++argv, --argc == 1)
		  pm_usage(usage);
	       *(int *)(options[i].value) = atoi(argv[1]);
	       break;
	    case STRING:
	      if (++argv, --argc == 1)
		pm_usage(usage);
	      *(char **)options[i].value = argv[1];
	    }
	    break;
	 }
      }
      if (i >= (int)(sizeof(options)/sizeof(struct options)))
	 pm_usage(usage);
      argv++; argc--;
   }
   if (argc > 2)
      pm_usage(usage);
   else if (argc == 2)
      ifd = pm_openr(argv[1]);
   else
      ifd = stdin ;

   if ( version ) {
     fprintf(stderr,version_string);
     exit(0);
   }
   /* validate arguments */

   /* first thing we need to know is the model */
   if ( strcmp("ppmtocpva",argv[0]) == 0 )
     page_info.model = modelPrintiva600U;
   else 
     page_info.model = DEFAULT_MODEL;
   if ( opt_model ) {
     int m;
     m = parse_option(model_table,opt_model);
     if ( m < 0 ) pm_error("bad model name %s",opt_model);
     page_info.model = m;
   }

   /* default transfer mode is colour */
   for ( i = 0 ; i < numComps ; i ++ )
     page_info.transfer_mode[i] = colourPlane;

   /* -black is equivalent to -transfermode Black -monochrome  */
   if ( opt_black ) {
     if ( opt_transfermode )
       pm_error("Cannot use -transfermode and -black together");
     opt_transfermode = "Black";
     opt_mono = 1;
   }

   /* -draft is equiv to -transfermode Econoblack -monochrome */
   if ( opt_econoblack ) {
     if ( opt_transfermode )
       pm_error("Cannot use -transfermode and -econoblack together");
     opt_transfermode = "Econoblack";
     opt_mono = 1;
   }

   /* -usemulti is colourRaster with a fix */
   if ( opt_usemulti ) {
     if ( opt_transfermode )
       pm_error("Cannot use -transfermode and -usemulti together");
     opt_transfermode = "RasterColour";
   }
   /* explicit transfer mode given */
   if ( opt_transfermode ) {
     int n;
     n = parse_option(transfer_mode_table,opt_transfermode);
     if ( n < 0 ) 
       pm_error("%s data transfer mode %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_transfermode);
     /* in -usemulti, the K component is separately printed
	in colourPlane mode */
     for ( i = compC; i <= (opt_usemulti ? compY : compK) ; i++ )
       page_info.transfer_mode[i] = n;
   }

   /* mediatype. These are values given by the PCL spec */
   page_info.media_type = mediaPlainPaper; /* default paper */
   if ( opt_media ) {
     int m;
     m = parse_option(media_table,opt_media);
     if ( m < 0 )
       pm_error("%s media type %s",
	 m<-1 ? "ambiguous" : "unknown",
	 opt_media);
     if ( ( m == mediaDyeSubPaper || m == mediaDyeSubLabel )
       && ! mtest(modelHasDyeSub,page_info.model) )
       pm_error("model does not support dye sublimation printing");
     if ( ( m == mediaVPhotoFilm || m == mediaVPhotoCard )
       && ! mtest(modelHasVPhoto,page_info.model) )
       pm_error("model does not support VPhoto printing");
     page_info.media_type = m;
   }

   page_info.user_print_mode = defaultPrintMode;
   if ( opt_printmode ) {
     int n;
     n = parse_option(user_print_mode_table,opt_printmode);
     if ( n < 0 ) {
       pm_error("%s print mode %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_printmode);
     }
     page_info.user_print_mode = n; 
   }

   /* dithering mode */
   if ( IS_DYESUB(&page_info) ) {
     page_info.dither_type = ditherFS;
   } else {
     page_info.dither_type = ditherNone;
   }
   if ( opt_dither ) {
     int n;
     n = parse_option(dither_table,opt_dither);
     if ( n < 0 ) 
       pm_error("%s dither mode %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_dither);
     page_info.dither_type = n;
   }
   if ( opt_halftone ) page_info.dither_type = ditherHT;

   /* set global flag for colour conversion */
   if ( page_info.dither_type == ditherFS ) fsdith = 1;

   /* set the input type */
   page_info.input_type = defaultInput;
   if ( opt_mono ) page_info.input_type = monoInput;
   if ( opt_informat ) {
     int n;
     n = parse_option(input_table,opt_informat);
     if ( n < 0 ) 
       pm_error("%s input format %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_informat);
     page_info.input_type = n;
   }

   /* Is there an input spotfile ? */
   page_info.spot_file = NULL;
   if ( opt_spotfile ) {
     if ( ( page_info.spot_file = fopen(opt_spotfile,"r") ) == NULL )
       pm_error("Unable to open spotfile %s",opt_spotfile);
   }

   /* set the output type */
   page_info.output_type = rglOut;
   if ( opt_outformat ) {
     int n;
     n = parse_option(output_table,opt_outformat);
     if ( n < 0 ) 
       pm_error("%s output format %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_outformat);
     page_info.output_type = n;
   }

   if ( opt_ppmout ) page_info.output_type = ppmOut;

   /* should check media types against printers here */

   /* get the colour correction */
   if ( page_info.dither_type == ditherNone )
     page_info.colcor_type = (IS_DYESUB(&page_info) ? colcorPhoto : colcorPlain);
   else
     page_info.colcor_type = colcorPhoto;
   if ( opt_colourcorrection ) {
     int n;
     n = parse_option(colcor_table,opt_colourcorrection);
     if ( n < 0 ) 
       pm_error("%s colour correction mode %s",
		n<-1 ? "ambiguous" : "unknown",
		opt_colourcorrection);
     page_info.colcor_type = n;
   }

   /* initialize the print modes for 5k */
   for ( i = compU1; i <= compS4; i++ ) 
     page_info.print_mode[i] = byMediaMode;

   /* Set up the printing order. This is the same as the order
      in the enum, except that dyesub swaps y and c */
   for ( i = 0; i < numComps; i++ ) {
     page_info.comp_print_order[i] = i;
   }
   if ( IS_DYESUB(&page_info) ) {
     page_info.comp_print_order[compC] = compY;
     page_info.comp_print_order[compY] = compC;
   }

   /* Set the components we're going to print.
   */
   
   if ( page_info.transfer_mode[compK] == blackRaster
	|| page_info.transfer_mode[compK] == econoblackRaster ) { 
     opt_pageK = 1 ; opt_pageC = opt_pageM = opt_pageY = 0 ;
   } else {
     if ( opt_pageC + opt_pageM + opt_pageY + opt_pageK == 0 )
     opt_pageK = opt_pageC = opt_pageM = opt_pageY = 1;
   }
   /* The following is the default setup, or what can be specified
      by the deprecated -page? options */
   page_info.comp_colours[compU1] = -1 ;
   page_info.comp_colours[compU2] = -1 ;
   page_info.comp_colours[compU3] = -1 ;
   page_info.comp_colours[compU4] = -1 ;
   page_info.comp_colours[compK] = opt_pageK ? colBlack : -1 ;
   page_info.comp_colours[compC] = opt_pageC ? colCyan : -1 ;
   page_info.comp_colours[compM] = opt_pageM ? colMagenta : -1 ;
   page_info.comp_colours[compY] = opt_pageY ? colYellow : -1 ;
   page_info.comp_colours[compS1] = -1 ;
   page_info.comp_colours[compS2] = -1 ;
   page_info.comp_colours[compS3] = -1 ;
   page_info.comp_colours[compS4] = -1 ;
   if ( IS_DYESUB(&page_info) ) page_info.comp_colours[compK] = -1;
   for ( i = 0; i < numComps; i++ ) 
     page_info.comp_inputs[i].type =  compInputDefault;
   /* now if the new -colours option has been used, use it instead */
   if ( opt_colours ) {
     int l = 0;
     char *opt, *endopt, *arg;
     int len = strlen(opt_colours);
     int i;
     /* syntax is comma separated list of component[=colour] pairs */
     for ( i = compC ; i <= compK ; i++ ) page_info.comp_colours[i] = -1;
     /* special case: if value is - or none, then no CMYK colours
	to be printed. */
     if ( strncmp(opt_colours,"-",1) == 0
       || strncasecmp(opt_colours,"none",4) == 0 )
       len = -1; /* hacky way of avoiding rest of parse */
     while ( l < len ) {
       int n;
       opt = (char *)(opt_colours+l);
       endopt = strchr(opt,',');
       if ( endopt ) *endopt = 0;
       l += strlen(opt) + (endopt ? 1 : 0);
       arg = strchr(opt,'=');
       if ( arg ) { *(arg++) = 0; }
       n = parse_option(component_table,opt);
       if ( n < 0 ) { pm_error("bad colour spec %s",opt); }
       /* Set the natural colour for the component */
       switch ( n ) {
       case compC:
       case compM:
       case compY:
       case compK:
	 page_info.comp_colours[n] = comp_default_col[n] ;
	 if ( n == compK && IS_DYESUB(&page_info) )
	   page_info.comp_colours[n] = -1;
	 break;
       default: pm_error("Bad component in colour spec %s",opt);
       }
       if ( arg ) {
	 int c;
	 int bad = 0;
	 c = parse_option(colour_table,arg);
	 if ( c < 0 ) { pm_error("bad colour %s", arg); }
	 /* we don't support colour messing in dyesub mode */
	 if ( IS_DYESUB(&page_info) ) {
	   pm_error("Colour swapping is not supported in dyesub mode");
	 }
	 /* some colours only work on some printers */
	 switch ( c ) {
	 case colWhite:
	   bad = !mtest(modelHasWhite,page_info.model);
	   break;
	 case colMetallicSilver:
	   bad = (page_info.model == modelPrintiva600);
	   break;
	 case colGoldFoil:
	 case colSilverFoil:
	   bad = !mtest(modelHasFoil,page_info.model);
	   break;
	 }
	 if ( bad ) {
	   pm_error("Printer model %s does not support colour %s\n",
	     model_table[page_info.model].name,
	     colour_table[c].name);
	 }
	 page_info.comp_colours[n] = c;
       }
     }
   }

   /* do we have any spot colours specified ? */
   if ( opt_spotcolours ) {
     int l = 0;
     char *opt, *endopt, *arg;
     int len = strlen(opt_spotcolours);

     /* syntax is n=colour=input */
     while ( l < len ) {
       int n;
       opt = (char *)(opt_spotcolours+l);
       endopt = strchr(opt,',');
       if ( endopt ) *endopt = 0;
       l += strlen(opt) + (endopt ? 1 : 0);
       arg = strchr(opt,'=');
       if ( ! arg ) { pm_error("missing colour in spot spec %s",opt); }
       *(arg++) = 0;
       /* which component ? Unfortunately, we can't
	  use the component parser, as we just have numbers */
       n = atoi(opt);
       n += (compS1 - 1);
       if ( n < compS1 || n > compS4 ) 
	 pm_error("Bad spot colour number in spec %s",opt);
       /* now look for the input spec (arg is pointing at
	  the colour now */
       opt = strchr(arg,'=');
       if ( ! opt ) pm_error("missing input spec in spot %s",arg);
       *(opt++) = 0;
       /* which colour? */
       {
	 int c;
	 int bad = 0;
	 c = parse_option(colour_table,arg);
	 if ( c < 0 ) { pm_error("bad colour %s", arg); }
	 /* some colours only work on some printers */
	 switch ( c ) {
	 case colWhite:
	   bad = !mtest(modelHasWhite,page_info.model);
	   break;
	 case colGlossyFinish:
	   bad = !mtest(modelHasFinish,page_info.model);
	   break;
	 case colVPhotoPrimer:
	   bad = !mtest(modelHasVPhotoPrimer,page_info.model);
	   break;
	 case colMetallicSilver:
	   bad = (page_info.model == modelPrintiva600);
	   break;
	 case colGoldFoil:
	 case colSilverFoil:
	   bad = !mtest(modelHasFoil,page_info.model);
	   break;
	 }
	 if ( bad ) {
	   pm_error("Printer model %s does not support colour %s\n",
	     model_table[page_info.model].name,
	     colour_table[c].name);
	 }
	 page_info.comp_colours[n] = c;
       }
       /* finally, get the input type. */
       if ( ! parse_comp_input(opt,&page_info.comp_inputs[n]) ) {
	 pm_error("Bad input spec %s in spot colour",opt);
       }
     }
   }
   
   /* this code is almost the same as above. We should avoid the
      repetition */
   /* do we have any under colours specified ? */
   if ( opt_undercolours ) {
     int l = 0;
     char *opt, *endopt, *arg;
     int len = strlen(opt_undercolours);
     if ( IS_DYESUB(&page_info) ) {
       pm_error("Undercolours are not supported in dyesub mode");
     }
     /* syntax is n=colour=input */
     while ( l < len ) {
       int n;
       opt = (char *)(opt_undercolours+l);
       endopt = strchr(opt,',');
       if ( endopt ) *endopt = 0;
       l += strlen(opt) + (endopt ? 1 : 0);
       arg = strchr(opt,'=');
       if ( ! arg ) { pm_error("missing colour in under spec %s",opt); }
       *(arg++) = 0;
       /* which component ? Unfortunately, we can't
	  use the component parser, as we just have numbers */
       n = atoi(opt);
       n += (compU1 - 1);
       if ( n < compU1 || n > compU4 ) 
	 pm_error("Bad under colour number in spec %s",opt);
       /* now look for the input spec (arg is pointing at
	  the colour now */
       opt = strchr(arg,'=');
       if ( ! opt ) pm_error("missing input spec in under %s",arg);
       *(opt++) = 0;
       /* which colour? */
       {
	 int c;
	 int bad = 0;
	 c = parse_option(colour_table,arg);
	 if ( c < 0 ) { pm_error("bad colour %s", arg); }
	 /* some colours only work on some printers */
	 switch ( c ) {
	 case colWhite:
	   bad = !mtest(modelHasWhite,page_info.model);
	   break;
	 case colVPhotoPrimer:
	   bad = !mtest(modelHasVPhotoPrimer,page_info.model);
	   break;
	   /* no idea why you would want to use finish as an
	      undercolour, but perhaps some special effect
	      can be achieved. */
	 case colGlossyFinish:
	   bad = !mtest(modelHasFinish,page_info.model);
	   break;
	 case colMetallicSilver:
	   bad = (page_info.model == modelPrintiva600);
	   break;
	 case colGoldFoil:
	 case colSilverFoil:
	   pm_error("Foil colours cannot be used as undercolours");
	   break;
	 }
	 if ( bad ) {
	   pm_error("Printer model %s does not support colour %s\n",
	     model_table[page_info.model].name,
	     colour_table[c].name);
	 }
	 page_info.comp_colours[n] = c;
       }
       /* finally, get the input type. */
       if ( ! parse_comp_input(opt,&page_info.comp_inputs[n]) ) {
	 pm_error("Bad input spec %s in under colour",opt);
       }
     }
   }
   
   /* add primer if requested */
   if ( opt_primer ) {
     if ( !mtest(modelHasVPhotoPrimer,page_info.model) ) {
       pm_error("model does not support VPhoto Primer");
     }
     for ( i = compU1; page_info.comp_colours[i] >= 0 && i <= compU4; i++ );
     if ( i > compU4 ) {
       pm_error("no undercolour slots free for VPhoto Primer");
     }
     page_info.comp_colours[i] = colVPhotoPrimer;
     page_info.comp_inputs[i].type = compInputAlways;
   }

   /* glossy is on by default for dye-sub */
   if ( IS_DYESUB(&page_info) && ! opt_noglossy ) opt_glossy = 1;

   /* if glossy, check we're OK */
   if ( opt_glossy && !IS_DYESUB(&page_info) && !mtest(modelHasFinish,page_info.model) ) {
     pm_error("model does not support Glossy Finish ribbon");
   }

   /* Now we need to do some thinking. Firstly, if any of the spot
      colours is foil, we need to print all the CMYK colours, since
      (according to Alps, at any rate) the foils need a thick underlay
      of ink. */
   {
     int hasfoil = 0, hasspot = 0, hascmyk = 0;
     for (i=compC; i <= compK; i++) {
       if ( page_info.comp_colours[i] >= 0 ) hascmyk = 1;
     }
     for (i=compS1; i <= compS4; i++) {
       if ( page_info.comp_colours[i] >= 0 ) hasspot = 1;
       if ( page_info.comp_colours[i] == colGoldFoil
	 || page_info.comp_colours[i] == colSilverFoil 
	 || page_info.comp_colours[i] == colNullFoil ) hasfoil = 1;
     }
     if ( hasfoil ) {
       for (i=compC; i <= compK; i++) {
	 if ( page_info.comp_colours[i] < 0
	   || page_info.comp_colours[i] != comp_default_col[i] ) {
	   fprintf(stderr,"Warning: missing or substituted CMYK ribbons while using foils\n");
	 }
       }
     }
     /* Really, we ought to count how many colours there are
	and decide what to do. However, I think we'll take the
	easy way out: for the pre-5000 series, we'll print in
	two passes, CMYK and spot, and for the 5000 series we'll
	print in one pass, and complain if we have more than
	seven colours. It'll be up to the user to make sure
	the right things are loaded, and to understand what's
	happening when the printer flashes its lights. 
	(At least until we provide the spooler.) */
     if ( mtest(modelHasMultiRaster,page_info.model) ) {
       /* pre-5000 model. We can handle four colours in
	  one pass, or CMKY and two spots if we have the
	  multicolour ribbon. For the moment, we won't worry
	  about the multicolour ribbon */
       int i,n=0;
       if ( opt_glossy ) n++;
       for ( i = 0; i < numComps; i++ )
	 if ( page_info.comp_colours[i] >= 0 
	      && page_info.comp_colours[i] < colNullSpot ) n++;
       if ( n > 4 ) twopassesneeded = 1;
       /* the way we currently do it, a cartridge gets left
	  on the print head when we change. So we have only
	  three slots available for changing */
       if ( n > 7 ) pm_error("Too many printing colours");
     } else {
       /* 5000 model. We can handle 7 colours in one pass.
	  We don't at present offer the option of two pass printing.
	  I think in principle one can do many colours in one pass,
	  relying on the user to load ribbons, but whether this
	  actually works is another matter. */
       int i,n=0;
       if ( opt_glossy ) n++;
       for ( i = 0; i < numComps; i++ )
	 if ( page_info.comp_colours[i] >= 0 
	      && page_info.comp_colours[i] < colNullSpot ) n++;
       if ( n > 7 ) pm_error("Too many printing colours");
       /* switch to multiplane printing */
       for ( i = 0; i < numComps; i++ ) {
	 if ( page_info.transfer_mode[i] == colourPlane && n > 4 )
	   page_info.transfer_mode[i] = multiPlane;
       }
     }
   }

   /* now we have the media type, we can initialize the print and nybble
      fields.
      We assume that undercolours and spot colours are not printed in
      vphoto mode. */

   {
     int m = page_info.media_type;
     int vphotoprimer = 0;
     /* bpp implied by the media type */
     int bpp4 = ( m == mediaDyeSubPaper || m == mediaDyeSubLabel
       || m == mediaVPhotoFilm || m == mediaVPhotoCard );
     for ( i = compU1 ; i <= compU4; i++ ) {
       /* is vphoto primer laid down in the undercoat? */
       if ( page_info.comp_colours[i] == colVPhotoPrimer ) vphotoprimer = 1;
       /* set mode - always bivalued */
       page_info.nybble_mode[i] = 0;
       if ( m == mediaVPhotoFilm || m == mediaVPhotoCard ) {
	 page_info.print_mode[i] = normalColourFilmMode;
       } else if ( m == mediaDyeSubPaper || m == mediaDyeSubLabel ) {
	 /* this is moot, since we don't allow it anyway */
	 page_info.print_mode[i] = normalColourMode;
       } else {
	 page_info.print_mode[i] = byMediaMode;
       }
     }

     for ( i = compC ; i <= compK; i++ ) {
       /* do what the user requested */
       switch ( page_info.user_print_mode ) {
       case defaultPrintMode:
	 page_info.nybble_mode[i] = bpp4;
	 page_info.print_mode[i] = byMediaMode;
	 break;
       case standardPrintMode:
	 page_info.nybble_mode[i] = 0;
	 if ( bpp4 ) {
	   page_info.print_mode[i] = normalColourFilmMode;
	 } else {
	   page_info.print_mode[i] = byMediaMode;
	 }
	 break;
       case vphotoPrintMode:
	 page_info.nybble_mode[i] = 1;
	 if ( m == mediaVPhotoFilm || m == mediaVPhotoCard ) {
	   page_info.print_mode[i] = byMediaMode;
	 } else if ( m == mediaDyeSubPaper || m == mediaDyeSubLabel ) {
	   page_info.print_mode[i] = vphotoColourFilmMode;
	 } else {
	   page_info.print_mode[i] = vphotoprimer ? vphotoColourPrimerMode : vphotoColourPaperMode;
	 }
	 break;
       case dyesubPrintMode:
	 page_info.nybble_mode[i] = 1;
	 if ( m == mediaDyeSubPaper || m == mediaDyeSubLabel ) {
	   page_info.print_mode[i] = byMediaMode;
	 } else {
	   page_info.print_mode[i] = dyeSubColourMode;
	 }
	 break;
       } 
     }
     for ( i = compS1 ; i <= compS4; i++ ) {
       page_info.nybble_mode[i] = 0;
       if ( m == mediaVPhotoFilm || m == mediaVPhotoCard ) {
	 page_info.print_mode[i] = normalColourFilmMode;
       } else if ( m == mediaDyeSubPaper || m == mediaDyeSubLabel ) {
	 /* it might well be the case that dyesub paper looks more
	    like vphoto film. In fact, it probably does, so let's use it */
	 page_info.print_mode[i] = normalColourFilmMode;
       } else {
	 page_info.print_mode[i] = byMediaMode;
       }
     }
   }

   if (! firstpass && ! lastpass && !midpass) {
     firstpass = 1;
     lastpass = 1;
   }

   /* output resolution */
   page_info.output_resolution =
     opt_resolution == 300 ? res300 :
     opt_resolution == 600 ? res600 :
     opt_resolution == 1200 ? res1200 :
     (pm_error("Bad resolution %d",opt_resolution),600);

   /* input resolution. We don't actually take much notice of
      this, except to determine whether we should expand 600 to 1200 */
   if ( opt_inresolution ) {
     if ( opt_resolution == 1200 && opt_inresolution == 600 ) {
       expandx = 1;
     } else if ( opt_resolution != opt_inresolution )
       pm_error("Unsupported input resolution %d with output resolution %d",
	 opt_inresolution, opt_resolution);
   }
   /* papersize */
   page_info.paper_size = paperA4 ; /* default to a4 */
   if ( opt_papersize && opt_papersize[0] ) {
     int p;
     p = parse_option(paper_table,opt_papersize);
     if ( p < 0 ) {
       fprintf(stderr,"bad paper size %s, using A4",opt_papersize);
     } else {
       page_info.paper_size = p;
     }
   }
   /* it appears that the papersize command does not actually set
      the logical page length and width!  How very stupid.
      So we do it ourselves, unless the user specified it */
   if ( opt_pagelength == 0 ) {
     opt_pagelength = 
       mtest(modelHas5000PageSizes,page_info.model) ?
       papersize_info_5000[page_info.paper_size].length :
       papersize_info_pre5000[page_info.paper_size].length;
   }
   if ( page_info.output_resolution == res300 ) opt_pagelength /= 2;

   if ( opt_pagewidth == 0 ) {
     opt_pagewidth = 
       mtest(modelHas5000PageSizes,page_info.model) ?
       papersize_info_5000[page_info.paper_size].width :
       papersize_info_pre5000[page_info.paper_size].width;
   }
   if ( page_info.output_resolution == res300 ) opt_pagewidth /= 2;
   if ( page_info.output_resolution == res1200 ) opt_pagewidth *= 2;

   page_info.x_shift = opt_xshift;
   if ( opt_autoshift ) 
     page_info.x_shift -= 
       mtest(modelHas5000PageSizes,page_info.model) ?
       papersize_info_5000[page_info.paper_size].left :
	 papersize_info_pre5000[page_info.paper_size].left;
   if ( page_info.output_resolution == res300 ) page_info.x_shift /= 2;
   if ( page_info.output_resolution == res1200 ) page_info.x_shift *= 2;
   
   page_info.y_shift = opt_yshift;
   if ( opt_autoshift )
     page_info.y_shift -= 
       mtest(modelHas5000PageSizes,page_info.model) ?
       papersize_info_5000[page_info.paper_size].top :
	 papersize_info_pre5000[page_info.paper_size].top;
   
   if ( page_info.output_resolution == res300 ) page_info.y_shift /= 2;

   /* compute initgamma table  */
   if ( initgam == 0 ) {
     if ( page_info.dither_type == ditherHT 
       || page_info.dither_type == ditherHTcoarse ) {
       if ( page_info.nybble_mode[compC] ) {
	 initgam = -0.8;
       } else {
	 initgam = (page_info.output_resolution == res1200 ? -0.9 : 0.8);
       }
     } else {
       if ( page_info.nybble_mode[compC] ) {
	   initgam = -0.9;
       } else {
	 initgam = 1.2;
       }
     }
   }
   for ( i = 0 ; i < 256 ; i++ ) {
     double ii = i/255.0;
     if ( initgam > 0.0 ) {
       ii = pow(ii,initgam);
     } else {
       ii = 1.0 - pow(1.0-ii,-initgam);
     }
     initgamma[i] = floor(255.0*ii+0.5);
   }
   /* compute satgamma */
   if ( satgam == 0 ) satgam = fsdith ? 0.8 : 
     (page_info.nybble_mode[compC] ? 1.5 : 1.0);
   for ( i = 0 ; i < 256 ; i++ ) {
     double ii = i/255.0;
     if ( satgam > 0.0 ) {
       ii = pow(ii,satgam);
     } else {
       ii = 1.0 - pow(1.0-ii,-satgam);
     }
     satgamma[i] = floor(255.0*ii+0.5);
   }

   
   /* compute the colour conversion lookup table */
   if ( ! opt_mono && page_info.colcor_type == colcorPhoto ) {
     colconv = (unsigned char *)malloc(4*64*64*64);
     expand_lut((unsigned char (*)[16][16][4])(IS_DYESUB(&page_info) ? dyesub_colcor : page_info.nybble_mode[compC] ? vphoto_colcor : photo_colcor), (unsigned char (*)[64][64][4])colconv);
   }

   /* set up the default halftone screens. Undercolours and spot colours
      are not currently halftoned, but just in case we set them to have the
      screen as black */
   for ( i = 0; i < numComps; i++ ) {
     if ( i < compC || i >= compK ) {
       htset(kht,i,1,0,1); // no rotation
     }
   }
   htset(kht,compC,12,5,13); // abt 30 degrees
   htset(kht,compM,12,-5,13); // abt 30 degrees the other way
   htset(kht,compY,3,4,5); // abt 50 degrees
   if ( page_info.dither_type == ditherHTcoarse ) {
     /* change magenta screen angle to be more perpendicular to cyan */
     htset(kht,compM,5,-12,13);
   }

   /* set magenta screen angle from options */
   if ( opt_htscreen ) {
     char *p = opt_htscreen;
     int c; int n;
     while ( *p ) {
       sscanf(p,"%d:%n",&c,&n);
       p += n;
       sscanf(p,"%d,%d,%d%n",&kht.comps[c].x,&kht.comps[c].y,&kht.comps[c].z,&n);
       p += n;
       if ( kht.comps[c].y < 0 ) {
	 kht.comps[c].y = -kht.comps[c].y;
	 kht.comps[c].yneg = 1;
       } else {
	 kht.comps[c].yneg = 0;
       }
     }
   }

   page_info.lfadj = opt_lfadj;


   if ( page_info.transfer_mode[compK] == econoblackRaster
     && !mtest(modelHasEconoBlack,page_info.model) ) {
     pm_error("Printer model %s does not support the EconoBlack ribbon\n",
       model_table[page_info.model].name);
   }

   if ( (page_info.transfer_mode[compC] == colourRaster
	  || page_info.transfer_mode[compC] == cassetteRaster)
     && !mtest(modelHasMultiRaster,page_info.model) ) {
     pm_error("Printer model %s does not support transfer mode %s\n",
       model_table[page_info.model].name,
       transfer_mode_table[page_info.transfer_mode[compC]].name);
   }

   /* At this point, we have analysed all the options. 
      Now do the sanity checking that we couldn't do before. */
   if ( page_info.output_resolution == res1200
     && page_info.transfer_mode[compK] != blackRaster
     && !mtest(modelHasHighResColour,page_info.model) ) {
     pm_error("Printer model %s does not support colour at 1200dpi\n",
       model_table[page_info.model].name);
   }   

   /* Because we may be used as a ghostscript driver, we should cope
      with the possibility of receiving multiple pages as concatenated
      PPM files. */
   while ( !feof(ifd) ) {
     int scols,srows;
     pixval smaxval;
     int sformat;

     /* These ppm routines don't return status.
	They will kill the program on any error. (Bit antisocial, 
	isn't it?) */

     ppm_readppminit( ifd, &cols, &rows, &maxval,&format );
     page_info.in_maxval = maxval;
     page_info.in_format = format;

     if ( page_info.spot_file ) {
       ppm_readppminit(page_info.spot_file,&scols,&srows,&smaxval,&sformat);
       if ( scols != cols || srows != rows 
	 || smaxval != maxval || sformat != format )
	 pm_error("Spotfile must be the same format as main input file");
     }

     /* it's a real pain handling different maxvals. So let's
	change everything to 255 */
     maxval = 255;

     page_info.in_rows = rows;
     page_info.in_cols = cols;
     page_info.x_double = 0;
     if ( page_info.output_resolution == res1200 && expandx ) {
       cols *= 2 ;
       page_info.x_double = 1;
     }
     if (page_info.output_type >= ppmOut) 
       ppm_writeppminit(stdout,
	 cols+(page_info.x_shift<0?page_info.x_shift : 0),
	 rows+(page_info.y_shift<0?page_info.y_shift : 0),maxval,0);
     
     /* limit checks */
     if (cols > (page_info.output_resolution == res1200 ? 2 : 1)*PCL_MAXWIDTH || rows > PCL_MAXHEIGHT)
       pm_error("image too large; reduce with ppmscale");
     if (maxval > PCL_MAXVAL)
       pm_error("colour range too large; reduce with ppmcscale");
     
     
     /* now set up page_info and call print_page */
     page_info.in_file = ifd;
     page_info.out_function = 
       (page_info.output_type == rglOut) ? print_out : 
       (page_info.output_type == packetRglOut) ? packet_out : ppm_out;
     
     /* if we are printing multiple components in page plane,
	we need to set up temporary files for print_out */
     if ( page_info.output_type < ppmOut ) {
       int ncomps,i,j;
       for (ncomps=0, j=0; j < numComps; j++ ) {
	 i = page_info.comp_print_order[j];
	 page_info.comp_fds[i] = -1;
	 if ( page_info.comp_colours[i] >= 0 
	      && page_info.comp_colours[i] < colNullSpot ) {
	   page_info.comp_fds[i] = STDOUT;
	   ncomps++;
	 }
       }
       if ( ncomps > 1 ) {
	 int firstcomp = 1;
	 /* Leave the first printing component to stdout, send
	    the others to temporary file. */
	 for (j=0; j < numComps; j++ ) {
	   i = page_info.comp_print_order[j];
	   /* in raster modes, we have to map CMYK on to 1 fd */
	   if ( (page_info.transfer_mode[i] == colourRaster
		  || page_info.transfer_mode[i] == cassetteRaster)
	     /* However, if the -usemulti option has been given,
		we print K separately */
	     && i > compC && i <= (opt_usemulti ? compY : compK) ) {
	     page_info.comp_fds[i] = page_info.comp_fds[compC];
	     continue;
	   }
	   if ( page_info.comp_colours[i] >= 0 
		&& page_info.comp_colours[i] < colNullSpot ) {
	     if ( !firstcomp ) {
	       char buf[1024];
	       strcpy(buf,"/tmp");
	       if ( getenv("TMPDIR") ) 
		 strcpy(buf,getenv("TMPDIR"));
	       strcat(buf,"/ppmtomdXXXXXX");
	       mkstemp(buf);
	       page_info.comp_fds[i] = open(buf,O_RDWR|O_CREAT|O_TRUNC,0777);
	       if ( page_info.comp_fds[i] < 0 ) 
		 pm_error("Unable to open temporary file\n");
	       /* having opened it, we may as well unlink it so
		  it automatically goes away when we exit */
	       unlink(buf);
	     } else {
	       firstcomp = 0;
	     }
	   }
	 }
       }
     }
     
     
     /* initialize the printer, if appropriate */   
     if ( !printer_initted && page_info.output_type < ppmOut && firstpass) {
       rgl_init_page(&page_info,STDOUT);
       printer_initted = 1;
     }

     /* for each component, generate the cartridge selection command,
	when appropriate.
	At this point, we also deal with the fact that we
	may have to make several passes in different modes.
	So if the component's mode is different from the
	previous, we change it. */
     if ( page_info.output_type < ppmOut ) {
       /* we want to know: the first component to be
	  printed; the last component to be printed;
	  and the first component to be printed in the
	  last mode used */
       int j,prev, c,lastcomp=-1,firstcomp=-1,firstinlast=-1;
       /* array of flags saying whether to change mode at this
	  component */
       int changemode[numComps];
       printMode lastprintmode = byMediaMode;
       int numinpass;

       /* we need to change mode whenever either the mode
	  changes, or if we have processed four components
	  in colourPlane or cassettePlane. */
       for ( prev = -1, j = 0, numinpass = 0; j < numComps; j++ ) {
	 i = page_info.comp_print_order[j];
	 changemode[i] = 0;
	 if ( page_info.comp_colours[i] >= 0 
	      && page_info.comp_colours[i] < colNullSpot ) {
	   if ( firstcomp < 0 ) firstcomp = i;
	   lastcomp = i;
	   if ( (int)page_info.transfer_mode[i] != prev 
		|| ((prev == (int)colourPlane || prev == (int)cassettePlane)
		    && numinpass == 4 ) ) {
	     changemode[i] = 1;
	     firstinlast = i;
	     numinpass = 1;
	     prev = page_info.transfer_mode[i];
	   } else {
	     numinpass++;
	   }
	 }
       }

       for ( j = 0; j < numComps ; j++ ) {
	 i = page_info.comp_print_order[j];
	 c = page_info.comp_colours[i];
	 /* at this point, we are in data transfer mode, unless this
	    is the first component */
	 if ( changemode[i] 
	   || (c >= 0 && c < colNullSpot && page_info.print_mode[i] != lastprintmode)) {
	   /* changing either the transfer mode or the print mode
	      requires leaving DTM.
	      The documentation says that leaving DTM resets
	      the transfer mode; however, the programming examples show that
	      it doesn't, at least for N-Colour mode */
	   int curlco;
	   if ( i != firstcomp ) {
	     sprintf(buffer,"\033*rC"); /* end raster graphics */
	     page_info.out_function(page_info.comp_fds[i],buffer,4);
	   }
	   /* Change transfer mode if required */
	   /* Note: for some brain-dead reason, the Printiva 
	      (at least) screams about needing the next
	      cartridge *without* unloading the current
	      cartridge from the carriage. You might think
	      that this could be fixed by leaving RGL mode
	      altogether. I tried that, and it makes no
	      difference. So it seems there is no way
	      to fix that. */
	   if ( changemode[i] ) {
	     /* set curl correction on or off. We always set it,
		even though the manual claims that curl
		correction is reset at the end of data transfer
		mode; it makes it surer ;-) and easier to debug.
		A setting of 1 *disables* curl correction */
	     curlco = ( !opt_curlcorrection  
	       && ( i < firstinlast || opt_overlay || opt_nocurl ));
	     sprintf(buffer,"\033\032%c%cC",curlco,0);
	     page_info.out_function(page_info.comp_fds[i],buffer,5);
	     /* if overlay mode was requested, set multipage
		mode again */
	     if ( opt_overlay ) {
	       sprintf(buffer,"\033\032%c%cA",1,0);
	       page_info.out_function(page_info.comp_fds[i],buffer,5);
	     }
	     /* set the transfer mode again */
	     sprintf(buffer,"\033*r%cU",
	       page_info.transfer_mode[i]);
	     page_info.out_function(page_info.comp_fds[i],buffer,5);
	   }
	   /* change the print mode if required */
	   if ( c >= 0 && page_info.print_mode[i] != lastprintmode ) {
	     lastprintmode = page_info.print_mode[i];
	     sprintf(buffer,"\033\032%c%cU",print_mode_bytes[lastprintmode],0);
	     page_info.out_function(page_info.comp_fds[i],buffer,5);
	   }
	   /* and start raster graphics again */
	   sprintf(buffer,"\033*r%cA",0);
	   page_info.out_function(page_info.comp_fds[i],buffer,5);
	 }
	 if ( c >= 0 && c < colNullSpot ) {
	   /* Now send the colour selection command.
	      For the CMYK colours, we only do this if the mode
	      is colourPlane or multiPlane */
	   if ( page_info.transfer_mode[i] == colourPlane
	     || page_info.transfer_mode[i] == multiPlane
	     || i < compC || i > compK ) {
	     /* In the 4-colour page plane mode, the "colour" is the
		zero-based cartridge holder number; otherwise, it's
		a colour byte.
		We happen to know that the colour_to_command array
		is the identity up to 0x16, so this works for cassette numbers 
	     */
	     sprintf(buffer,"\033\032%c%c%c",
	       colour_to_command[page_info.comp_colours[i]],
	       ((i == lastcomp) && lastpass) ? 0x80 : 0 /* this is the last plane flag */,
	       (page_info.transfer_mode[i] == cassettePlane) ?
	       'c' : 'r');
	     page_info.out_function(page_info.comp_fds[i],buffer,5);
	   }
	 }
       }
     }
     
     print_page(&page_info);
     
     /* now, if the MYK components were not printed to stdout,
	we need to copy them to stdout */
     if ( page_info.output_type < ppmOut ) {
       int k;
       for ( k=0; k < numComps; k++ ) {
	 int fd,n,j;
	 char buf[1024];
	 i = page_info.comp_print_order[k];
	 fd = page_info.comp_fds[i];
	 if ( fd < 0 || fd == STDOUT ) continue;
	 /* back feed the paper */
	 sprintf(buffer,"\033\032%c%c\014",0,0);
	 page_info.out_function(STDOUT,buffer,5);
	 lseek(fd,0,SEEK_SET);
	 while ( (n = read(fd,buf,1024)) > 0 ) {
	   write(1,buf,n);
	 }
	 close(fd);
	 /* don't do this one again... */
	 for ( j=i+1; j < numComps; j++ ) {
	   if ( page_info.comp_fds[i] == fd )
	     page_info.comp_fds[i] = -1;
	 }
       }
     }
     
     /* Now we've finished that page. If we have another page
	coming in, then formfeed.
	Note: the effect of combining multipage input with
	options like -overlay is not defined!
     */
     /* However, because PPM reads precisely what it wants,
	we won't have hit end of file even if there is no
	further data. Hence the following ... */
     { int c;
     c = getc(ifd);
     if ( c >= 0 ) { ungetc(c,ifd); }
     }

     if (page_info.output_type < ppmOut && !feof(ifd) ) {
       sprintf(buffer,"\014"); /* formfeed */
       page_info.out_function(STDOUT,buffer,1);
     }
   }

   /* we have printed all the pages, but haven't formfed for
      the last one */
   if (page_info.output_type < ppmOut) {
     /* This is a bit of a mess.
	If we do not have lastpass set, then we should backfeed
	the paper, and stay in data transfer mode, i.e. do
	nothing else. */
     if ( ! lastpass ) {
       sprintf(buffer,"\033\032%c%c\014",0,0);
       page_info.out_function(STDOUT,buffer,5);
     } else {
       /* If the ffonly switch is set, then we will form-feed
	  and stay in data transfer mode. Otherwise, we will
	  exit data transfer mode, and then form-feed,
	  exit RGL mode, and reset the printer */
       if ( !ffonly ) {
	 sprintf(buffer,"\033*rC"); /* end raster graphics */
	 page_info.out_function(STDOUT,buffer,4);
       }
       sprintf(buffer,"\014"); /* formfeed */
       page_info.out_function(STDOUT,buffer,1);
       /* end rgl mode */
       sprintf(buffer,"\033%%%cX",0);
       page_info.out_function(STDOUT,buffer,4);
       /* unless overlay or no reset, do a printer reset */
       if ( !opt_noreset && !opt_overlay ) {
	 sprintf(buffer,"\033e"); /* full printer reset */
	 page_info.out_function(STDOUT,buffer,2);
       }
     }
   }
   exit(0);
}


/* This routine takes a row of 1-bit pixels in byte form,
   packs it to a bit vector, and compresses it.
   The first two arguments are the input pixel row and its length.
   The third (which must be long enough) will contain the uncompressed
   packed vector; the fourth the compressed version, unless it
   is null on entry, in which case no compression is attempted.
   The return value is the length of the uncompressed vector if
   that is shorter than the compressed, or minus the length of
   the compressed vector otherwise.
*/
static int packbits(char *bp, int n, char *outu, char *outc)
{
   int out = 0;
   int num = 0;
   /* workspace */
   static char *runcnt = NULL; 
   static int runcnt_length = 0;
   int i, bitno;

   /* first pack the byte array into bits */
   for ( i = 0, bitno = 0 ; i < n; i++, bitno++,bp++ ) {
     if ( bitno == 0 ) out = 0;
     if ( *bp ) out |= (1 << (7 - bitno));
     if ( bitno == 7 || i == n-1) {
       outu[num++] = out;
       bitno -= 8;
     }
   }
   
   /* now pack the row */
   for (; num > 0 && outu[num-1] == 0; num--); /* remove trailing zeros */
   if (num && outc) {            /* TIFF 4.0 packbits encoding */
     int start = 0;
     int next;

     /* make sure there's enough workspace, without
	reallocating all the time */
     if ( runcnt_length < num+16 ) {
       runcnt = realloc(runcnt,num+128);
     }

     runcnt[start] = 0;
     for (i = 1; i < num; i++) {
       if (outu[i] == outu[i-1]) {
	 if (runcnt[start] <= 0 && runcnt[start] > -127)
	   runcnt[start]--;
	 else
	   runcnt[start = i] = 0;
       } else {
	 if (runcnt[start] >= 0 && runcnt[start] < 127)
	   runcnt[start]++;
	 else
	   runcnt[start = i] = 0;
       }
     }
     start = 0;
     for (i = 0; i < num; i = next) {
       int count = runcnt[i];
       int from = i;
       if (count >= 0) { /* merge two-byte runs */
	 for (;;) {
	   next = i+1+runcnt[i];
	   if(next >= num || runcnt[next] < 0 ||
	      count+runcnt[next]+1 > 127)
	     break;
	   count += runcnt[next]+1;
	   i = next;
	 }
       }
       next =  i + 1 + ((runcnt[i] < 0) ? -runcnt[i] : runcnt[i]);
       /* I do not understand this next section of code.
	  It seems to be responsible for producing spurious lines
	  in the image. */
#if 0
       if (next < num && count > 0 && 
	   /* adding   outu[next-1] == outu[next] &&
	      to the previous line seems to fix it, and seems
	      to make more sense; but I don't see the point. */
	   runcnt[next] < 0 && runcnt[next] > -127) {
	 count--;
	 next--;
	 runcnt[next] = runcnt[next+1]-1;
       }
#endif
       outc[start++] = count;
       assert(count != -128);
       if (count >= 0) {
	 while (count-- >= 0)
	   outc[start++] = outu[from++];
       } else
	 outc[start++] = outu[from];
     }
     if (start < num) {
       return -start;
     } else {
       return num;
     }
   } else {
     return num;
   }
}


/* This routine takes a row of pixels in byte form,
   and packs them into nybbles.
   The first two arguments are the input pixel row and its length.
   The third (which must be long enough) will contain the output vector.
   The return value is the length (in bytes) of the output
   vector.
   If the input pixels are all zero, return zero (so the calling
   layer will just skip the row).
*/
static int packnybbles(unsigned char *bp, int n, unsigned char *outu)
{
  int i, topnyb,allzero;

  allzero = 1;
   /* zero the output buffer*/
   for ( i = 0; i < (n+1)/2; i++ ) outu[i] = 0;
   /* first pack the byte array into bits */
   for ( i = 0, topnyb = 1 ; i < n; i++, topnyb ^= 1 ) {
     if ( topnyb ) outu[i/2] |= (bp[i] & 0xF0);
     else outu[i/2] |= (bp[i] >> 4);
   }
   for ( i = 0; i < (n+1)/2; i++ ) { if ( outu[i] ) allzero = 0; }
   return allzero ? 0 : (n+1)/2;
}


/* This routine handles initialization of the printer.
   It may also be called in the middle to reinitialize,
   so the comp arg says where to print the init sequence. */
static void rgl_init_page(pageInfo *page_info, int fd)
{
  char buffer[80]; /* buffer for printing */

  page_info->out_function(fd,"\033%\200A",4); /* select RGL mode */     /* resolution defaults to 600 dpi */
  sprintf(buffer,"\033*t%cR",page_info->output_resolution);
  page_info->out_function(fd,buffer,6);
  /* media type and fineness code */
  sprintf(buffer,"\033&l%c%cM",
    media_byte1[page_info->media_type],
    media_byte2[page_info->media_type]);
  page_info->out_function(fd,buffer,6);
  sprintf(buffer,"\033&l%c%cA",page_info->paper_size,0); /* select paper size */
  page_info->out_function(fd,buffer,6);
  /* page length, if set */
  if ( opt_pagelength ) {
    sprintf(buffer,"\033&l%c%cP",opt_pagelength%256,opt_pagelength/256);
    page_info->out_function(fd,buffer,6);
  }
  /* page width, if given by user */
  if ( opt_pagewidth ) {
    sprintf(buffer,"\033&a%c%cM",opt_pagewidth%256,opt_pagewidth/256);
    page_info->out_function(fd,buffer,6);
  }
  /* line feed adjustment */
  if ( page_info->lfadj != 0 ) { 
    sprintf(buffer,"\033\032%c%cL",(char) page_info->lfadj,0);
    page_info->out_function(fd,buffer,5);
  }
  /* print head drive signal adjustment */
  if ( opt_phadj != 0 ) { 
    sprintf(buffer,"\033\032%c%cV",(char) opt_phadj,0); 
    page_info->out_function(fd,buffer,5);
  }
  
  /* make ff mean tof in overlay mode */
  if ( opt_overlay ) {
    sprintf(buffer,"\033\032%c%cA",1,0);
    page_info->out_function(fd,buffer,5);
  }
  
  /* on 5000 etc, specify cassettes to be used */
  if ( ! mtest(modelHasMultiRaster,page_info->model) 
    && page_info->transfer_mode[compC] == multiPlane ) {
    int i,j, num;
    sprintf(buffer,"\033&l%c%cC",0,0);
    for ( j = 0, num = 0; j < numComps; j++ ) {
      i = page_info->comp_print_order[j];
      if ( page_info->comp_colours[i] >= 0 
	   && page_info->comp_colours[i] < colNullSpot ) {
	buffer[num+6] = IS_DYESUB(page_info) ? colour_to_barcode_dyesub[page_info->comp_colours[i]] : colour_to_barcode[page_info->comp_colours[i]];
	num++;
      }
    }
    if ( opt_glossy ) {
      buffer[num+6] = IS_DYESUB(page_info) ? colour_to_barcode_dyesub[colDyeSubGlossyFinish] : colour_to_barcode[colGlossyFinish];
      num++;
    }
    buffer[3] = num;
    page_info->out_function(fd,buffer,num+6);
  }
  
  /* now the x and y offsets (if positive; if negative, we have
     to deal with them) */
  if (page_info->x_shift > 0 ) {
    sprintf(buffer,"\033&a%c%cL", page_info->x_shift%256, page_info->x_shift/256);
    page_info->out_function(fd,buffer,6);
  }
  if ( page_info->y_shift > 0 ) {
    sprintf(buffer,"\033&l%c%cE", page_info->y_shift%256,page_info->y_shift/256);
    page_info->out_function(fd,buffer,6);
  }
  
  
  /* if glossy finish, request it ... */
  if ( opt_glossy ) {
    sprintf(buffer,"\033\032%c%cO",0,0);
    page_info->out_function(fd,buffer,5);
  }
  
}

/* This routine generates the print data for a page.
   The argument is a structure containing all the
   mode information.
   When this routine is called, the printer should already be in 
   data transfer mode; this routine will exit leaving the printer
   in data transfer mode, without feeding the paper in either direction.
   The interpretation of the input PPM depends on the control
   information; the components that are generated also depend
   on the control information. When a printer command is generated,
   it is passed to the output via the prn_out routine; this is
   given an argument specified in the control information to
   identify which component it belongs to. Thus a single pass
   through this function can rasterize all four components
   (and in future, other components as well).
*/
static void print_page(pageInfo *page_info)
{
  /* local variables */
  int hrhalf, hrtoggle ; /* variables used in highres halftoning */
  /* we may be calculating at a higher resolution than the in
     or output resolutions. These variables are the factors by which
     the input is expanded. Maximum of MAX_FACTOR = 4 */
  const int MAX_FACTOR = 4;
  int row_factor = 1, col_factor = 1; 
  /* subrow is for higher resolution stuff */
  int  row, subrow, col, subcol, i;
  int pcol;
  int cols;
  int mono;
  int black;
  int vphoto;
  int skipcols;
  int rowcounter;
  pixel *pixels = NULL;
  pixel *spotpixels = NULL;
  /* these will store the output pixels, with subrows.
     This will be malloc to hold col_factor*col entries */
  unsigned char *outbytes[numComps][MAX_FACTOR];
  /* working space to give to packbits */
  unsigned char *outu, *outc, *buffer;
  void (*prn_out)(int,unsigned char *,int);
  int r,g,b,c,m,y,k;
  int col_index;
  int fs_direction = 1;
  int ppmindiag;
  int ppmout;
  int comp;
  int highres;
  int rowstoskip[numComps]; /* stores rows we need to skip */
  int compression[numComps]; /* current compression state */
  int thispixel[numComps]; /* get comp value from input */
  int multiraster;
  long *temperr;
  long *nexterr[numComps];
  long *thiserr[numComps];
  long scaled[numComps], err=0,
    actual[numComps], scaled_actual[numComps];

  prn_out = page_info->out_function;
  hrhalf = hrtoggle = 0 ;
  cols = page_info->in_cols;
  if ( page_info-> x_double ) cols *= 2;
  highres = (page_info->output_resolution == res1200);
  if ( highres ) row_factor = 2;
  if ( page_info->nybble_mode[compC] &&
    (page_info->dither_type == ditherHT 
      || page_info->dither_type == ditherHTcoarse) ) {
    row_factor = col_factor = 1;
  }

  vphoto = page_info->nybble_mode[compC];

  /* quick sanity check: we can't cope with printing different process
     colours in different printing modes */
  for ( i = compC ; i <= compK ; i++ ) {
    if ( page_info->nybble_mode[i] != vphoto ) {
      pm_error("Can't print process colours in mix of vphoto and standard");
    }
  }

  ppmindiag = (page_info->input_type == diagInput);
  ppmout = (page_info->output_type >= ppmOut); /* any form of ppm output */
  /* mono means the input should be treated as greyfile.
     I.e. the input type is monoInput */
  mono = (page_info->input_type == monoInput);
  /* black is true if we are printing in black ink only */
  black = (page_info->transfer_mode[compK] == blackRaster
	   || page_info->transfer_mode[compK] == econoblackRaster);

  /* if we have a negative xshift, we need to skip input columns.
     However, skipcols is in output resolution: we expand to 
     output resolution before we do the skipping */
  skipcols = 0;
  if ( page_info->x_shift < 0 ) { 
    skipcols = -page_info->x_shift;
    /* and this means we're producing less output */
    cols -= skipcols;
  }

  /* handy abbreviation */
#define printing(c) (page_info->comp_colours[c] >= 0)

  /* in multiplane raster modes, various complicated things
     happen */
  multiraster = (page_info->transfer_mode[compC] == colourRaster
		 || page_info->transfer_mode[compC] == cassetteRaster);

  /* allocate pixel row */
  
  pixels = ppm_allocrow(cols+skipcols);
  if ( page_info->spot_file ) spotpixels = ppm_allocrow(cols+skipcols);
  
  /* alloc bitrows for rgb -> cmyk, initialize flags, alloc workspace */
  for ( i = 0; i < numComps; i++ ) {
    for ( subrow = 0; subrow < row_factor; subrow++ )
      outbytes[i][subrow] = (unsigned char *)malloc(cols*MAX_FACTOR);
    if ( page_info->nybble_mode[i] ) {
      compression[i] = 0; // compression not supported
    } else {
      compression[i] = -1 ; /* forces (un)compression to be specified */
    }
    rowstoskip[i] = 0;
    if ((page_info->dither_type == ditherFS)) {
      nexterr[i] = (long *)malloc(sizeof(int)*(cols*MAX_FACTOR+2));
      thiserr[i] = (long *)malloc(sizeof(int)*(cols*MAX_FACTOR+2));
    } else {
      nexterr[i] = 0;
      thiserr[i] = 0;
    }
	
  }
  outu = (unsigned char *)malloc(cols);
  if ( opt_nopack ) outc = NULL;
  else outc = (unsigned char *)malloc(cols);
  /* buffer needs to be long enough to have any printer command */
  buffer = (unsigned char *)malloc(cols+16);
  
  ithresh = ((double)(FS_SCALE)*fthresh);
  idot[compC] = FS_BASIC_SCALE*dot[compC];
  idot[compM] = FS_BASIC_SCALE*dot[compM];
  idot[compY] = FS_BASIC_SCALE*dot[compY];
  idot[compK] = FS_BASIC_SCALE*dot[compK];

  /* Initialize Floyd-Steinberg error vectors. */
  /** inital errors should not be too large */
  if (page_info->dither_type == ditherFS) {
    for ( col = 0; col < cols*MAX_FACTOR + 2; ++col ) {
      for ( i=0; i < numComps; i++ ) {
	thiserr[i][col] = (random( ) % ( FS_SCALE * 2 ) - FS_SCALE)/8;
	/* (random errors in [-1 .. 1]) */
      }
    }
    fs_direction = 1;
  }

  /* build the dither matrices */
  {
    int four[] = { 0, 2, 3, 1 };
    components c;
    int *dith,*dith1;
    if ( page_info->output_resolution == res1200 ) {
      /* "photo-realistic" mode. This is still very much
	 in beta. */
      dith = build_dith(10,(int *)dithmat10,2,four);
      for (c = compC; c <= compK; c++) {
	kht.comps[c].cellsize_c = 20;
	kht.comps[c].dither = dith;
      }
    } else if ( vphoto ) {
#if 0
      dith = build_dith(5,(int *)dithmat5line,2,four);
      for (c = compC; c <= compK; c++) {
	kht.comps[c].cellsize_c = 10;
	kht.comps[c].dither = dith;
      }
#else
      /* Vphoto mode. Still under development */
      /* the x field stores the shear for each sucessive row */
      kht.comps[compC].cellsize_c = 11;
      kht.comps[compC].cellsize_r = 2;
      kht.comps[compC].dither = (int *)tile112;
      kht.comps[compC].x = 8;
      kht.comps[compM].cellsize_c = 7;
      kht.comps[compM].cellsize_r = 3;
      kht.comps[compM].dither = (int *)tile73;
      kht.comps[compM].x = 2;
      kht.comps[compY].cellsize_c = 6;
      kht.comps[compY].cellsize_r = 6;
      kht.comps[compY].dither = (int *)tile66y;
      kht.comps[compY].x = 0;
      kht.comps[compK].cellsize_c = 6;
      kht.comps[compK].cellsize_r = 6;
      kht.comps[compK].dither = (int *)tile66;
      kht.comps[compK].x = 0;
#endif
    } else {
      /* normal 600dpi mode */
      if ( page_info->dither_type == ditherHTcoarse ) {
	/* coarse dot dither */
	dith = dith1 = (int *) dithmat10;
	for (c = compC; c <= compK; c++) {
	  kht.comps[c].cellsize_c = 10;
	  kht.comps[c].dither = (c < compY) ? dith : dith1;
	}
      } else {
	/* fine line dither */
	dith = build_dith(6,(int *)dithmat6line,2,four);
	dith1= build_dith(6,(int *)dithmat6dot,2,four);
	for (c = compC; c <= compK; c++) {
	  kht.comps[c].cellsize_c = 12;
	  kht.comps[c].dither = (c < compY) ? dith : dith1;
	}
      }
    }
  }

  /* if we have a negative yshift, we need to skip rows */
  for ( row = 0 ; row < -page_info->y_shift ; row++ ) {
    ppm_readppmrow(page_info->in_file, pixels, 
		   page_info->in_cols, page_info->in_maxval, 
		   page_info->in_format );
    if ( page_info->spot_file ) {
      ppm_readppmrow(page_info->spot_file, spotpixels, 
	page_info->in_cols, page_info->in_maxval, 
	page_info->in_format );
    }
  }
  
  /* Now we process the rows */

  /* row  is the number of the row in the image file.
     rowcounter is the number of the row relative to the start
     of the print pass.
  */
  for ( /* row is set above */
	 rowcounter = (page_info->y_shift>=0 ? page_info->y_shift : 0);
       row < rows;
       row++,rowcounter++) {
    ppm_readppmrow(page_info->in_file, pixels, 
		   page_info->in_cols, page_info->in_maxval, 
		   page_info->in_format );
    if ( page_info->spot_file ) {
      ppm_readppmrow(page_info->spot_file, spotpixels, 
	page_info->in_cols, page_info->in_maxval, 
	page_info->in_format );
    }

    /* if we've gone past the page length, skip it */
    if ( rowcounter >= opt_pagelength ) {
      static int warned = 0;
      if ( warned++ == 0 ) {
	fprintf(stderr,"image exceeds page length; truncating\n");
      }
      continue;
    }

    if ( page_info->x_double ) {
      /* do the expansion */
      for ( i = page_info->in_cols - 1 ; i >= 0 ; i-- ) 
	pixels[2*i+1] = pixels[2*i] = pixels[i] ;
      if ( page_info->spot_file ) {
	for ( i = page_info->in_cols - 1 ; i >= 0 ; i-- ) 
	  spotpixels[2*i+1] = spotpixels[2*i] = spotpixels[i] ;
      }
    }
    
    /* change maxval if necessary */
    if ( page_info->in_maxval != (int)maxval ) {
      pixel p;
      for ( col = 0; col < cols; col++ ) {
	PPM_DEPTH(p,pixels[col],page_info->in_maxval,maxval);
	pixels[col] = p;
      }
    }

    /* a subrow is used when we are halftoning etc at higher resolution
       that printing */
    for ( subrow = 0; subrow < row_factor; subrow++ ) {
    
      if ((page_info->dither_type == ditherFS))
	for ( col = 0; col < cols + 2; ++col )
	  nexterr[compC][col] = nexterr[compM][col] = nexterr[compY][col] = nexterr[compK][col]= 0;
      
      
      /* if halftoning, initialize the halftone variables */
      if ( page_info->dither_type == ditherHT
	   || page_info->dither_type == ditherHTcoarse ) {
	if ( vphoto ) {
	  vph_ht_init(&kht,compK,row*row_factor+subrow);
	  vph_ht_init(&kht,compC,row*row_factor+subrow);
	  vph_ht_init(&kht,compM,row*row_factor+subrow);
	  vph_ht_init(&kht,compY,row*row_factor+subrow);
	} else {
	  ht_init(&kht,compK,row*row_factor+subrow);
	  ht_init(&kht,compC,row*row_factor+subrow);
	  ht_init(&kht,compM,row*row_factor+subrow);
	  ht_init(&kht,compY,row*row_factor+subrow);
	}
      } 
      /* calculate boolean kb,mb,cb,yb values for the row --
	 then deal with each plane */
      /* for FS this is tedious, owing to boustrophedon */
      /* to make life easier when we have subcols, we'll use
	 pcol as an index, and compute col and subcol from it.
	 This is not at all efficient, but I don't think I care */

      for (
	(!(page_info->dither_type == ditherFS) || fs_direction) ? 
	  (pcol = 0) : (pcol = cols*col_factor-1);
	(!(page_info->dither_type == ditherFS) || fs_direction) ? 
	  (pcol < cols*col_factor) : (pcol >= 0);
	(!(page_info->dither_type == ditherFS) || fs_direction) ?
	  (pcol++) : (pcol--)
	) {
	int isspot = 0, isfoil = 0,j; /* for tracking underlay */
	col = pcol/col_factor; subcol = pcol - col*col_factor;
	/* Here are the input RGB values. These are not changed
	   in the rest of this loop body */
	r = PPM_GETR( pixels[col+skipcols] );
	g = PPM_GETG( pixels[col+skipcols] );
	b = PPM_GETB( pixels[col+skipcols] );
	/* we now calculate cmyk assuming no bgucr, or according
	   to spec in mono mode */
	if ( mono ) {
	  k = maxval - floor(PPM_LUMIN(pixels[col+skipcols])+0.5);
	  /* Yes, this is correct. The spec of mono has changed so
	     that all printed components are given the grey value.
	     Caveat usor!
	  */
	  c = m = y = k;
	} else {
	  k = 0; c = maxval - r ; m = maxval - g ; y = maxval - b ;
	}
	/* if the input is in ppmdiag format, we're supposed to print it
	   as is, so we totally ignore what we've just done ... */
	if ( ppmindiag ) {
	  unsigned int w;
	  /* the input is in ppmdiag format */
	  w = (r & 0x80)
	    | ((r & 0x20) << 1)
	    | ((r & 0x08) << 2)
	    | ((r & 0x02) << 3);
	  w = w | (w >> 4);
	  outbytes[compK][subrow][col] = maxval - w;
	  r = ((r & 0x40) << 1)
	    | ((r & 0x10) << 2)
	    | ((r & 0x04) << 3)
	    | ((r & 0x01) << 4);
	  r = r | (r >> 4);
	  outbytes[compC][subrow][col] = maxval - r;
	  g = ((g & 0x40) << 1)
	    | ((g & 0x10) << 2)
	    | ((g & 0x04) << 3)
	    | ((g & 0x01) << 4);
	  g = g | (g >> 4);
	  outbytes[compM][subrow][col] = maxval - g;
	  b = ((b & 0x40) << 1)
	    | ((b & 0x10) << 2)
	    | ((b & 0x04) << 3)
	    | ((b & 0x01) << 4);
	  b = b | (b >> 4);
	  outbytes[compY][subrow][col] = maxval - b;
	} else {
	  /* do colour correction */
	  switch ( page_info->colcor_type ) {
	  case colcorNone:
	    break;
	  case colcorPlain:
	    if ( ! mono ) {
	      k = c ; if ( m < k ) k = m ; if ( y < k ) k = y ;
	      m -= k ; c -= k ; y -= k ;
	    }
	    break;
	  case colcorPhoto:
	    if ( mono ) {
	      if ( k < 255 ) k = initgamma[k]; /* ensure solid black */
	      c = m = y = k;
	    } else {
	      if ( opt_keepblack && ((c & m & y) == 255) ) {
		c = m = y = 0;
		k = 255;
	      } else {
		/* the colconv table matches the windows driver.
		   We also allow initgamma to brighten photos */
		c = initgamma[c];
		m = initgamma[m];
		y = initgamma[y];
		col_index = ((c & 0xFC) << 12) |
		  ((m & 0xFC) << 6) | (y & 0xFC);
		c = colconv[col_index++];
		m = colconv[col_index++];
		y = colconv[col_index++];
		k = colconv[col_index];
	      }
	    }
	    break;
	  }
	  /* store c m y k in an array for use below */
	  thispixel[compC] = c;
	  thispixel[compM] = m;
	  thispixel[compY] = y;
	  thispixel[compK] = k;
	  
	  /* now calculate the remaining component values */
	  /* spot colours are not subject to halftoning or
	     fake higher resolution, and we only want to do this once */
	  if ( subcol == 0 && subrow == 0 ) {
	    /* r,g,b are still the original input values;
	       introduce local versions, and local 
	       c,m,y,k with no col cor and
	       bgucr as specified in -spotcolour man page */
	    unsigned int c,m,y,k;
	    int sr = r;
	    int sg = g;
	    int sb = b;
	    int comp;
	    if ( page_info->spot_file ) {
	      sr = PPM_GETR( spotpixels[col+skipcols] );
	      sg = PPM_GETG( spotpixels[col+skipcols] );
	      sb = PPM_GETB( spotpixels[col+skipcols] );
	    }
	    c = maxval - sr;
	    m = maxval - sg;
	    y = maxval - sb;
	    k = c ; if ( m < k ) k = m ; if ( y < k ) k = y ;
	    m -= k ; c -= k ; y -= k ;
	    for ( comp = compU1; 
		  comp <= compS4; 
		  (comp == compU4) ? (comp = compS1) : comp++ ) {
	      if ( ! printing(comp) ) continue;
	      switch ( page_info->comp_inputs[comp].type ) {
	      case compInputDefault:
		thispixel[comp] = 0;
		break;
	      case compInputC:
		thispixel[comp] = maxval*(c >= (maxval+1)/2);
		break;
	      case compInputM:
		thispixel[comp] = maxval*(m >= (maxval+1)/2);
		break;
	      case compInputY:
		thispixel[comp] = maxval*(y >= (maxval+1)/2);
		break;
	      case compInputK:
		thispixel[comp] = maxval*(k >= (maxval+1)/2);
		break;
	      case compInputRGBRange:
		thispixel[comp] = maxval *
		  ( sr >= page_info->comp_inputs[comp].r_start
		    && sr <= page_info->comp_inputs[comp].r_end
		    && sg >= page_info->comp_inputs[comp].g_start
		    && sg <= page_info->comp_inputs[comp].g_end
		    && sb >= page_info->comp_inputs[comp].b_start
		    && sb <= page_info->comp_inputs[comp].b_end );
		break;
	      case compInputNotWhite:
		thispixel[comp] = maxval *
		  ( sr < (int)maxval || sg < (int)maxval || sb < (int)maxval );
		break;
	      case compInputAlways:
		thispixel[comp] = maxval;
	      }
	      if ( thispixel[comp] && comp >= compS1 ) {
		switch ( page_info->comp_colours[comp] ) {
		case colDyeSubOvercoat:
		case colDyeSubGlossyFinish:
		case colGlossyFinish:
		  /* nothing to do */
		  break;
		case colGoldFoil:
		case colSilverFoil:
		case colNullFoil:
		  /* need to print CMYK undercoat */
		  isfoil = 1;
		  break;
		default:
		  /* don't print whatever is underneath */
		  isspot = 1;
		}
	      }
	      /* there is no dithering for spot colours, so we just... */
	      outbytes[comp][subrow][col] = thispixel[comp];
	    }
	  }
	  
	  /* now dither it as appropriate */
	  if ( page_info->dither_type == ditherNone ) {
	    int comp;
	    for ( comp = compC; comp <= compK; comp++ )
	      if  ( page_info->nybble_mode[comp] )
		outbytes[comp][subrow][pcol] = thispixel[comp];
	      else
		outbytes[comp][subrow][pcol] = maxval*(thispixel[comp] >= (int)(maxval+1)/2);
	  } else if ( page_info->dither_type == ditherHT
		      || page_info->dither_type == ditherHTcoarse ) {
	    int t;
	    /* Standard halftoning */
	    
	    if ( page_info->output_type == ppmColourDiag ) {
	      /* coldiag, leave unchanged */
	      outbytes[compK][subrow][pcol] = k; outbytes[compC][subrow][pcol] = c; outbytes[compM][subrow][pcol] = m; outbytes[compY][subrow][pcol] = y;
	    } else {
	      /* non standard screen angles to avoid banding on black */
	      if ( printing(compY) ) {
		t = y - ht_elt(&kht,compY);
		outbytes[compY][subrow][pcol] =  (t > (15*page_info->nybble_mode[compY]))? maxval :
		  (t <= 0) ? 0 : t*maxval/16;
		if (vphoto) { vph_ht_inc(&kht,compY); } else ht_inc(&kht,compY);
	      }
	      if (printing(compK)) {
		t = k - ht_elt(&kht,compK);
		outbytes[compK][subrow][pcol] = (t > (15*page_info->nybble_mode[compK])) ? maxval :
		  (t <= 0) ? 0 : t*maxval/16;
		if ( vphoto ) { vph_ht_inc(&kht,compK); } else ht_inc(&kht,compK);
	      }
	      if (printing(compC)) {
		t = c - ht_elt(&kht,compC);
		outbytes[compC][subrow][pcol] = (t > (15*page_info->nybble_mode[compC])) ? maxval :
		  (t <= 0) ? 0 : t*maxval/16;
		if ( vphoto ) { vph_ht_inc(&kht,compC); } else ht_inc(&kht,compC);
	      }
	      if (printing(compM)) {
		t = m - ht_elt(&kht,compM);
		outbytes[compM][subrow][pcol] = (t > (15*page_info->nybble_mode[compM])) ? maxval :
		  (t <= 0) ? 0 : t*maxval/16;
		if (vphoto) { vph_ht_inc(&kht,compM); } else ht_inc(&kht,compM);
	      }
	    }
	  } else if ( (page_info->dither_type == ditherFS) ) {
	    int comp;
	    for ( comp = compC ; comp <= compK ; comp++ ) {
	      if ( printing(comp) ) {
		/* Use Floyd-Steinberg errors to adjust actual color. */
		scaled[comp] = thispixel[comp]*FS_BASIC_SCALE + thiserr[comp][col + 1];
		/* new colours are 1 bit or 4 bits according to vphoto */
		/* actual etc are the pixel values in the output 
                   scaled_actual etc  are the values
		   used in error correction 
		*/
		if ( vphoto ) {
		  int c;
		  /* go from a value in the range 0..255 to a value 
		     in the range 0..15 */
		  c = thispixel[comp]/17;
		  if ( (scaled[comp] - c*17*FS_BASIC_SCALE) >= ithresh ) c++;
		  actual[comp] = c*17;
		} else {
		  actual[comp] = (scaled[comp] >= ithresh)*maxval;
		}
		scaled_actual[comp] = idot[comp]*actual[comp];
		/* Propagate Floyd-Steinberg error terms. */
		if ( fs_direction ) {
		  err = ( scaled[comp] - (long) scaled_actual[comp] );
		  thiserr[comp][col + 2] += ( err * 7 ) / 16;
		  nexterr[comp][col    ] += ( err * 3 ) / 16;
		  nexterr[comp][col + 1] += ( err * 5 ) / 16;
		  nexterr[comp][col + 2] += ( err     ) / 16;
		} else {
		  err = ( scaled[comp] - (long) scaled_actual[comp] );
		  thiserr[comp][col    ] += ( err * 7 ) / 16;
		  nexterr[comp][col + 2] += ( err * 3 ) / 16;
		  nexterr[comp][col + 1] += ( err * 5 ) / 16;
		  nexterr[comp][col    ] += ( err     ) / 16;
		}
		outbytes[comp][subrow][pcol] = actual[comp];
	      }
	    }
	  } else {
	    /* dithering in the square pattern with unrotated screens */
	    if ( highres ) {
	      static int offset ;
	      /* set each halftone cell in a random orientation */
	      if ( row%4 == 0 && col%8 == 0 ) offset = 4*((rand()&0xFF)>>7) ;
	      if (printing(compK)) outbytes[compK][subrow][pcol] = (k > hcluster[row%4][(col+4-offset)%8]);
	      if (printing(compC)) outbytes[compC][subrow][pcol] = (c > hcluster[row%4][(col+offset)%8]);
	      if (printing(compM)) outbytes[compM][subrow][pcol] = (m > hcluster[row%4][(col+offset)%8]);
	      if (printing(compY)) outbytes[compY][subrow][pcol] = (y > hcluster[row%4][(col+offset)%8]);
	    } else {
	      if (printing(compK)) outbytes[compK][subrow][pcol] = (k > cluster8[row%8][(col+4)%8]);
	      if (printing(compC)) outbytes[compC][subrow][pcol] = (c > cluster8[row%8][col%8]);
	      if (printing(compM)) outbytes[compM][subrow][pcol] = (m > cluster8[row%8][col%8]);
	      if (printing(compY)) outbytes[compY][subrow][pcol] = (y > cluster8[row%8][col%8]);
	    }
	  }
	}
	/* if we have a spot colour, don't print anything
	   underneath it (I think this is OK...); however,
	   if it's foil, then print CMYK underneath, since
	   that's required */
	if ( isfoil )
	  for ( j=compC; j <= compK; j++ ) outbytes[j][subrow][pcol] = maxval;
	else if ( isspot )
	  for ( j=compC; j <= compK; j++ ) outbytes[j][subrow][pcol] = 0;
      } /* end of the column for loop */

      if ((page_info->dither_type == ditherFS)) {
	for ( comp = compC; comp <= compK; comp++ ) {
	  temperr = thiserr[comp];
	  thiserr[comp] = nexterr[comp];
	  nexterr[comp] = temperr;
	}
	fs_direction = ! fs_direction;
      }

    } // end of the subrow loop
    /* now we need to convert the subrow*subcol data into a single
       row and column to output, stored in entry outbytes[comp][0][col]
    */
    for ( comp = compC ; comp <= compK ; comp++ ) {
      for ( col = 0; col < cols ; col++ ) {
	unsigned int v = 0;
	for ( subrow = 0; subrow < row_factor; subrow++ ) {
	  for ( subcol = 0; subcol < col_factor; subcol++ ) {
	    v += outbytes[comp][subrow][col*col_factor+subcol];
	  }
	}
	v = v/(row_factor*col_factor);
	/* if we're in nybble mode, that's OK; if we're in bit mode,
	   we now have to flip it to on or off */
	if (! page_info->nybble_mode[comp] && page_info->output_type != ppmColourDiag )
	  v = ((v >= (maxval+1)/2) ? maxval: 0);
	outbytes[comp][0][col] = v;
      }
    }

    /* if ppmout, reassign the pixels */
    if ( ppmout ) {
      for ( col = 0 ;
	    col < cols ; col++ )
	if ( page_info->output_type == ppmColourDiag ) {
	  PPM_ASSIGN(pixels[col],
		     maxval-((printing(compC) ? outbytes[compC][0][col] : 0) + (printing(compK) ? outbytes[compK][0][col]: 0)),
		     maxval-((printing(compM) ? outbytes[compM][0][col] : 0) + (printing(compK) ? outbytes[compK][0][col]: 0)),
		     maxval-((printing(compY) ? outbytes[compY][0][col] : 0) + (printing(compK) ? outbytes[compK][0][col]: 0)));
	} else if ( page_info->output_type == ppmDiag ) {
	  /* new diagnostic output. 4 bits of k are put into the 
	     0,2,4,6 bits of rgb, and 4 bits of c,m,y in the others. */
	  unsigned int r,g,b,w;
	  w = maxval - outbytes[compK][0][col];
	  w = (w & 0x80) 
	    | ((w & 0x40) >> 1)
	    | ((w & 0x20) >> 2)
	    | ((w & 0x10) >> 3);
	  r = maxval - outbytes[compC][0][col];
	  r = ((r & 0x80) >> 1)
	    | ((r & 0x40) >> 2)
	    | ((r & 0x20) >> 3)
	    | ((r & 0x10) >> 4);
	  g = maxval - outbytes[compM][0][col];
	  g = ((g & 0x80) >> 1)
	    | ((g & 0x40) >> 2)
	    | ((g & 0x20) >> 3)
	    | ((g & 0x10) >> 4);
	  b = maxval - outbytes[compY][0][col];
	  b = ((b & 0x80) >> 1)
	    | ((b & 0x40) >> 2)
	    | ((b & 0x20) >> 3)
	    | ((b & 0x10) >> 4);
	  PPM_ASSIGN(pixels[col],r|w,g|w,b|w);
	} else {
	  int r,g,b;
	  r = maxval - (printing(compK)*outbytes[compK][0][col]
	    + printing(compC)*outbytes[compC][0][col]);
	  if ( r < 0 ) r = 0;
	  g = maxval - (printing(compK)*outbytes[compK][0][col]
	    + printing(compM)*outbytes[compM][0][col]);
	  if ( g < 0 ) g = 0;
	  b = maxval - (printing(compK)*outbytes[compK][0][col]
	    + printing(compY)*outbytes[compY][0][col]);
	  if ( b < 0 ) b = 0;
	  PPM_ASSIGN(pixels[col],r,g,b);
	}
    }
     
    if ( ppmout ) {
      prn_out(STDOUT,(char *)(pixels),cols);
    } else {
      int j;
      /* now we need to pack the bytes and pass them to the 
	 output. This is complicated by the need to avoid
	 printing blank rows, and to maintain the compression
	 state. */
      /* It is a fantastically annoying feature of the pre-5000
	 printers that although the actual printing order is
	 CMYK, and so we have the components in that order,
	 in the colourRaster mode the scanlines have to be sent
	 in the order KCMY. So we'll iterate through components
	 but re-arrange them for printing */
      for ( j = 0; j < numComps; j++ ) {
	int n, tag, fd;
	i = j;
	if ( page_info->transfer_mode[compC] == colourRaster 
	  && j >= compC && j <= compK ) {
	  i = (j - compC + 3) % 4 + compC;
	}
	/* are we printing this component? */
	if ( page_info->comp_colours[i] >= 0 
	     && page_info->comp_colours[i] < colNullSpot ) {
	  /* pack the row into outu/outc (uncompress, compressed),
	     or just outu in photo mode */
	  if ( page_info->nybble_mode[i] ) {
	    n = packnybbles(outbytes[i][0],cols,outu);
	  } else {
	    n = packbits(outbytes[i][0],cols,outu,outc);
	  }
	} else {
	  /* if we are in a multiplane raster mode, we
	     need to print this component anyway, even though
	     it's blank. */
	  if ( multiraster && i >= compC && i <= compK ) {
	    n = 0;
	  } else {
	    continue;
	  }
	}
	/* This tag is both the argument to print_out, and
	   the way to maintain the compression state. 
	   In multiraster modes, the CMYK components get
	   mapped on to one -- or if the usemulti option is
	   given, so that we're doing the sensible thing,
	   CMY are mapped on to one, and K is separate.
	   This could be done more cleanly, but I can't be
	   bothered */
	tag = i;
	/* as usual, in -usemulti K is separate */
	if ( multiraster 
	  && i >= compC && i <= (opt_usemulti ? compY : compK) ) tag = compC;
	/* the associated file descriptor */
	fd = page_info->comp_fds[tag];
	/* if there are no bytes to print, we'll just add one to
	   our count of rows to skip; except in multiraster
	   modes, where it's too complicated to be worth while */
	if ( n == 0 && 
	  ! (multiraster 
	    && i >= compC && i <= (opt_usemulti ? compY : compK)) ) {
	  rowstoskip[i]++;
	} else {
	  if ( n >= 0 ) {
	    if ( compression[tag] != 0 ) {
	      /* select uncompressed */
	      sprintf(buffer,"\033*b%c%cM",0,0);
	      prn_out(fd,buffer,6);
	      compression[tag] = 0;
	    }
	  } else {
	    if ( compression[tag] != 1 ) {
	      /* select compressed */
	      sprintf(buffer,"\033*b%c%cM",2,0);
	      prn_out(fd,buffer,6);
	      compression[tag] = 1;
	    }
	  }
	  /* do we have some rows to skip ? */
	  if ( rowstoskip[i] ) {
	    sprintf(buffer,"\033*b%c%cY",rowstoskip[i]%256,rowstoskip[i]/256);
	    prn_out(fd,buffer,6);
	    rowstoskip[i] = 0;
	  }
	  /* if we are in 4-color raster mode, we have to select
	     the cassette for every row transferred. Sigh. */
	  if ( page_info->transfer_mode[i] == cassetteRaster ) {
	    sprintf(buffer,"\033\032%c%cc",page_info->comp_colours[i],0);
	    prn_out(fd,buffer,5);
	  }
	  /* Another horrible hack for -usemulti. We are printing
	     CMY in colour raster, and K in a separate pass; so we
	     have to print a null row for K to the CMY component before
	     we print C */
	  if ( opt_usemulti && i == compC ) {
	    sprintf(buffer,"\033*b%c%cV",0,0);
	    prn_out(fd,buffer,6);
	  }
	  /* create the printer command */
	  sprintf(buffer,"\033*b%c%c%c",
		  (n>=0 ? n : -n)%256,(n>=0 ? n : -n)/256,
		  /* do we advance the raster pointer? 
		     We don't if it's the last row; and we
		     don't if this is not a last component of
		     a multiraster transfer.
		     Note that we use j, not i, here, because
		     we want the last component printed,
		     which is compY in colourRaster */
	    ((row == rows-1)
	      || (rowcounter == opt_pagelength-1)
		   || ((page_info->transfer_mode[i] == colourRaster
			 || page_info->transfer_mode[i] == cassetteRaster)
			 && j >= compC && j < compK)) ? 'V' : 'W');
	  memcpy(buffer+6,n>=0 ? outu : outc,n>=0 ? n : -n);
	  prn_out(fd,buffer,6+(n>=0 ? n : -n));
	}
      }
    } /* if ( ppmout ) else */
  } /* end of row loop */
  /* now free everything we allocated */
  for ( i = 0; i < numComps; i++ ) {
    for ( subrow = 0; subrow < row_factor; subrow++ ) {
      free(outbytes[i][subrow]);
    }
    if ( thiserr[i] ) free(thiserr[i]);
    if ( nexterr[i] ) free(nexterr[i]);
  }
  free(outu);
  free(outc);
  free(buffer);
}
		
/* given array of option entries and an input string, return the value
   for the input string. The input matches a name if it is a case-insensitive
   prefix of the target.
   Return -1 if no target found. return -2 if two or more
   targets match and have different values.
   As a special case, an exact match always succeeds, even if there
   is another option that is an extension. */
static int parse_option(const OptEntry table[], char *input)
{
  int i,n; unsigned int l;

  l = strlen(input);
  for (n=-1, i=0; table[i].name ; i++) {
    if ( strncasecmp(table[i].name,input,l) == 0 ) {
      if ( strlen(table[i].name) == l ) return table[i].value;
      if ( n >= 0 && n != table[i].value ) return -2;
      n = table[i].value;
    }
  }
  return n;
}
			    

/* expand a 4 bit 3D lookup table to 6 bits using trilinear
   interpolation.
   Input: a 16x16x16x4 array, where the 4 values are unsigned
   chars; the 4th component is ignored. Intepreted as
   input[c][m][y] = { c', m', y', k' }
   Output (passed by ref): a 64x64x64x4 array.
*/
static void expand_lut(unsigned char inlut[][16][16][4],
  unsigned char outlut[][64][64][4]) {
  int i,j,k,m,ii,jj,kk;
  int cube[2][2][2][4];
  int res;
  /* note that we actually map 15 to 15<<2 instead of to 15<<2 + 15>>2.
     This is wrong, but I really can't be bothered to do the right thing.
     So anything above 15*4 is the same as 15*4. */
  for ( i=0; i < 16; i++ ) {
    for ( j=0; j < 16; j++ ) {
      for ( k=0; k < 16; k++ ) {
	/* i,j,k are now the origin most corner of the cube.
	   There are 16 points in the cube, at offsets 0..3 */
	/* get the corner points */
	for ( ii=0; ii<2; ii++)
	  for (jj=0; jj<2; jj++)
	    for (kk=0; kk<2; kk++)
	      for (m=0;m<4;m++)
		cube[ii][jj][kk][m] =
		inlut[i+(i==15?0:ii)][j+(j==15?0:jj)][k+(k==15?0:kk)][m];
	/* and now do the points */
	for ( ii=0; ii<4; ii++ ) {
	  for ( jj=0; jj<4; jj++ ) {
	    for (kk=0; kk<4; kk++) {
	      for (m=0; m<4; m++) {
		res = 0
		  + cube[0][0][0][m] * (4-ii)*(4-jj)*(4-kk)
		  + cube[0][0][1][m] * (4-ii)*(4-jj)*(kk)
		  + cube[0][1][0][m] * (4-ii)*(jj)*(4-kk)
		  + cube[0][1][1][m] * (4-ii)*(jj)*(kk)
		  + cube[1][0][0][m] * (ii)*(4-jj)*(4-kk)
		  + cube[1][0][1][m] * (ii)*(4-jj)*(kk)
		  + cube[1][1][0][m] * (ii)*(jj)*(4-kk)
		  + cube[1][1][1][m] * (ii)*(jj)*(kk);
		outlut[i*4+ii][j*4+jj][k*4+kk][m] = res/(4*4*4);
	      }
	    }
	  }
	}
      }
    }
  }
}
