# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['asyncmy', 'asyncmy.constants', 'asyncmy.replication']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'asyncmy',
    'version': '0.2.5',
    'description': 'A fast asyncio MySQL driver',
    'long_description': '# asyncmy - A fast asyncio MySQL/MariaDB driver\n\n[![image](https://img.shields.io/pypi/v/asyncmy.svg?style=flat)](https://pypi.python.org/pypi/asyncmy)\n[![image](https://img.shields.io/github/license/long2ice/asyncmy)](https://github.com/long2ice/asyncmy)\n[![pypi](https://github.com/long2ice/asyncmy/actions/workflows/pypi.yml/badge.svg)](https://github.com/long2ice/asyncmy/actions/workflows/pypi.yml)\n[![ci](https://github.com/long2ice/asyncmy/actions/workflows/ci.yml/badge.svg)](https://github.com/long2ice/asyncmy/actions/workflows/ci.yml)\n\n## Introduction\n\n`asyncmy` is a fast asyncio MySQL/MariaDB driver, which reuse most of [pymysql](https://github.com/PyMySQL/PyMySQL)\nand [aiomysql](https://github.com/aio-libs/aiomysql) but rewrite core protocol with [cython](https://cython.org/) to\nspeedup.\n\n## Features\n\n- API compatible with [aiomysql](https://github.com/aio-libs/aiomysql).\n- Faster by [cython](https://cython.org/).\n- MySQL replication protocol support with `asyncio`.\n- Tested both MySQL and MariaDB in [CI](https://github.com/long2ice/asyncmy/blob/dev/.github/workflows/ci.yml).\n\n## Benchmark\n\nThe result comes from [benchmark](./benchmark).\n\n> The device is iMac Pro(2017) i9 3.6GHz 48G and MySQL version is 8.0.26.\n\n![benchmark](./images/benchmark.png)\n\n### Conclusion\n\n- There is no doubt that `mysqlclient` is the fastest MySQL driver.\n- All kinds of drivers have a small gap except `select`.\n- `asyncio` could enhance `insert`.\n- `asyncmy` performs remarkable when compared to other drivers.\n\n## Install\n\n```shell\npip install asyncmy\n```\n\n### Installing on Windows\n\nTo install asyncmy on Windows, you need to install the tools needed to build it.\n\n1. Download *Microsoft C++ Build Tools* from https://visualstudio.microsoft.com/visual-cpp-build-tools/\n2. Run CMD as Admin (not required but recommended) and navigate to the folder when your installer is downloaded\n3. Installer executable should look like this `vs_buildtools__XXXXXXXXX.XXXXXXXXXX.exe`, it will be easier if you rename\n   it to just `vs_buildtools.exe`\n4. Run this command (Make sure you have about 5-6GB of free storage)\n\n```shell\nvs_buildtools.exe --norestart --passive --downloadThenInstall --includeRecommended --add Microsoft.VisualStudio.Workload.NativeDesktop --add Microsoft.VisualStudio.Workload.VCTools --add Microsoft.VisualStudio.Workload.MSBuildTools\n```\n\n5. Wait until the installation is finished\n6. After installation will finish, restart your computer\n7. Install asyncmy via PIP\n\n```shell\npip install asyncmy\n```\n\nNow you can uninstall previously installed tools.\n\n## Usage\n\n### Use `connect`\n\n`asyncmy` provides a way to connect to MySQL database with simple factory function `asyncmy.connnect()`. Use this\nfunction if you want just one connection to the database, consider connection pool for multiple connections.\n\n```py\nfrom asyncmy import connect\nfrom asyncmy.cursors import DictCursor\nimport asyncio\n\n\nasync def run():\n    conn = await connect()\n    async with conn.cursor(cursor=DictCursor) as cursor:\n        await cursor.execute("create database if not exists test")\n        await cursor.execute(\n            """CREATE TABLE if not exists test.asyncmy\n    (\n        `id`       int primary key auto_increment,\n        `decimal`  decimal(10, 2),\n        `date`     date,\n        `datetime` datetime,\n        `float`    float,\n        `string`   varchar(200),\n        `tinyint`  tinyint\n    )"""\n        )\n\n\nif __name__ == \'__main__\':\n    asyncio.run(run())\n```\n\n### Use `pool`\n\n`asyncmy` provides connection pool as well as plain Connection objects.\n\n```py\nimport asyncmy\nimport asyncio\n\n\nasync def run():\n    pool = await asyncmy.create_pool()\n    async with pool.acquire() as conn:\n        async with conn.cursor() as cursor:\n            await cursor.execute("SELECT 1")\n            ret = await cursor.fetchone()\n            assert ret == (1,)\n\n\nif __name__ == \'__main__\':\n    asyncio.run(run())\n```\n\n## Replication\n\n`asyncmy` supports MySQL replication protocol\nlike [python-mysql-replication](https://github.com/noplay/python-mysql-replication), but powered by `asyncio`.\n\n```py\nfrom asyncmy import connect\nfrom asyncmy.replication import BinLogStream\nimport asyncio\n\n\nasync def run():\n    conn = await connect()\n    ctl_conn = await connect()\n\n    stream = BinLogStream(\n        conn,\n        ctl_conn,\n        1,\n        master_log_file="binlog.000172",\n        master_log_position=2235312,\n        resume_stream=True,\n        blocking=True,\n    )\n    async for event in stream:\n        print(event)\n\n\nif __name__ == \'__main__\':\n    asyncio.run(run())\n```\n\n## ThanksTo\n\n> asyncmy is build on top of these awesome projects.\n\n- [pymysql](https://github/pymysql/PyMySQL), a pure python MySQL client.\n- [aiomysql](https://github.com/aio-libs/aiomysql), a library for accessing a MySQL database from the asyncio.\n- [python-mysql-replication](https://github.com/noplay/python-mysql-replication), pure Python Implementation of MySQL\n  replication protocol build on top of PyMYSQL.\n\n## License\n\nThis project is licensed under the [Apache-2.0](./LICENSE) License.\n',
    'author': 'long2ice',
    'author_email': 'long2ice@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/long2ice/asyncmy',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
