// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { TextItem } from '@jupyterlab/statusbar';
import { nullTranslator } from '@jupyterlab/translation';
import { VDomModel, VDomRenderer } from '@jupyterlab/ui-components';
import { JSONExt } from '@lumino/coreutils';
import React from 'react';
/**
 * Helper function to translate kernel statuses mapping by using
 * input translator.
 *
 * @param translator - Language translator.
 * @return The translated kernel status mapping.
 */
export function translateKernelStatuses(translator) {
    translator = translator || nullTranslator;
    const trans = translator.load('jupyterlab');
    const translated = {
        unknown: trans.__('Unknown'),
        starting: trans.__('Starting'),
        idle: trans.__('Idle'),
        busy: trans.__('Busy'),
        terminating: trans.__('Terminating'),
        restarting: trans.__('Restarting'),
        autorestarting: trans.__('Autorestarting'),
        dead: trans.__('Dead'),
        connected: trans.__('Connected'),
        connecting: trans.__('Connecting'),
        disconnected: trans.__('Disconnected'),
        initializing: trans.__('Initializing'),
        '': ''
    };
    return translated;
}
/**
 * A pure functional component for rendering kernel status.
 */
function KernelStatusComponent(props) {
    const translator = props.translator || nullTranslator;
    const trans = translator.load('jupyterlab');
    let statusText = '';
    if (props.status) {
        statusText = ` | ${props.status}`;
    }
    return (React.createElement(TextItem, { onClick: props.handleClick, source: `${props.kernelName}${statusText}`, title: trans.__('Change kernel for %1', props.activityName) }));
}
/**
 * A VDomRenderer widget for displaying the status of a kernel.
 */
export class KernelStatus extends VDomRenderer {
    /**
     * Construct the kernel status widget.
     */
    constructor(opts, translator) {
        super(new KernelStatus.Model(translator));
        this.translator = translator || nullTranslator;
        this._handleClick = opts.onClick;
        this.addClass('jp-mod-highlighted');
    }
    /**
     * Render the kernel status item.
     */
    render() {
        if (this.model === null) {
            return null;
        }
        else {
            return (React.createElement(KernelStatusComponent, { status: this.model.status, kernelName: this.model.kernelName, activityName: this.model.activityName, handleClick: this._handleClick, translator: this.translator }));
        }
    }
}
/**
 * A namespace for KernelStatus statics.
 */
(function (KernelStatus) {
    /**
     * A VDomModel for the kernel status indicator.
     */
    class Model extends VDomModel {
        constructor(translator) {
            super();
            this._activityName = '';
            this._kernelName = '';
            this._kernelStatus = '';
            this._sessionContext = null;
            translator = translator !== null && translator !== void 0 ? translator : nullTranslator;
            this._trans = translator.load('jupyterlab');
            this._statusNames = translateKernelStatuses(translator);
        }
        /**
         * The name of the kernel.
         */
        get kernelName() {
            return this._kernelName;
        }
        /**
         * The current status of the kernel.
         */
        get status() {
            return this._kernelStatus
                ? this._statusNames[this._kernelStatus]
                : undefined;
        }
        /**
         * A display name for the activity.
         */
        get activityName() {
            return this._activityName;
        }
        set activityName(val) {
            const oldVal = this._activityName;
            if (oldVal === val) {
                return;
            }
            this._activityName = val;
            this.stateChanged.emit();
        }
        /**
         * The current client session associated with the kernel status indicator.
         */
        get sessionContext() {
            return this._sessionContext;
        }
        set sessionContext(sessionContext) {
            var _a, _b, _c, _d;
            (_a = this._sessionContext) === null || _a === void 0 ? void 0 : _a.statusChanged.disconnect(this._onKernelStatusChanged, this);
            (_b = this._sessionContext) === null || _b === void 0 ? void 0 : _b.connectionStatusChanged.disconnect(this._onKernelStatusChanged, this);
            (_c = this._sessionContext) === null || _c === void 0 ? void 0 : _c.kernelChanged.disconnect(this._onKernelChanged, this);
            const oldState = this._getAllState();
            this._sessionContext = sessionContext;
            this._kernelStatus = sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.kernelDisplayStatus;
            this._kernelName =
                (_d = sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.kernelDisplayName) !== null && _d !== void 0 ? _d : this._trans.__('No Kernel');
            sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.statusChanged.connect(this._onKernelStatusChanged, this);
            sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.connectionStatusChanged.connect(this._onKernelStatusChanged, this);
            sessionContext === null || sessionContext === void 0 ? void 0 : sessionContext.kernelChanged.connect(this._onKernelChanged, this);
            this._triggerChange(oldState, this._getAllState());
        }
        /**
         * React to changes to the kernel status.
         */
        _onKernelStatusChanged() {
            var _a;
            this._kernelStatus = (_a = this._sessionContext) === null || _a === void 0 ? void 0 : _a.kernelDisplayStatus;
            this.stateChanged.emit(void 0);
        }
        /**
         * React to changes in the kernel.
         */
        _onKernelChanged(_sessionContext, change) {
            var _a;
            const oldState = this._getAllState();
            // sync setting of status and display name
            this._kernelStatus = (_a = this._sessionContext) === null || _a === void 0 ? void 0 : _a.kernelDisplayStatus;
            this._kernelName = _sessionContext.kernelDisplayName;
            this._triggerChange(oldState, this._getAllState());
        }
        _getAllState() {
            return [this._kernelName, this._kernelStatus, this._activityName];
        }
        _triggerChange(oldState, newState) {
            if (JSONExt.deepEqual(oldState, newState)) {
                this.stateChanged.emit(void 0);
            }
        }
    }
    KernelStatus.Model = Model;
})(KernelStatus || (KernelStatus = {}));
//# sourceMappingURL=kernelstatuses.js.map