import { EditorState, Extension, StateEffect } from '@codemirror/state';
import { EditorView } from '@codemirror/view';
import { ITranslator } from '@jupyterlab/translation';
import { ReadonlyJSONObject } from '@lumino/coreutils';
import { IObservableDisposable } from '@lumino/disposable';
import { ISignal } from '@lumino/signaling';
import { IConfigurableExtension, IEditorExtensionFactory, IEditorExtensionRegistry, IEditorThemeRegistry, IExtensionsHandler } from './token';
/**
 * Editor configuration handler options
 */
export interface IEditorHandlerOptions {
    /**
     * The base configuration options for all editor.
     */
    baseConfiguration?: Record<string, any>;
    /**
     * The configuration options for the editor.
     *
     * They take precendence over the base configuration.
     */
    config?: Record<string, any>;
    /**
     * Editor default extensions.
     *
     * Extension defined in the mapping without a base configuration
     * will not be configurable.
     */
    defaultExtensions?: [string, IConfigurableExtension<any>][];
}
/**
 * Editor configuration handler
 *
 * It stores the editor configuration and the editor extensions.
 * It also allows to inject new extensions into an editor.
 */
export declare class ExtensionsHandler implements IExtensionsHandler, IObservableDisposable {
    constructor({ baseConfiguration, config, defaultExtensions }?: IEditorHandlerOptions);
    /**
     * Signal triggered when the editor configuration changes.
     * It provides the mapping of the new configuration (only those that changed).
     *
     * It should result in a call to `IExtensionsHandler.reconfigureExtensions`.
     */
    get configChanged(): ISignal<this, Record<string, any>>;
    /**
     * A signal emitted when the object is disposed.
     */
    get disposed(): ISignal<this, void>;
    /**
     * Tests whether the object is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources held by the object.
     */
    dispose(): void;
    /**
     * Get a config option for the editor.
     */
    getOption(option: string): unknown;
    /**
     * Whether the option exists or not.
     */
    hasOption(option: string): boolean;
    /**
     * Set a config option for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `IExtensionsHandler.configChanged`.
     */
    setOption(option: string, value: unknown): void;
    /**
     * Set a base config option for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `IExtensionsHandler.configChanged`.
     */
    setBaseOptions(options: Record<string, any>): void;
    /**
     * Set config options for the editor.
     *
     * You will need to reconfigure the editor extensions by listening
     * to `EditorHandler.configChanged`.
     *
     * This method is preferred when setting several options. The
     * options are set within an operation, which only performs
     * the costly update at the end, and not after every option
     * is set.
     */
    setOptions(options: Record<string, any>): void;
    /**
     * Reconfigures the extension mapped with key with the provided value.
     *
     * @param view Editor view
     * @param key Parameter unique key
     * @param value Parameter value to be applied
     */
    reconfigureExtension<T>(view: EditorView, key: string, value: T): void;
    /**
     * Reconfigures all the extensions mapped with the options from the
     * provided partial configuration.
     *
     * @param view Editor view
     * @param configuration Editor configuration
     */
    reconfigureExtensions(view: EditorView, configuration: Record<string, any>): void;
    /**
     * Appends extensions to the top-level configuration of the
     * editor.
     *
     * Injected extension cannot be removed.
     *
     * ### Notes
     * You should prefer registering a IEditorExtensionFactory instead
     * of this feature.
     *
     * @alpha
     * @param view Editor view
     * @param extension Editor extension to inject
     */
    injectExtension(view: EditorView, extension: Extension): void;
    /**
     * Returns the list of initial extensions of an editor
     * based on the configuration.
     *
     * @returns The initial editor extensions
     */
    getInitialExtensions(): Extension[];
    /**
     * Get a extension builder
     * @param key Extension unique identifier
     * @returns The extension builder
     */
    protected get(key: string): IConfigurableExtension<any> | undefined;
    /**
     * Whether the editor has an extension for the identifier.
     *
     * @param key Extension unique identifier
     * @returns Extension existence
     */
    protected has(key: string): boolean;
    protected getEffect<T>(state: EditorState, key: string, value: T): StateEffect<unknown> | null;
    private _getChangedOptions;
    private _baseConfig;
    private _config;
    private _configChanged;
    private _configurableBuilderMap;
    private _disposed;
    private _isDisposed;
    private _immutables;
}
/**
 * CodeMirror extensions registry
 */
export declare class EditorExtensionRegistry implements IEditorExtensionRegistry {
    /**
     * Base editor configuration
     *
     * This is the default configuration optionally modified by the user;
     * e.g. through user settings.
     */
    get baseConfiguration(): Record<string, any>;
    set baseConfiguration(v: Record<string, any>);
    /**
     * Default editor configuration
     *
     * This is the default configuration as defined when extensions
     * are registered.
     */
    get defaultConfiguration(): Record<string, any>;
    /**
     * Editor configuration JSON schema
     */
    get settingsSchema(): ReadonlyJSONObject;
    /**
     * Add a default editor extension
     *
     * @template T Extension parameter type
     * @param factory Extension factory
     */
    addExtension<T>(factory: IEditorExtensionFactory<T>): void;
    /**
     * Create a new extensions handler for an editor
     *
     * @param options Extensions options and initial editor configuration
     */
    createNew(options: IEditorExtensionFactory.IOptions & {
        /**
         * The configuration options for the editor.
         */
        config?: Record<string, any>;
    }): IExtensionsHandler;
    protected configurationBuilder: Map<string, IEditorExtensionFactory<any>>;
    protected configurationSchema: Record<string, any>;
    protected defaultOptions: Record<string, any>;
    protected handlers: Set<ExtensionsHandler>;
    protected immutableExtensions: Set<string>;
    private _baseConfiguration;
}
/**
 * Editor extension registry namespace
 */
export declare namespace EditorExtensionRegistry {
    /**
     * Creates a dynamically configurable editor extension.
     *
     * @param builder Extension builder
     * @return The extension
     */
    function createConfigurableExtension<T>(builder: (value: T) => Extension): IConfigurableExtension<T>;
    /**
     * Creates a configurable extension returning
     * one of two extensions depending on a boolean value.
     *
     * @param truthy Extension to apply when the parameter is true
     * @param falsy Extension to apply when the parameter is false
     * @return The extension
     */
    function createConditionalExtension(truthy: Extension, falsy?: Extension): IConfigurableExtension<boolean>;
    /**
     * Creates an immutable extension.
     *
     * @param extension Immutable extension
     * @return The extension
     */
    function createImmutableExtension(extension: Extension): IConfigurableExtension<undefined>;
    /**
     * Get the default editor extensions.
     *
     * @returns CodeMirror 6 extension factories
     */
    function getDefaultExtensions(options?: {
        themes?: IEditorThemeRegistry;
        translator?: ITranslator | null;
    }): ReadonlyArray<Readonly<IEditorExtensionFactory<any>>>;
}
