/// <reference types="jest" />
import { ISessionContext, SessionContext } from '@jupyterlab/apputils';
import { Kernel, KernelMessage, ServiceManager, Session } from '@jupyterlab/services';
import { AttachedProperty } from '@lumino/properties';
import { ISignal } from '@lumino/signaling';
import { Context } from './context';
import { TextModelFactory } from './default';
import { DocumentRegistry, IDocumentWidget } from './registry';
/**
 * Create a context for a file.
 */
export declare function createFileContext(path?: string, manager?: ServiceManager.IManager): Context<DocumentRegistry.IModel>;
export declare function createFileContextWithKernel(path?: string, manager?: ServiceManager.IManager): Promise<Context>;
export declare function createFileContextWithMockedServices(startKernel?: boolean, manager?: ServiceManager.IManager): Promise<Context>;
/**
 * Create a session and return a session connection.
 */
export declare function createSession(options: Session.ISessionOptions): Promise<Session.ISessionConnection>;
/**
 * Create a session context given a partial session model.
 *
 * @param model The session model to use.
 */
export declare function createSimpleSessionContext(model?: Private.RecursivePartial<Session.IModel>): ISessionContext;
/**
 * Emit an iopub message on a session context.
 *
 * @param sessionContext The session context
 * @param msg Message created with `KernelMessage.createMessage`
 */
export declare function emitIopubMessage(context: ISessionContext, msg: KernelMessage.IIOPubMessage): void;
/**
 * Forceably change the status of a session context.
 * An iopub message is emitted for the change.
 *
 * @param sessionContext The session context of interest.
 * @param newStatus The new kernel status.
 */
export declare function updateKernelStatus(sessionContext: ISessionContext, newStatus: KernelMessage.Status): void;
/**
 * A mock session context.
 *
 * @param session The session connection object to use
 */
export declare const SessionContextMock: jest.Mock<ISessionContext, [Partial<SessionContext.IOptions>, Session.ISessionConnection | null], any>;
/**
 * A namespace for module private data.
 */
declare namespace Private {
    type RecursivePartial<T> = {
        [P in keyof T]?: RecursivePartial<T[P]>;
    };
    const textFactory: TextModelFactory;
    /**
     * Get or create the service manager singleton.
     */
    function getManager(): ServiceManager;
    const lastMessageProperty: AttachedProperty<Kernel.IKernelConnection, string>;
    const RUNNING_KERNELS: Kernel.IKernelConnection[];
}
/**
 * A mock document widget opener.
 */
export declare class DocumentWidgetOpenerMock {
    get opened(): ISignal<DocumentWidgetOpenerMock, IDocumentWidget>;
    open(widget: IDocumentWidget): void;
    private _opened;
}
export {};
