import unittest
import numpy as np

from ..distance_statistics import G, F, J, K, L
from ..pointpattern import PointPattern


class TestDistanceStatistics(unittest.TestCase):
    def setUp(self):
        self.points = [
            [66.22, 32.54],
            [22.52, 22.39],
            [31.01, 81.21],
            [9.47, 31.02],
            [30.78, 60.10],
            [75.21, 58.93],
            [79.26, 7.68],
            [8.23, 39.93],
            [98.73, 77.17],
            [89.78, 42.53],
            [65.19, 92.08],
            [54.46, 8.48],
        ]
        self.pp = PointPattern(self.points)

    def test_distance_statistics_G(self):
        g = G(self.pp, intervals=20)
        distance_domain_sequence = [
            0.0,
            1.73156208378,
            3.46312416757,
            5.19468625135,
            6.92624833514,
            8.65781041892,
            10.3893725027,
            12.1209345865,
            13.8524966703,
            15.5840587541,
            17.3156208378,
            19.0471829216,
            20.7787450054,
            22.5103070892,
            24.241869173,
            25.9734312568,
            27.7049933406,
            29.4365554243,
            31.1681175081,
            32.8996795919,
            34.6312416757,
            36.3628037595,
        ]
        envelop = [
            0.0,
            0.0146894910196,
            0.0574759094992,
            0.124697772575,
            0.210831326089,
            0.309238901911,
            0.413008064625,
            0.515735506438,
            0.612136108644,
            0.698406273989,
            0.772327023751,
            0.83314205264,
            0.881278728773,
            0.917991964321,
            0.945004075083,
            0.964194409454,
            0.977368288342,
            0.986112304319,
            0.991726501147,
            0.995214862073,
            0.997313134422,
            0.998535316975,
        ]
        np.testing.assert_array_almost_equal(g.ev, envelop)
        np.testing.assert_array_almost_equal(g.d, distance_domain_sequence)

    def test_distance_statistics_F(self):
        f = F(self.pp, intervals=20)
        distance_domain_sequence = [
            0.0,
            1.73156208378,
            3.46312416757,
            5.19468625135,
            6.92624833514,
            8.65781041892,
            10.3893725027,
            12.1209345865,
            13.8524966703,
            15.5840587541,
            17.3156208378,
            19.0471829216,
            20.7787450054,
            22.5103070892,
            24.241869173,
            25.9734312568,
            27.7049933406,
            29.4365554243,
            31.1681175081,
            32.8996795919,
            34.6312416757,
            36.3628037595,
        ]
        envelop = [
            0.0,
            0.0146894910196,
            0.0574759094992,
            0.124697772575,
            0.210831326089,
            0.309238901911,
            0.413008064625,
            0.515735506438,
            0.612136108644,
            0.698406273989,
            0.772327023751,
            0.83314205264,
            0.881278728773,
            0.917991964321,
            0.945004075083,
            0.964194409454,
            0.977368288342,
            0.986112304319,
            0.991726501147,
            0.995214862073,
            0.997313134422,
            0.998535316975,
        ]
        np.testing.assert_array_almost_equal(f.ev, envelop)
        np.testing.assert_array_almost_equal(f.d, distance_domain_sequence)

    def test_distance_statistics_J(self):
        j = J(self.pp, intervals=20)
        distance_domain_sequence = [
            0.0,
            1.73156208378,
            3.46312416757,
            5.19468625135,
            6.92624833514,
            8.65781041892,
            10.3893725027,
            12.1209345865,
            13.8524966703,
            15.5840587541,
            17.3156208378,
            19.0471829216,
            20.7787450054,
            22.5103070892,
        ]
        for val in j.ev:
            np.testing.assert_approx_equal(val, 1.0)
        for i in range(min(len(j.d), len(distance_domain_sequence))):
            np.testing.assert_approx_equal(j.d[i], distance_domain_sequence[i])

    def test_distance_statistics_K(self):
        k = K(self.pp, intervals=20)
        distance_domain_sequence = [
            0.0,
            1.055,
            2.11,
            3.165,
            4.22,
            5.275,
            6.33,
            7.385,
            8.44,
            9.495,
            10.55,
            11.605,
            12.66,
            13.715,
            14.77,
            15.825,
            16.88,
            17.935,
            18.99,
            20.045,
            21.1,
            22.155,
        ]

        envelope = [
            0.0,
            3.49667116,
            13.98668465,
            31.47004047,
            55.94673861,
            87.41677908,
            125.88016188,
            171.336887,
            223.78695445,
            283.23036422,
            349.66711633,
            423.09721075,
            503.52064751,
            590.93742659,
            685.347548,
            786.75101173,
            895.1478178,
            1010.53796618,
            1132.9214569,
            1262.29828994,
            1398.6684653,
            1542.031983,
        ]

        np.testing.assert_array_almost_equal(k.ev, envelope)
        np.testing.assert_array_almost_equal(k.d, distance_domain_sequence)

    def test_distance_statistics_L(self):
        l = L(self.pp, intervals=20)
        distance_domain_sequence = [
            0.0,
            1.055,
            2.11,
            3.165,
            4.22,
            5.275,
            6.33,
            7.385,
            8.44,
            9.495,
            10.55,
            11.605,
            12.66,
            13.715,
            14.77,
            15.825,
            16.88,
            17.935,
            18.99,
            20.045,
            21.1,
            22.155,
        ]
        l_stats = [
            0.0,
            -1.055,
            -2.11,
            -3.165,
            -4.22,
            -5.275,
            -6.33,
            -7.385,
            -8.44,
            -3.42556019,
            -4.48056019,
            -5.53556019,
            -6.59056019,
            -7.64556019,
            -8.70056019,
            -7.24151591,
            -8.29651591,
            -9.35151591,
            -10.40651591,
            -11.46151591,
            -12.51651591,
            -10.01612038,
        ]

        np.testing.assert_array_almost_equal(l.d, distance_domain_sequence)
        np.testing.assert_array_almost_equal(l.l, l_stats)
