# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['questionary', 'questionary.prompts']

package_data = \
{'': ['*']}

install_requires = \
['prompt_toolkit>=2.0,<4.0']

setup_kwargs = {
    'name': 'questionary',
    'version': '1.8.0',
    'description': 'Python library to build pretty command line user prompts ⭐️',
    'long_description': '# questionary\n\n[![version](https://img.shields.io/pypi/v/questionary.svg)](https://pypi.org/project/questionary/)\n[![license](https://img.shields.io/pypi/l/questionary.svg)](https://pypi.org/project/questionary/)\n[![Build Status](https://travis-ci.com/tmbo/questionary.svg?branch=master)](https://travis-ci.com/tmbo/questionary)\n[![Coverage Status](https://coveralls.io/repos/github/tmbo/questionary/badge.svg?branch=master)](https://coveralls.io/github/tmbo/questionary?branch=master)\n[![Supported Python Versions](https://img.shields.io/pypi/pyversions/questionary.svg)](https://pypi.python.org/pypi/questionary)\n[![FOSSA Status](https://app.fossa.io/api/projects/git%2Bgithub.com%2Ftmbo%2Fquestionary.svg?type=shield)](https://app.fossa.io/projects/git%2Bgithub.com%2Ftmbo%2Fquestionary?ref=badge_shield)\n\n✨An easy to use python library to build pretty command line user prompts ✨\n\n![example-gif](docs/images/example.gif)\n\nYou need input from a user, e.g. how an output file should be named or if he really \nwants to execute that dangerous operation? This library will help you make the \ninput prompts easy to read and answer for the user.\n\nUsed and Supported by:\n\n[<img src="https://rasa.com/static/686aee8071dd209f198d500b1164e350/4828e/rasa.png" width="100">](https://github.com/RasaHQ/rasa)\n\n## Quickstart\n\n`questionary` can be installed using `pip`:\n\n```bash\n$ pip install questionary\n✨🎂✨\n```\n\nSatisfaction guaranteed. Let\'s create a first question:\n\n```python\nimport questionary\n\nquestionary.select(\n    "What do you want to do?",\n    choices=[\n        \'Order a pizza\',\n        \'Make a reservation\',\n        \'Ask for opening hours\'\n    ]).ask()  # returns value of selection\n```\n\nThat\'s all it takes to create a user prompt! There are different types of prompts, \nyou\'ll find examples for all of them further down.\n\n<details>\n<summary>Alternative: Building from source</summary>\n\nquestionary uses Poetry for packaging and dependency management. If you want \nto build it from source, you have to install Poetry first. \n\nThis is how it can be done:\n\n```\ncurl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python3\n```\n\nThere are several other ways to install Poetry. Please, follow \n[the official guide](https://python-poetry.org/docs/#installation) to see all possible options.\n\nTo install dependencies and `questionary` itself in editable mode execute\n```\nmake install\n```\n</details>\n\n## Documentation\n\n### Different question types\n\n<details><summary>text</summary>\n\n   A free text input for the user.\n\n   ```python\n   questionary.text("What\'s your first name").ask()\n   ```\n   ![example-gif](docs/images/text.gif)\n\n</details>\n<details><summary>password</summary>\n\n   A free text input for the user where the input is not\n   shown but replaced with `***`.\n\n   ```python\n   questionary.password("What\'s your secret?").ask()\n   ```\n\n   ![example-gif](docs/images/password.gif)\n\n</details>\n<details><summary>path</summary>\n\n   A text input for a file or directory path with autocompletion enabled.\n\n   ```python\n   questionary.path("Path to the projects version file").ask()\n   ```\n\n   ![example-gif](docs/images/path.gif)\n\n</details>\n<details><summary>confirm</summary>\n\n   A yes or no question. The user can either confirm or deny.\n\n   ```python\n   questionary.confirm("Are you amazed?").ask()\n   ```\n\n   ![example-gif](docs/images/confirm.gif)\n\n</details>\n<details><summary>select</summary>\n\n   A list of items to select a choice from. The user can pick\n   one option and confirm it.\n\n   ```python\n   questionary.select(\n       "What do you want to do?",\n       choices=[\n           "Order a pizza",\n           "Make a reservation",\n           "Ask for opening hours"\n       ]).ask()\n   ```\n\n   ![example-gif](docs/images/select.gif)\n\n</details>\n<details><summary>rawselect</summary>\n\n   A list of items to select a choice from. The user can pick\n   one option using shortcuts and confirm it.\n\n   ```python\n   questionary.rawselect(\n       "What do you want to do?",\n       choices=[\n           "Order a pizza",\n           "Make a reservation",\n           "Ask for opening hours"\n       ]).ask()\n   ```\n\n   ![example-gif](docs/images/rawselect.gif)\n\n</details>\n\n<details><summary>checkbox</summary>\n\n   A list of items to select multiple choices from. The user can pick\n   none, one or multiple options and confirm the selection.\n\n   ```python\n   questionary.checkbox(\n       \'Select toppings\',\n       choices=[\n           "foo",\n           "bar",\n           "bazz"\n       ]).ask()\n   ```\n   ![example-gif](docs/images/checkbox.gif)\n\n</details>\n\n<details><summary>autocomplete</summary>\n\n   Text input with autocomplete help.\n\n   ```python\n   questionary.autocomplete(\n       \'Choose ant specie\',\n       choices=[\n            \'Camponotus pennsylvanicus\',\n            \'Linepithema humile\',\n            \'Eciton burchellii\',\n            "Atta colombica",\n            \'Polyergus lucidus\',\n            \'Polyergus rufescens\',\n       ]).ask()\n   ```\n   ![example-gif](docs/images/autocomplete.gif)\n\n</details>\n\n### Additional Features\n<details><summary>Printing formatted text</summary>\n\nSometimes you want to spice up your printed messages a bit, `questionary.print`\nis a helper to do just that:\n\n```python\n\nquestionary.print("Hello World 🦄", style="bold italic fg:darkred")\n```\n![example-gif](docs/images/print.gif) \n\nThe style argument uses the prompt [toolkit style strings](https://python-prompt-toolkit.readthedocs.io/en/master/pages/advanced_topics/styling.html#style-strings).\n</details>\n\n<details><summary>Skipping questions using conditions</summary>\n\nSometimes it is helpful to e.g. provide a command line flag to your app\nto skip any prompts, to avoid the need for an if around any question you\ncan pass that flag when you create the question:\n\n```python\nDISABLED = True\n\nresponse = questionary.confirm("Are you amazed?").skip_if(DISABLED, default=True).ask()\n```\n\nIf the condition (in this case `DISABLED`) is `True`, the question will be\nskipped and the default value gets returned, otherwise the user will be\nprompted as usual and the default value will be ignored.\n</details>\n\n<details><summary>Alternative style to create questions using a configuration dictionary</summary>\n\nInstead of creating questions using the python functions, you can also create them using a configuration dictionary.\n```python\nquestions = [\n    {\n        \'type\': \'text\',\n        \'name\': \'phone\',\n        \'message\': "What\'s your phone number",\n    },\n    {\n        \'type\': \'confirm\',\n        \'message\': \'Do you want to continue?\',\n        \'name\': \'continue\',\n        \'default\': True,\n    }\n]\n\nanswers = prompt(questions)\n```\n\nThe returned `answers` will be a dict containing the responses, e.g. `{"phone": "0123123", "continue": False, ""}`. The questions will be prompted one after another and `prompt` will return once all of them are answered.\n\nEach configuration dictionary needs to contain the following keys:\n\n* `\'type\'` - The type of the question.\n* `\'name\'` - The name of the question (will be used as key in the `answers` dictionary)\n* `\'message\'` - Message that will be shown to the user\n\nOptional Keys:\n\n* `\'qmark\'` - Question mark to use - defaults to `?`.\n* `\'default\'` - Preselected value.\n* `\'choices\'` - List of choices (applies when `\'type\': \'select\'`) or function returning a list of choices.\n* `\'when\'` - Function checking if this question should be shown or skipped (same functionality than `.skip_if()`).\n* `\'validate\'` - Function or Validator Class performing validation (will be performed in real time as users type).\n* `filter` - Receive the user input and return the filtered value to be used inside the program. \n\n</details>\n\n<details><summary>Advanced workflow examples</summary>\nQuestionary allows creating quite complex workflows when combining all of the above concepts.\n\n``` python\nfrom questionary import Separator, prompt\nquestions = [\n    {\n        \'type\': \'confirm\',\n        \'name\': \'conditional_step\',\n        \'message\': \'Would you like the next question?\',\n        \'default\': True,\n    },\n    {\n        \'type\': \'text\',\n        \'name\': \'next_question\',\n        \'message\': \'Name this library?\',\n        # Validate if the first question was answered with yes or no\n        \'when\': lambda x: x[\'conditional_step\'],\n        # Only accept questionary as answer\n        \'validate\': lambda val: val == \'questionary\'\n    },\n    {\n        \'type\': \'select\',\n        \'name\': \'second_question\',\n        \'message\': \'Select item\',\n        \'choices\': [\n            \'item1\',\n            \'item2\',\n            Separator(),\n            \'other\',\n        ],\n    },\n    {\n        \'type\': \'text\',\n        \'name\': \'second_question\',\n        \'message\': \'Insert free text\',\n        \'when\': lambda x: x[\'second_question\'] == \'other\'\n    },\n]\nprompt(questions)\n```\n\nThe above workflow will show to the user as follows:\n1. Yes/No question `Would you like the next question?`.\n2. `Name this library?` - only shown when the first question is answered with yes\n3. A question to select an item from a list.\n4. Free text inpt if `\'other\'` is selected in step 3.\n\nDepending on the route the user took, the result will look as follows:\n\n``` python\n{ \n    \'conditional_step\': False,\n    \'second_question\': \'Testinput\'   # Free form text\n}\n```\n``` python\n{ \n    \'conditional_step\': True,\n    \'next_question\': \'questionary\',\n    \'second_question\': \'Testinput\'   # Free form text\n}\n```\n\nYou can test this workflow yourself by running the [advanced_workflow.py example](https://github.com/tmbo/questionary/blob/master/examples/advanced_workflow.py).\n\n</details>\n\n<details><summary>Styling your prompts with your favorite colors</summary>\n\nYou can customize all the colors used for the prompts. Every part of the prompt has an identifier, which you can use to style it. Let\'s create our own custom style:\n```python\nfrom prompt_toolkit.styles import Style\n\ncustom_style_fancy = Style([\n    (\'qmark\', \'fg:#673ab7 bold\'),       # token in front of the question\n    (\'question\', \'bold\'),               # question text\n    (\'answer\', \'fg:#f44336 bold\'),      # submitted answer text behind the question\n    (\'pointer\', \'fg:#673ab7 bold\'),     # pointer used in select and checkbox prompts\n    (\'highlighted\', \'fg:#673ab7 bold\'), # pointed-at choice in select and checkbox prompts\n    (\'selected\', \'fg:#cc5454\'),         # style for a selected item of a checkbox\n    (\'separator\', \'fg:#cc5454\'),        # separator in lists\n    (\'instruction\', \'\'),                # user instructions for select, rawselect, checkbox\n    (\'text\', \'\'),                       # plain text\n    (\'disabled\', \'fg:#858585 italic\')   # disabled choices for select and checkbox prompts\n])\n```\n\nTo use our custom style, we need to pass it to the question type:\n```python\nquestionary.text("What\'s your phone number", style=custom_style_fancy).ask()\n```\n\nIt is also possible to use a list of token tuples as a `Choice` title. This\nexample assumes there is a style token named `bold` in the custom style you are\nusing:\n```python\nChoice(\n    title=[\n        (\'class:text\', \'plain text \'),\n        (\'class:bold\', \'bold text\')\n    ]\n)\n```\nAs you can see it is possible to use custom style tokens for this purpose as\nwell. Note that Choices with token tuple titles will not be styled by the\n`selected` or `highlighted` tokens. If not provided, the `value` of the Choice\nwill be the text concatenated (`\'plain text bold text\'` in the above example).\n</details>\n\n## How to Contribute\n\nContributions are highly welcomed and appreciated. Every little help counts, \nso do not hesitate!\n\n1.  Check for open issues or open a fresh issue to start a discussion\n    around a feature idea or a bug. There is a [Contributor\n    Friendly](https://github.com/tmbo/questionary/issues?direction=desc&labels=good+first+issue&page=1&sort=updated&state=open)\n    tag for issues that should be ideal for people who are not very\n    familiar with the codebase yet.\n2.  Fork [the repository](https://github.com/tmbo/questionary) on\n    GitHub to start making your changes to the **master** branch (or\n    branch off of it).\n3.  Write a test which shows that the bug was fixed or that the feature\n    works as expected.\n4.  Ensure your code passes running `black questionary`.\n5.  Send a pull request and bug the maintainer until it gets merged and\n    published. 🙂\n\n## Contributors\n\n`questionary` is written and maintained by Tom Bocklisch.\n\nIt is based on the great work of [Oyetoke Toby](https://github.com/CITGuru/PyInquirer) \nas well as the work from [Mark Fink](https://github.com/finklabs/whaaaaat).\n\n## Changelog\n\n<details><summary>1.8.0 (08.11.2020)</summary>\n\n* Added additional question type `questionary.path`\n* Added possibility to validate select and checkboxes selections before submitting them\n* Added a helper to print formatted text `questionary.print`\n* Added API method to call prompt in an unsafe way\n* Hide cursor on select only showing the item marker\n\n</details>\n\n<details><summary>1.7.0 (15.10.2020)</summary>\n\n* Added support for python 3.9\n* Better UX for multiline text input\n* Allow passing custom lexer\n\n</details>\n\n<details><summary>1.6.0 (04.10.2020)</summary>\n\n* Updated black code style formatting and fixed version.\n* Fixed colour of answer for some prompts.\n* Added `py.typed` marker file.\n* Documented multiline input for devs and users and added tests.\n* Accept style tuples in `title` argument annotation of `Choice`.\n* Added `default` for select and `initial_choice` for checkbox prompts.\n* Removed check for choices if completer is present.\n\n</details>\n\n<details><summary>1.5.2 (16.04.2020)</summary>\n\nBug fix release.\n\n* Added `.ask_async` support for forms.\n</details>\n\n<details><summary>1.5.1 (22.01.2020)</summary>\n\nBug fix release.\n\n* Fixed `.ask_async` for questions on `prompt_toolkit==2.*`. Added tests for it.\n</details>\n\n<details><summary>1.5.0 (22.01.2020)</summary>\n\nFeature release.\n\n* Added support for prompt_toolkit 3\n* All tests will be run against prompt_toolkit 2 and 3\n* Removed support for python 3.5 (prompt_toolkit 3 does not support that anymore)\n</details>\n\n<details><summary>1.4.0 (10.11.2019)</summary>\n\nFeature release.\n\n* Added additional question type `autocomplete`\n* Allow pointer and highlight in select question type\n</details>\n\n<details><summary>1.3.0 (25.08.2019)</summary>\n\nFeature release.\n\n* Add additional options to style checkboxes and select prompts https://github.com/tmbo/questionary/pull/14\n\n</details>\n\n<details><summary>1.2.1 (19.08.2019)</summary>\n\nBug fix release.\n\n* Fixed compatibility with python 3.5.2 by removing `Type` annotation (this time for real)\n</details>\n\n<details><summary>1.2.0 (30.07.2019)</summary>\n\nFeature release.\n\n* Allow a user to pass in a validator as an instance https://github.com/tmbo/questionary/pull/10\n\n</details>\n\n<details><summary>1.1.1 (21.04.2019)</summary>\n\nBug fix release.\n\n* Fixed compatibility with python 3.5.2 by removing `Type` annotation\n\n</details>\n\n<details><summary>1.1.0 (10.03.2019)</summary>\n\nFeature release.\n\n* Added `skip_if` to questions to allow skipping questions using a flag\n\n\n</details>\n\n<details><summary>1.0.2 (23.01.2019)</summary>\n\nBug fix release.\n\n* Fixed odd behaviour if select is created without providing any choices\n  instead, we will raise a `ValueError` now. ([#6](https://github.com/tmbo/questionary/pull/6))\n\n\n</details>\n\n<details><summary>1.0.1 (12.01.2019)</summary>\n\nBug fix release, adding some convenience shortcuts.\n\n* Added shortcut keys `j` (move down^ the list) and `k` (move up) to\n  the prompts `select` and `checkbox` (fixes [#2](https://github.com/tmbo/questionary/issues/2))\n* Fixed unclosed file handle in `setup.py`\n* Fixed unnecessary empty lines moving selections to far down (fixes [#3](https://github.com/tmbo/questionary/issues/3))\n\n</details>\n\n<details><summary>1.0.0 (14.12.2018)</summary>\n\nInitial public release of the library\n\n* Added python interface\n* Added dict style question creation\n* Improved the documentation\n* More tests and automatic travis test execution\n</details>\n\n## Developer Info\n\n<details>\n<summary>Create one of the commandline recordings in the readme</summary>\n\n0. Install `brew install asciinema` and `npm install --global asciicast2gif`\n1. Run `asciinema rec`\n2. Do the thing\n3. Convert to giv `asciicast2gif -h 7 -w 120 -s 2 <recoding> output.gif`\n\n</details>\n\n<details>\n<summary>Cutting a new release</summary>\n\n0. Update the version number in `questionary/version.py` AND `pyproject.toml`\n1. Add a new section for the release in the changelog in this readme\n2. commit this changes\n3. git tag the commit with the release version \n\nTravis will build and push the updated library to pypi.\n\n</details>\n\n## License\nLicensed under the MIT License. Copyright 2020 Tom Bocklisch. [Copy of the license](LICENSE).\n\n\n[![FOSSA Status](https://app.fossa.io/api/projects/git%2Bgithub.com%2Ftmbo%2Fquestionary.svg?type=large)](https://app.fossa.io/projects/git%2Bgithub.com%2Ftmbo%2Fquestionary?ref=badge_large)\n',
    'author': 'Tom Bocklisch',
    'author_email': 'tombocklisch@gmail.com',
    'maintainer': 'Tom Bocklisch',
    'maintainer_email': 'tombocklisch@gmail.com',
    'url': 'https://github.com/tmbo/questionary',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<3.10',
}


setup(**setup_kwargs)
