//  Copyright Takuma Yoshimura 2024.
//  Copyright Matt Borland 2024.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifdef __clang__
#  pragma clang diagnostic push 
#  pragma clang diagnostic ignored "-Wliteral-range"
#elif defined(__GNUC__)
#  pragma GCC diagnostic push 
#  pragma GCC diagnostic ignored "-Woverflow"
#endif

#define BOOST_TEST_MAIN
#define BOOST_TEST_MODULE StatsMapAiryTest
#include <boost/math/tools/config.hpp>
#include <boost/test/included/unit_test.hpp>
#include <boost/test/tools/floating_point_comparison.hpp>

#include <boost/math/distributions/mapairy.hpp>

#if __has_include(<stdfloat>)
# include <stdfloat>
#endif

using boost::math::mapairy_distribution;

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
#include <boost/multiprecision/cpp_bin_float.hpp>
using boost::multiprecision::cpp_bin_float_quad;
#endif

template<class RealType, int N>
void do_test_mapairy_pdf(){
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    // Left tail of MapAiry distribution inherently limits accuracy due to the rapid decay of the function value.
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.48902525259161778456663373601227358266560067430936e-4524), tolerance * 10000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.35000261887613150163386664453709286782071190547944e-3558), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.59514743738055878289124765721120558977841779893312e-2741), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.23001358749520957481536587430803343649600724408224e-2059), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.91092696891623714443869745182216876382044809172723e-1501), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07670398427978297595825050147335321586451776148320e-1054), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.75337241594382145623417946757631570270349308241803e-869), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.98271669611043473474221732621841457710581601420544e-707), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.16498740545788718692214006589654128484512724342869e-566), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.49681712247552092430360122510975673204884714780707e-445), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.55053727524864779623317373572781282350111182284876e-343), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.19590456107919919197512531538035620765530980528894e-258), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.73694941159984922185033097915349420991674156548250e-188), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.81449463353684665334966063303815524281822160159818e-132), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.74974309038596237884318081144949974855326622002492e-109), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.29136035353058491211769990284207292488489027379633e-89), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01585226509378718416615473646349537501789989363229e-71), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37033445628502141399361852883285362383866808742814e-56), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34096664947790338152990789779350354526301978444227e-43), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68877771535989487461767283800177475346958322976577e-33), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.82006696453724258217996541433935771573414304428384e-24), tolerance * 10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54482240843563125964619012431713490038548382437138e-17), tolerance * 10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95446089249867191439127373027848930673014745742088e-14), tolerance * 10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.50933920988971071711086365057147867083089476835078e-12), tolerance * 10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.93525639583557913222186453754518161964863338679333e-10), tolerance * 10);

    #ifndef BOOST_MATH_HAS_GPU_SUPPORT
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.34369873917936310843493426518284206307542912325702e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.77757355685380561185988935905104213909281485602180e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.67935873875498302175093073530783568485155988744416e-5), tolerance);
    #else
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.34369873917936310843493426518284206307542912325702e-8), tolerance*10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.77757355685380561185988935905104213909281485602180e-6), tolerance*10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.67935873875498302175093073530783568485155988744416e-5), tolerance*10);
    #endif
    
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63150314678587224494834533488274597847689803128076e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.66981984951457236403520646020207894651500649740558e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04268591430185285447948956767040247962753669747157e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.09373105986325625209096934627967360395341037677581e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.80640911685196781581446750934825559197579090806069e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.30714423198107251239523977966407204142168461729995e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.58842018103237928265884912263898874481595562824878e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34619157601942089113215976120224270739528013289579e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.75682912797714827638251175725141830254079472660215e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.14483832832989822788477500521594411868042501441052e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31592413389237158115773602554741285666601187136132e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46510720433737601145589738058516645303764778406120e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.58858272589879457467907897007761708777068403816847e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.68368758744508533942159519264649183356982627934537e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.74896964572402100481706929301679979523752439534115e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.78416838013401425639828509948809230256694219667844e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79011912859672711819203848701778087056321687064703e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76859868856746781256050397658493368372419755176282e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.74830251162716136956231178987638601629634664517919e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72213275423897601118140737012980725371274046892426e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69048759814952957802002736887143054018790117979041e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65378566438926186069214998647211849395648980215839e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.61245973171416933849635378258875498386938909727468e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56695079986621470612303095505427278554320952987612e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51770257100814033767420472605045312962380868746169e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46515640532722451246144916562599607636382596292905e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.40974679016952551097229293394786875981693451661862e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.35189734654776422654774161240658760813142609084491e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.29201738275575132039254286232840244115188891902544e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.23049899232453590038819913858072064402640465533782e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16771468279209758359211730425922824067563080471965e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.10401551277763824950435892543895916698505857289016e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.03972970763820126963564357339894388732551496850922e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97516171847191855609649452292217911972760948598791e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.91059168530983397246791731314734572731339598377834e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.84627526286757244484016179988853282946752058004097e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78244376604908808362842960720569088074360938826161e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.71930459233559687354598878881397561629142415087483e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65704187907786028551352925788813828181432590627815e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.59581735536596915694662484173754591721771767208970e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.53577135041201613422320904399972678346505180911383e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.47702392309337834830220258805531647259773671641587e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41967608032785834866384133800904772318199442922830e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36381105516273814949846883397888219175600292125024e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30949561875080790965818583863955676147392613436720e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.25678140366749341891056123744605385205177726623077e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.20570621922063209990039201192595153753061823668443e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.15629534246005614282003528019134390685312636509026e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10856277146400733535248998121026984388464547942987e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06251243013238748252181151646220197947016365256337e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.75430585815861237530788049942964394941727146945119e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.94921730852144630624722291210866171228958719672719e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.20763203888240939750507494683136041919542710442553e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.52664687715163418854003417614711690443203995990429e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.90290701026305425949257079944431253582616228954748e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.33278922127799799766017618158404889945497518669482e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.81254742548053622180113078981721846479230408163441e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.33842514891989443409465171800884519330814876941244e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.51394330031190703409753192401513286012367671189008e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.83173172055121251708819783948372658040389943407125e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.26731989005960717430587605328985374107046652534330e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79973178629260510506067822703511722927370913396772e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.41138621807489585364575132430814179891782893106309e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.08776953835588218788258294328943072921113243548837e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.81702027546943420543987006667270192025143215867522e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.58950538583133457383574346194006716984204842764692e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23449076237421124835496017947109115185689559096654e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.76628876438498186161275834549825153942245265285021e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.85705896089536983465980519032147649251164182381151e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.41725094843861993121315653435763289609915116903808e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.31264184292047732621870128909412110588268320306673e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.45166594303360764259552410935368398129332569193241e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.77078907451534652348813394683088320326083698637357e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22517551525042172427941302520759668293351400085609e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.41981613066309674421788083490352250077859372362907e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.86815272152296425461960094149288322349331776254317e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.47672748088907799783338276475559452864086595323508e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.19065885055336418361425251181752339740309873019493e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.76274498770011155322885494753359606185503116800378e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.12117840379514058437502004817680010914158504614072e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.84060120900265238260840046704576012273231257957218e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.82527663232857270992129793621400616909426228224712e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.34356222389517260476288187086191333487351565998487e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.33976274743353308671561884595553840397953308913098e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.63275420187782102031440426363742618331512228841827e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.11866107787483058586064754989935066085956111714438e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73477843721978250663679288968512001020601996296490e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.44160680253022056601252575834792105383788377162879e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21335317476130403409545135558158224084771179668233e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03264853379349880038687006045193401398742550912390e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.69348218890827802115199892879428047102258557679491e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.91237960791142062915636488977731277854393252956156e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.65910753386639565315739798326227665091960854776722e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.74840664437921635017204117045344765826413564522111e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.06868586281732253998273705361085730421392207516563e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54975706173930172767473002839017573406030702226132e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.14583740785288043496760695039190976938266337954109e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.82612140613373038344483207825545571090342007879025e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36036300551703042854867518460643847165588663407845e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04535842750631349697889823754314492592824987833936e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.23732650061248767465314076860143478427557447053122e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.62699739226146233957434830649961751777393343487704e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.42516447612776691413186917108911633335528994759200e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.50767100225890854218096000852153526257706640350006e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.79354212598752921854694346382153053933709178870365e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22829848834607910587947957072698363475405049514861e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.70691063363569289518344600523741530561019748614806e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(512)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.00884949274594535609095332297684714775260624951825e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78341094639115825398956813042728307580960455542744e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.15265496830848341113806943808080759239756631409525e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.57315927453204720744811655603371636021315489208197e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.85204679078224624495444948682547526667095102524654e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74161227361870459284450362091832192677055853279409e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.07876462219180661771716253718191722284989080940137e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.44253835507444012610805302754717855328014016707006e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 17)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.62113944435292862305033337145279742309373660330666e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70079323596084060715031479698836179832962876226040e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 19)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.00660607636030744539992938919621601865776189138704e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.31497886237763070924617652541844323609885629317601e-16), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 21)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.39564398862596160919395415621974601387332621717152e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66093089449300961525223017779109302708950139673730e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 23)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.93613874644561300698599726959494859627890174964948e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.19040904529065504857204743126112225011195501281143e-19), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.17543358264254064703206600829360293655988827362640e-20), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.62200282665332970268320245960144424095679956397013e-20), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 27)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.86732299457579395219850121616992176031753071031156e-21), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.06875883329165532088522436776488158979648490753308e-22), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 29)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.96038435804935610062036418083267399916819002834560e-23), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.58398713540364228777663367294171284220596445493878e-23), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 31)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80012011189042378144386404030074611545380892760386e-24), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.94995979813638214930198027960375045149298281432705e-25), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 34)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54686243691761942165686883762842249372631955082800e-26), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.83394511536756069267771511760113721074800848019252e-28), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 38)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51060784855236271646178597425041551955920802058680e-29), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.72064952672613348894308116953255143520457854922274e-31), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 42)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.47520297710191671529471286547892233784021035466379e-32), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.61000930344348973529597770462163230645079307893137e-34), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 46)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.44062790732609054227999303269426009576929146861433e-35), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.50196221039403294462497822716956279927920276478346e-37), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 50)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40686319074813529519530569599048837477475167906008e-38), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.39644747108792279748533029997027617117109903686087e-40), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 54)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.37388983471497587421416571874071130349096844921335e-41), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.29340573348429960691926787106472282340927640380120e-43), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 58)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34168929171384362716227120970772588231539887618792e-44), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.19277903660576133488209753033664338223562148808726e-46), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 62)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31024344893930041715065547823020105694863171502727e-47), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.09451077793531380359579836946937830296447410946021e-49), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 68)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.99854568157745488632402184518493974898874424751974e-52), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.90482976716548328742580258318841772362182055421850e-55), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 76)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.81331031949754227287676033514493918322443413497900e-58), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.72393585888431862585621126478997967111761145994043e-61), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 84)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.63665611219171740806270631327146452257579244134808e-64), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.55142198456222403131123663405416457282792230600398e-67), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 92)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.46818553179904690557737952544352009065226787695701e-70), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.38689993339750674372790969281593758852760534859083e-73), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 100)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.30751946620850267942178680939056405129648959823324e-76), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22999947871924089787283868104547270634422812327464e-79), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 108)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.15429636593675868932894402445846943978928527663539e-82), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.08036754486011590754779689888522406229422390296425e-85), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 116)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.00817143052745694096464540906760162333420303023853e-88), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.93766741262446966891078653229257971028730764671731e-91), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 124)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.86881583264108366104568997294197237332744887374738e-94), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80157796156355826273993161420114489582758679076892e-97), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80157796156355826273993161420114489582758679076892e-97), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 136)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.67179294735294176362984811229814996321448020054714e-103), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 144)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54802031264585663187966166715445514985511798910822e-109), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 152)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42998152985177672565427939143605723367225455199072e-115), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 160)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31741097436120674672534884589772914283013777922699e-121), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 168)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21005532680626558945212254133007921488775041506480e-127), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 176)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.10767300301195677705013517506606980789923707491379e-133), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 184)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.01003361035533597664845960146529179372714717379932e-139), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 192)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.91691742930921170868726692339448146221842496280605e-145), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 200)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.82811491900368853443838779773185869428484436302762e-151), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 208)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74342624569290975040282039426027173450931965163004e-157), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 216)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66266083306590056457788505006816075755054440653805e-163), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 224)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.58563693338956886728085045821014476542524758008771e-169), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 232)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51218121851879965522847219296469189207577474602480e-175), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 240)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.44212838985328641436431140228718938071801638224106e-181), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 248)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.37532080636337891994887485722273767539788854812723e-187), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31160812984788791651618467066072242297924856960987e-193), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_cdf() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    // Left tail of MapAiry distribution inherently limits accuracy due to the rapid decay of the function value.
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41267677661910675305382637108866964194971791718950e-4526), tolerance * 10000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.54257458395442655204313046627840914088415076161558e-3561), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.76526114681351440657658256501506814774341295354922e-2743), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.27125192542763584917976723724200097302881304411026e-2062), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.63420353908123838151105660636393196705034472693701e-1504), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.73063520387773809387189485017024200015891995562535e-1057), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87621737485296751690587747650586494006422187478425e-871), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.15431208733487028595846723973676765010037279135068e-709), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.43691228284359420616981379531226120901046405612148e-568), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.94968725712637201733674598027292284194096042437396e-447), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.29903053626343464859342616262182157628847143496274e-345), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.84047255379287327139957588241933355294847277196332e-260), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.79693358121995588307456046556605766916709047503732e-190), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.18430997009344263598977403572421080974427275299628e-134), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.48855014561480471542019201883561279219078606560987e-111), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21188569601559235190706821985880880180873775432009e-90), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.35130675597097791570481439579882762106711974084282e-73), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.37868088180310978639226395953519655144561270921646e-58), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.96213643222050268066173675729245650285594216946076e-45), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.54299664164424690506837069440249011982208536500223e-34), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55256887328573175640232009727759270424877561672436e-25), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.76658378173553845515260323095527685644595481193099e-18), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.53962634794926085939153812168733146892427038438756e-15), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.86644224834640735242565183060109673582728544145215e-13), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03383983622833649608546095853620837892086215933298e-10), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.91537779419133030075704775849634109699414218730481e-9), tolerance * 10);

    #ifndef BOOST_MATH_HAS_GPU_SUPPORT
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.98085764775551392183308859729746156105641469279537e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.74005376900861403666100209810033834996576655180465e-6), tolerance);
    #else
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.98085764775551392183308859729746156105641469279537e-7), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.74005376900861403666100209810033834996576655180465e-6), tolerance * 10);
    #endif

    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38210242545410097936371084112947842536958832190953e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.20388922732913226923590659641162717163243937441333e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.01190141003747964176020229987886598385662366685178e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.81314867410709227363443913418400543623685087660036e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40335466769957406990026955582239481134043242080576e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65057217408705944017729221733489193253150025104939e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.62249422314997336809404422537950480283670862386341e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.49463093988108215731880519187728405325669137953076e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.13731490830425435301322069056843269169838775278008e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.62598955251978523174755901843430986522046460911826e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90499210557846036762061906872202100639194335579553e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20405659651083834253744598353980626247223640040816e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.52019615753011340929652761968369152398962550184606e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85001847038770817549042011993996310385733185736928e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.18987376977895959785022298515900161162070947963445e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.53600587820316569763364664186012515534206385253539e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.88469588484092929432583134493879046529813482615104e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.23238998449671083670041452413316011919930693856385e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.40482473418234444080637631800355214131813723348029e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.57580541347059341976751497432836530189703719631253e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74497724348320019407539810613151808302214091800752e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.91201100390145330029548239186193815674350921221373e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.07660411254690680532566892253363954450731338298810e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.23848132271235424597841060643701240046967758423031e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.39739506217816326324441201522899717272689507530078e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.55312544188864070398568309403385234827691551635647e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.70547996523808722092270387778468756135745550977842e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.85429297088468989836573146625524898390641430572229e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.99942484305170440309173928292359597658678248461166e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.14076102348433999265595009037280955266566735857487e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.27821085871238004955876961091597273535850920355153e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.41170631513311055182782737833523982566081113132113e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.54120059278782917662600347641177702821526512639253e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.66666666666666666666666666666666666666666666666667e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.78809578204369927830516637473281842472755486735670e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.90549592781258492544813432997757039278246435622280e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.01889030914802776673544656107973638254340107333208e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.12831583813566709661955692651725308671816974125125e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.23382165835700226601867127791408227947098667962247e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.33546771684027578538449554398877013955180783305967e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.43332339433579641844979454704458465657719551526672e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.52746620257793576893322261719616438461321552900402e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.61798055507971738758762860873629690865112979477773e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.70495661608502783177600997939957544744070827966369e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.78848923058628458126731690124829998626066330325893e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86867693680400430433267457602884639416039914905443e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.94562106121611865967124973894234486976489176337189e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.01942489511196041360944230423166502583801465481058e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.09019295071828335416784977458008517243666113680032e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.15803029418984060112492565010063896412665755747313e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.28533238809939722865056318522346970220432633092471e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40216194581500702945839843944471675879234503695234e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.50932751104023843947651194694738668613497033313653e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.60760530927220029947644215642926175488027901722942e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.69773224168421631436187338078646751800825779589535e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.78040142637134263427912246171411479166776409276684e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.85625978983111842670865940159150361843054620140300e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.92590726602475875901684499549667744162908005254207e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.04873777288161758991649770533103505875566036067315e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.15278911856248478088063341213669352039898456875983e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.24130503839415290663064766600863877055306793445333e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.31696102735179394495180164200593784488582138327694e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.38195112746199551037885830108115610693678956104594e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.43806900885896605694390358343940460845582243989715e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.48677954836791634391273767979450406335892640585045e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.52927980046477119746205391036666872656611789316554e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.59939373386293535847669182576818465955927373035467e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.65433945362867879413538593557411941750837354200812e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.69816672594606569822299986511094581346810669802925e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.73368884642273321841166642302379087929797275815602e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.76289558464640301369422872101450331194067414657857e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.78721979671508698647720200666528542996105699216801e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.80771139145909061814002155447578879059651577452407e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.82515243555548699954994382507757214540918643206410e-1), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_ccdf() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.37401044748021476825244098156569013477953539088174e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.09500789442153963237938093127797899360805664420447e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.79594340348916165746255401646019373752776359959184e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.47980384246988659070347238031630847601037449815394e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.14998152961229182450957988006003689614266814263072e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.81012623022104040214977701484099838837929052036555e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.46399412179683430236635335813987484465793614746461e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.11530411515907070567416865506120953470186517384896e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.76761001550328916329958547586683988080069306143615e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.59517526581765555919362368199644785868186276651971e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.42419458652940658023248502567163469810296280368747e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.25502275651679980592460189386848191697785908199248e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.08798899609854669970451760813806184325649078778627e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.92339588745309319467433107746636045549268661701190e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.76151867728764575402158939356298759953032241576969e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.60260493782183673675558798477100282727310492469922e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.44687455811135929601431690596614765172308448364353e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.29452003476191277907729612221531243864254449022158e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.14570702911531010163426853374475101609358569427771e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.00057515694829559690826071707640402341321751538834e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.85923897651566000734404990962719044733433264142513e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.72178914128761995044123038908402726464149079644847e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.58829368486688944817217262166476017433918886867887e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.45879940721217082337399652358822297178473487360747e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.33333333333333333333333333333333333333333333333333e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.21190421795630072169483362526718157527244513264330e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.09450407218741507455186567002242960721753564377720e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98110969085197223326455343892026361745659892666792e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.87168416186433290338044307348274691328183025874875e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76617834164299773398132872208591772052901332037754e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.66453228315972421461550445601122986044819216694033e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56667660566420358155020545295541534342280448473328e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.47253379742206423106677738280383561538678447099598e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.38201944492028261241237139126370309134887020522227e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.29504338391497216822399002060042455255929172033631e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21151076941371541873268309875170001373933669674107e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.13132306319599569566732542397115360583960085094557e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.05437893878388134032875026105765513023510823662811e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.98057510488803958639055769576833497416198534518943e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90980704928171664583215022541991482756333886319968e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.84196970581015939887507434989936103587334244252687e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.71466761190060277134943681477653029779567366907529e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.59783805418499297054160156055528324120765496304766e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49067248895976156052348805305261331386502966686347e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.39239469072779970052355784357073824511972098277058e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30226775831578368563812661921353248199174220410465e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21959857362865736572087753828588520833223590723316e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.14374021016888157329134059840849638156945379859700e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.07409273397524124098315500450332255837091994745793e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.51262227118382410083502294668964941244339639326852e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.47210881437515219119366587863306479601015431240172e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.58694961605847093369352333991361229446932065546671e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.83038972648206055048198357994062155114178616723061e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.18048872538004489621141698918843893063210438954059e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.61930991141033943056096416560595391544177560102847e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.13220451632083656087262320205495936641073594149555e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.70720199535228802537946089633331273433882106834460e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.00606266137064641523308174231815340440726269645335e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.45660546371321205864614064425880582491626457991880e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.01833274053934301777000134889054186531893301970748e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.66311153577266781588333576976209120702027241843979e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37104415353596986305771278985496688059325853421426e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.12780203284913013522797993334714570038943007831995e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.92288608540909381859978445524211209403484225475927e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74847564444513000450056174922427854590813567935902e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(9))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.46888494885704354461170804278826523218710845325600e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.25613324230091750631240813030902908673461241410450e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08995384276796746727925033729177141093311272472945e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.57298412261448730944183018356841954457807894385431e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.49445667426015051170301411155410826055184233951431e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.60377136717752997401167765152949642869947243569953e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.85823720627621671017207210698248665551570440429385e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.22684103170563193014558918295924551172698239430713e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.22006950358287273393523413555812293504310335665744e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.45787594574479193246071067008198432980227781543963e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.86454025983790677529906306805195677315975322722262e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.39200041143645646258373331865031120883657317932172e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.00844464598329521611240330284790908003180945122524e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.69207105975977872414400493979073211647180301757125e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42749222811426799055113000065152671289607878504149e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20357145727036120652264700679701054983338793783565e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.84678186992189202260619598427428218285343162114238e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.57685005349633837416913019835892196933717853056275e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.36681189477727165566612776659042216546642387956225e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.19958501309529382285703116836639459985687034231775e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06387729083711703707125561075502300925516821745411e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.51956726422504037060971384488455660747488001979504e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.58369730588727742464054792329377206933983130862391e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.79171138168694768852946218859426934990849762414603e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.52989549449674176739274234303186543768254475184699e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.57534022322054445175812334907544905864245720421642e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.83262692804541386527340439387556377914021496108680e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.24132027619974492546489695559384126869565934766815e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.76147662885863808608468523190827912794216909524203e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.36574864341415676513997111497521230889735520043816e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03484906797876224319052497399923183994442800112441e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.75482620393356950171417803995157642981745916059418e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.73979922776430555363554632966080889507162029820752e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(512))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44353982639936003430898773389958569528340214901924e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1024))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21747521599099094720054911524728128905331418743558e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.30442492111514986227593224497576120783344940470632e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52184402609265962838062304205455964936542503385355e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.38053115409151479601077504716814204054241984396579e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90230503273504174320645323659877052404533613091567e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.72566394266708055503723315148592640827491725732841e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37788129092113064009710124961268854372530077851514e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 17))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40707992833487973793766069223107728401530584918012e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.97235161365145877787702193738699569533619914113956e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 19))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05088499104186197709404571098840228587836479911802e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.71543951706432436058369237052221627684485231071286e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 21))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31360623880232751062247585210413319083311528997997e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.64429939633040546807800247387131067206560952703714e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 23))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.64200779850290938904479244625064763530742033320990e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.80537424541300683543633877614223483555123688801843e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.05250974812863673632096512092113144155816177586613e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.25671780676625854430204135462707277286762193086669e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 27))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56563718516079592040149887308711394838177046916383e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.07089725845782318037768094883949095101477347530653e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 29))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.20704648145099490050187930370138656919662848438144e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.13386215730722789754721037105719349775366357126605e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 31))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.00880810181374362562734924119592255004499293746830e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41732769663403487193401296875220001924045479390391e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 34))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.77165962079254358991751621103655291754760707304047e-16), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21457452599067948739689526379757206282312287607911e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 38))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76821815748834935924611907974700181516735308796565e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.46027269686043669905764884968375298647166107661462e-19), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 42))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.32534087107554587382206106210469124710349176992174e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.40667608884443234227757632763086405915307399803018e-21), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 46))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.75834511105554042784697040953858007394668838202264e-22), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.44793138881942553480871301192322509243346488933465e-23), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 50))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05599142360242819185108912649040313655418331509614e-23), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31998927950303523981386140811300392069272914785317e-24), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 54))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.64998659937879404976732676014125490086591143489426e-25), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.06248324922349256220915845017656862608238929361934e-26), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 58))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.57810406152936570276144806272071078260298661702420e-27), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22263007691170712845181007840088847825373327128025e-28), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 62))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.02828759613963391056476259800111059781716658910032e-29), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.03535949517454238820595324750138824727145823637540e-30), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 68))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.86774921121022248157180194922091913636165349433656e-32), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22933581425159726274559405456576861505650835849009e-33), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 76))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.92083720976812072303999071025901346102579431014076e-35), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.00130814026268862974998548477970853285280360959494e-37), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 84))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.68954396916045098398435231996829458258250563999209e-39), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.32741245181320466247555049995046028528516506248764e-41), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 92))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.14490819559581322851180476561725941957580704101369e-42), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.78891905561845816954969494627696784308719850158390e-44), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 100))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79518602440384088992139835355776225482374765872484e-46), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.36747816313100139050218492743400352316210571675756e-48), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 108))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.82418462989218967265966394911563050494079018243369e-50), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06627884842065463635307249204931726639699846600526e-51), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 116))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66606070065727286930167576882705822874531010313323e-53), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.60321984477698885828386838879227848241454703614566e-55), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 124))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.06753100746404509106854435748793512877272974397760e-57), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.35551719916257045479460055857489863870739022496500e-59), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.35551719916257045479460055857489863870739022496500e-59), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 136))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55163994120179942744008802699582486296567144164185e-62), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 144))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.78818345019970563339865240965777554434978379307091e-66), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 152))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.24849475146412508153967873451605357507271433855203e-70), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25793328893167116248527312854395839235173689906056e-73), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 168))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.51253244368083779903631134898427341882748266372206e-77), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 176))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34583311613301704078034945043561362764342838469777e-80), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 184))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.28572538118412363471765002547757233311383882982854e-84), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.02179048140655184257238775751360432889120808063607e-88), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 200))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95844494174964644594052435486171980685820509781154e-91), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 208))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.78135972106847276840948328823662062221241478957896e-95), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 216))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16732415065148260947497150591714370659482782948705e-98), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.84991247717647121453850465311802662742877888058362e-102), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 232))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.95779413373161917611939612577643219587104218892485e-106), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 240))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.69868020843057108791977444476963676657007865940548e-109), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 248))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.14716847761369894511663682805087101213398110206416e-113), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01249230410490696902261641309835718069677272999613e-116), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 600))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.71969536983273370677446521953393512020163902133418e-272), tolerance);

    // The test is terminated because x is inf after this in fp64.
    if (N <= 53) {
        return;
    }

    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 10000))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41571262853590848961675783559267306723234880166889e-4516), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_quantile_nearzero() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.92947985498880164288717582971263069265106409267331e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.59822399410385083283727681965013516508738702183587e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.37051518906738394785544825118798831695789651589080e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.18765177572396470161180571018467021827812222940306e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.02990438632621036358966845969387883330987649629695e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.88799753980914934718534117636680696659950540760708e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.75672646260182080486757173789442594487758358719568e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.63281240925531315038207673147576301428806562043208e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.51400400121400801502802449510679455584862826208973e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.39864166928988824635332108295073034590821798923232e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.28542241490194660184907680173328998444992309715534e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.17326074020471664204142312429732771994689840460362e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.06119903663447640834652691309939508582577539512003e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.48344198262277235851026749871350785648904607023049e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.33817641975610807828267883212039917352738594297812e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.16710685455022053317001962780672309444401429230030e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.96039220307812911673187933977720502663701213461352e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.70691924008504753346480857177819427470334982130948e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.39365150798529936746993802101957358628812489096556e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.00474815142578902619056852805926625285378475306847e-1), tolerance);

    // Relative error decreases near the root.
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.20300033010874226241969971846939668616524880409485e-2), tolerance * 4);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08557609474704600193050942980969400222443967430903e-1), tolerance * 4);

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.84775355628419671941908878470333245751721020657513e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.81512108276093787175849069715334396559730936013748e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.05946829281923976565971908177998486051088750363345e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.69237131791870252926200620656937624841227431099908e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.29003620591891062966640208281844476036608475704263e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70276979914029738186601698003670175695927342267323e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.28080064428173146950050714257245479404308412291534e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22141517097499177560650456638736418866609207598601e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.36798741582712685566660459972731625067678977232527e0), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_quantile_lower() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.18765177572396470161180571018467021827812222940306e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.59822399410385083283727681965013516508738702183587e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.92947985498880164288717582971263069265106409267331e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.20998806047945704828514962310033638603842757237089e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.45496901537018787335070643452598202244805633093375e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.67354365380697578246790709817724830651682057476917e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.05331626220443158014611601879873776289585727270317e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.37844135506419820279451552495734835192576864639213e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.66475100891248273292011043832615771689316788780040e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -16)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.92187819510636694694450607724165688786106650410224e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -20)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.37198340489560108355199818136414022326129982959773e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -24)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.76028477470580699278679082649703786389435657070830e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -28)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.10419378761193307962534180728809528372838135595700e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -32)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.41443550638291131009585191506467027832121974024563e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -40)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.96026448032205844753104387192915744062974794241787e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -48)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.43355682497258196703130282301265363831865507562392e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -56)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.85432398223245647091315413807705550499215849013310e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -64)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.23500806363233607692361021471929015767129541901722e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -80)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.90695141954692987290075695228754376250679911499016e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -96)), BOOST_MATH_BIG_CONSTANT(RealType, N, -9.49149767947731938416802231743764792842977181401152e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -112)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.00124089585489463242206889945762289007601620222216e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -128)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.04845570631944023525776899386112795192930377369709e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -160)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.13196240444608571220864964743861585994042496495014e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.20475312075429969521072327133506584708675011147301e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -224)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.26971371983788322297191645356972895828456120573844e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.32865827226175697711590794217590458317537880046696e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.32865827226175697711590794217590458317537880046696e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -320)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.43303723297942256778502134022428247271630018451935e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -384)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.52413050129859916220402440029315858653640179700446e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -448)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.60549473918897820617071548963372947780704796096977e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -512)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.67937186583822375017526293948703697021001956377209e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -640)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.81028650954862232126230250634527453300745376729470e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -768)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.92461760606525027716536281516018610241283237089375e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -896)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.02678879167343824140492421290271637384390204302948e1), tolerance);

    // The test is terminated because p = 0 after this in fp64.
    if(N <= 53){
        return;
    }

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.11959316095291435774375635827672516757012476284109e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1280)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.28411538998117558460421271505353342631141242515516e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1536)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.42785430825019465839030492595482468932343681278349e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1792)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.55634220352411216642587067287258912926964865154135e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -2048)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.67307564006689676593687414536012112442665935238575e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -2560)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.88006705102088142572333576380332410442882971872985e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3072)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.06095108081567569614908765234696115918183394966145e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3584)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.22266885559263332213569359568928269416384800428618e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4096)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.36960987939726803544369406181770745085696438921663e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5120)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.63019918860979732881015320942243118103538614206846e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6144)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.85794964449030059786959849735574335608745736630897e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7168)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.06158611834421484993920289979452825015061756369038e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.24662793339079714510108682543625432044226557042768e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10240)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.57480988422877882295975852850886697641784458384426e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12288)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.86165509604814747106458415059041118242367113233764e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14336)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.11814205550106893579346476384206383922751551334008e1), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_quantile_upper() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.16710685455022053317001962780672309444401429230030e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.46875))), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.96039220307812911673187933977720502663701213461352e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.70691924008504753346480857177819427470334982130948e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.40625))), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.39365150798529936746993802101957358628812489096556e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.00474815142578902619056852805926625285378475306847e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.34375))), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.20300033010874226241969971846939668616524880409485e-2), tolerance * 4);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08557609474704600193050942980969400222443967430903e-1), tolerance * 4);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.28125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.84775355628419671941908878470333245751721020657513e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.81512108276093787175849069715334396559730936013748e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.21875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.05946829281923976565971908177998486051088750363345e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.69237131791870252926200620656937624841227431099908e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.15625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.29003620591891062966640208281844476036608475704263e0), tolerance);

    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70276979914029738186601698003670175695927342267323e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.22141517097499177560650456638736418866609207598601e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.36798741582712685566660459972731625067678977232527e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.63240190107315016002742117235901647573374032928275e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.37487073802383509843362175744776155324007021797019e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18429841828919685808654152263914024115105410242857e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.50552947051871870287466561935025391775332522082960e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38732922416208535732704088926085913683321415168727e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.49585416520688234273370911116680490544653394894587e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.80900106401310352569173769396359734191276945763559e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.59336704609415555521333325344685782416363311947039e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.55156669326461321628671444544286440554399508021258e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25510428200953290831680861672648808196801287129080e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43190196382396984755739962193081183260874381700500e6), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.77306696194440571351895040208958865072901805506645e7), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.32755474809804845725014907927003674268598932505496e9), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.38411271011676880227484089908793278086771148389202e10), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.78343716417988572841183668816998971166780683532365e12), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.14997210570212560225883973162628933329443239976683e15), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.99677152273063632531264438890163357898418349270434e18), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.62497388866235918645459765949830559011448320623360e22), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.64139290653028845793860203742403295803332310962125e25), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.97920970865877789202766313638336627695568626671453e31), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.84407885843159775798215826490261804734522750775796e38), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.87250989448760462117787200600339872607929758715096e44), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28743695332377477386366453958993754420344828061498e51), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28743695332377477386366453958993754420344828061498e51), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -320))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.98819430202303731872899900564216427734599065613620e63), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -384))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.27507518735966284635753747136817320622994832637767e76), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -448))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.38092093738495853701408205401696240265896114717023e89), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -512))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.05852403143769157513875467957386926795647775801563e102), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -640))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49075014586685014041688089060587827415714205982384e128), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -768))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.26604066065618724776472365999448082037897284653223e153), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -896))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.54152887582709267575372250070675971504362522185818e179), tolerance);
}

template<class RealType, int N>
void do_test_mapairy_locscale_param() {
    //
    // Basic sanity checks, tolerance is either 3 epsilon
    // expressed as a percentage:
    //

    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    mapairy_distribution<RealType> dist_0_1(static_cast<RealType>(0), static_cast<RealType>(1));
    mapairy_distribution<RealType> dist_1_3(static_cast<RealType>(1), static_cast<RealType>(3));
    mapairy_distribution<RealType> dist_0_invcbrt18(static_cast<RealType>(0), 1 / cbrt(static_cast<RealType>(18)));

    BOOST_CHECK_CLOSE(entropy(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.0072768184106563460003025875575283708), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_1_3), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.0072768184106563460003025875575283708) + log(static_cast<RealType>(3)), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_0_invcbrt18), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.0438195657786014485977283891231190450), tolerance);

    BOOST_CHECK_CLOSE(median(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, -0.71671068545502205331700196278067230944440), tolerance);
    BOOST_CHECK_CLOSE(
        median(dist_1_3),
        (1 + 3 * BOOST_MATH_BIG_CONSTANT(RealType, N, -0.71671068545502205331700196278067230944440)),
        tolerance
    );
    BOOST_CHECK_CLOSE(median(dist_0_invcbrt18), BOOST_MATH_BIG_CONSTANT(RealType, N, -0.27347630981017495237228835747364595601553), tolerance);

    BOOST_CHECK_CLOSE(mode(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.1615872711359706852500000803029112987), tolerance);
    BOOST_CHECK_CLOSE(
        mode(dist_1_3),
        (1 + 3 * BOOST_MATH_BIG_CONSTANT(RealType, N, -1.1615872711359706852500000803029112987)),
        tolerance
    );
    BOOST_CHECK_CLOSE(mode(dist_0_invcbrt18), BOOST_MATH_BIG_CONSTANT(RealType, N, -0.4432284977460014720866292801600737435), tolerance);

    BOOST_CHECK_EQUAL(mean(dist_0_1), static_cast<RealType>(0));
    BOOST_CHECK_EQUAL(mean(dist_1_3), static_cast<RealType>(1));
    BOOST_CHECK_EQUAL(mean(dist_0_invcbrt18), static_cast<RealType>(0));

    BOOST_CHECK((boost::math::isinf)(variance(dist_0_1)));

    BOOST_CHECK_CLOSE(pdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06251243013238748252181151646220197947016365256337e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.97516171847191855609649452292217911972760948598791e-1) / 3, tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_0_invcbrt18, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.29264884227495575949271548513687010027396013858107e-2), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.15803029418984060112492565010063896412665755747313e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.66666666666666666666666666666666666666666666666667e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_invcbrt18, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.19730152884211135907661924151362266458717453254074e-1), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_invcbrt18, quantile(dist_0_invcbrt18, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_invcbrt18, quantile(dist_0_invcbrt18, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
}

BOOST_AUTO_TEST_CASE(mapairy_pdf_fp64)
{
    do_test_mapairy_pdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_pdf_std64)
{
    do_test_mapairy_pdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_pdf_fp128)
{
    do_test_mapairy_pdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_cdf_fp64)
{
    do_test_mapairy_cdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_cdf_std64)
{
    do_test_mapairy_cdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_cdf_fp128)
{
    do_test_mapairy_cdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_ccdf_fp64)
{
    do_test_mapairy_ccdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_ccdf_std64)
{
    do_test_mapairy_ccdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_ccdf_fp128)
{
    do_test_mapairy_ccdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_quantile_nearzero_fp64)
{
    do_test_mapairy_quantile_nearzero<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_quantile_nearzero_std64)
{
    do_test_mapairy_quantile_nearzero<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_quantile_nearzero_fp128)
{
    do_test_mapairy_quantile_nearzero<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_quantile_lower_fp64)
{
    do_test_mapairy_quantile_lower<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_quantile_lower_std64)
{
    do_test_mapairy_quantile_lower<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_quantile_lower_fp128)
{
    do_test_mapairy_quantile_lower<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_quantile_upper_fp64)
{
    do_test_mapairy_quantile_upper<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_quantile_upper_std64)
{
    do_test_mapairy_quantile_upper<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_quantile_upper_fp128)
{
    do_test_mapairy_quantile_upper<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(mapairy_locscale_fp64)
{
    do_test_mapairy_locscale_param<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(mapairy_locscale_std64)
{
    do_test_mapairy_locscale_param<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(mapairy_locscale_fp128)
{
    do_test_mapairy_locscale_param<cpp_bin_float_quad, 113>();
}
#endif
