//! OpenLdap namespace
namespace OpenLdap {
//! The LdapClient class
/**@par Restrictions:
    @ref Qore::PO_NO_NETWORK

*/
class LdapClient {

public:
//! add ldap an entry and attributes
/** Give the new entry's objectclass as an attribute; an attribute value can be a list to add multiple values to an attribute

    @par Example:
    @code
$ldap.add("uid=temp,ou=people,dc=example,dc=com", ("objectclass": "inetorgperson", "sn": "Test", "cn": "test test"));
    @endcode

    @param dn the distinguished name of the entry to add
    @param attrs a hash of new attributes; the keys are attribute names and the values are the attribute values
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ADD-ERROR missing attribute value
    @throw LDAP-ERROR an error occurred performing the add operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
*/
nothing add(string dn, hash attrs, __7_ timeout timeout_ms);

public:
//! bind to the server with the given authentication parameters
/** The current session is disconnected before binding again.

    @par Example:
    @code
$ldap.bind($h);
    @endcode

    @param bind a hash of bind parameters, allowed keys are:
    - \c binddn: the dinstinguished name to use to bind to the LDAP server
    - \c password: the password to use for the connection
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-BIND-ERROR parameter type error or 'password' given with no 'binddn' value
    @throw LDAP-ERROR an error occurred performing the bind
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
 */
 bind(hash bind, __7_ timeout timeout_ms);

public:
//! check ldap attribute values; if any errors occur (entry does not exist, etc), an \c LDAP-ERROR exception will be thrown
/** @par Example:
    @code
my bool $b = $ldap.compare("uid=temp,ou=people,dc=example,dc=com", "uidnumber", 1000);
    @endcode

    @param dn the distinguished name of the entry to find for the attribute value comparison
    @param attr the name of the attribute for the value comparison
    @param vals a single string or a list of strings of values to compare; if any value is not a string it will be converted to a string
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @return \c True if the value(s) match, \c False if not

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ERROR an error occurred performing the comparison operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
 */
bool compare(string dn, string attr, softlist vals, __7_ timeout timeout_ms);

public:
//! Creates a new LdapClient object and establishes a connection to the server with optional bind parameters
/** A connection to the server is made with an empty search request even if no bind parameters are given.
    To ensure that a secure TLS connection is made to the server regardless of the scheme given in the URI,
    set the \c "starttls" argument to \c True.

    @par Example:
    @code
my LdapClient $ldap("ldaps://ldap.example.com:389", ("starttls": True, "timeout": 20s));
    @endcode

    Each LdapClient object represents a connection to the server.  Individual requests are wrapped in mutual exclusion locks to ensure atomicity and thread-safety, therefore if sharing a single LdapClient object between multiple threads, simultaneous requests will block if another request is already in progress.

    @param uri the URI of the ldap server (ex: \c "ldaps://ldap.example.com")
    @param options an optional hash of optional parameters, allowed keys are:
    - \c binddn: the dinstinguished name to use to bind to the LDAP server
    - \c password: the password to use for the connection
    - \c timeout: the default timeout for ldap operations; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond
    - \c no-referrals: (boolean) do not follow referrals (the default is to follow referrals)
    - \c starttls: (boolean) if set, then a \c STARTTLS command will be executed if a secure connection is not already established; note that setting this option will ensure a secure connection regardless of the scheme in the URI.  If a secure connection has already been established (for example by using a \c "ldaps" scheme in the URI), then this parameter is ignored

    @note If no \c "timeout" option is given, a default timeout value of 60 seconds is set automatically

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-ERROR an error occurred creating the ldap session context
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
 */
 constructor(string uri, __7_ hash options);

public:
//! Creates a new LdapClient object based on the original
/**@par Example:
    @code
my LdapClient $ldap2 = $ldap.copy();
    @endcode

    @throw LDAP-ERROR an error occurred copying the ldap session context
 */
 copy();

public:
//! delete ldap entries; if any errors occur (entry does not exist, etc), an \c LDAP-ERROR exception will be thrown
/** @par Example:
    @code
$ldap.del("uid=temp,ou=people,dc=example,dc=com");
    @endcode

    @param dn the distinguished name of the entry to delete
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ERROR an error occurred performing the delete operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
 */
nothing del(string dn, __7_ timeout timeout_ms);

public:
//! unbinds from the LDAP server (if a connection has been established) and destroys the object
/** @par Example:
    @code
delete $ldap;
    @endcode
 */
 destructor();

public:
//! returns the URI string used to connect to the LDAP server
/** @par Code Flags:
    @ref CONSTANT

    @par Example:
    @code
my string $uri = $ldap.getUri();
    @endcode

    @return the URI string used to connect to the LDAP server
*/
string getUri();

public:
//! returns \c True if the connection to the server is secure, \c False if not
/** @par Code Flags:
    @ref RET_VALUE_ONLY

    @par Example:
    @code
my bool $b = $ldap.isSecure();
    @endcode

    @return \c True if the connection to the server is secure, \c False if not

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
 */
bool isSecure();

public:
//! modify (add, replace, delete) ldap attributes; if any errors occur (entry does not exist, etc), an \c LDAP-ERROR exception will be thrown
/** @par Example:
    @code
$ldap.modify("uid=temp,ou=people,dc=example,dc=com", ("mod": LDAP_MOD_ADD, "attr": "someattr", "value": "new-value"));
    @endcode

    @param dn the distinguished name of the entry to modify
    @param mods a hash or list of hashes of modifications to make; each hash is made up of the following keys:
    - \c mod: a modification action; see @ref ldap_modify_constants for possible values
    - \c attr: the attribute to modify
    - [\c value]: the value to add or replace
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-MODIFY-ERROR invalid mod hash format; missing value for add or replace operation
    @throw LDAP-ERROR an error occurred performing the modify operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
*/
nothing modify(string dn, softlist mods, __7_ timeout timeout_ms);

public:
//! changes the LDAP password of a user
/** @par Example:
    @code
$ldap.passwd("uid=test,ou=people,dc=example,dc=com", "oldpwd", "newpwd");
    @endcode

    @param dn the distinguished name of the user whose password to change
    @param oldpwd the old password
    @param newpwd the new password
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ERROR an error occurred performing the password change operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
*/
nothing passwd(string dn, string oldpwd, string newpwd, __7_ timeout timeout_ms);

public:
//! renames entries in the Directory Information Tree
/** @par Example:
    @code
$ldap.rename("uid=test,ou=people,dc=example,dc=com", "uid=test1", "ou=people,dc=example,dc=com");
    @endcode

    @param dn the distinguished name of the entry to rename
    @param newrdn the new relative distinguished name of the entry
    @param newparent the distinguished name of the entry's new parent
    @param deleteoldrdn if this argument is \c False, then the old relative distinguished name will be maintained along with the new name, if \c True (the default), then the old attributes are deleted
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ERROR an error occurred performing the rename operation
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
*/
nothing rename(string dn, string newrdn, string newparent, softbool deleteoldrdn = True, __7_ timeout timeout_ms);

public:
//! performs a search on the LDAP server
/** @par Example:
    @code
my hash $h = $ldap.search(("base": "dc=example,dc=cp,", "filter": "(objectClass=*)", "attributes": "uid");
    @endcode

    @param h a hash of search options with one or more of the following keys:
    - \c "base": the search base distinguished name for the search (ex: \c "dc=example,dc=com")
    - \c "filter": the search filter (ex: \c "(objectClass=*)")
    - \c "attributes": one or more attribute names; if this is present then only the given attributes will be returned
    - \c "scope": an integer giving the search scope; see @ref ldap_scope_constants for allowed values; note that if this key value is not present then @ref LDAP_SCOPE_SUBTREE is used
    @param timeout_ms: an optional timeout in milliseconds (1/1000 second); if no timeout is given or a timeout of 0 is given, the default timeout for the LdapClient object is used instead; note that like all %Qore functions and methods taking timeout values, a relative date/time value can be used to make the units clear (i.e. \c 20s = twenty seconds, etc.); integers are treated as values in milliseconds, relative date/time values have a maximum resolution of 1 millisecond

    @return a hash of the return value of the search; the hash is empty if no search results are available; the hash is keyed by Distinguished Names; each value is also a hash of attributes and attribute values

    @note strings are converted to UTF-8 before sending to the server if necessary

    @throw LDAP-NO-CONTEXT the LDAP session is not connected or the session context is not bound
    @throw LDAP-ERROR an error occurred performing the search
    @throw ENCODING-CONVERSION-ERROR this exception could be thrown if there is an error converting any string's encoding to UTF-8 before sending to the server
 */
hash search(hash h, __7_ timeout timeout_ms);

public:
//! Returns a hash with information about the openldap library
/** @par Code Flags:
    @ref CONSTANT

@return a hash with information about the openldap library with the following keys:
    - \c ApiVersion: the API version number
    - \c Protocol: the protocol number
    - \c VendorName: the vendor of the library
    - \c VendorVersion: the version of the library
    - \c Extensions: a list of extensions supported (strings)
    - \c ApiFeature: a hash with a single key; the API feature name, where the value is the feature version
*/
static  getInfo();
};
/** @defgroup ldap_scope_constants LDAP Search Scope Constants
//@{
 */

//! limits the search scope to the object itself
    const LDAP_SCOPE_BASE = LDAP_SCOPE_BASE;
//! limits the search scope to all of the descendants
/** this option is only supported on LDAP servers that support the LDAP Subordinates Search Scope extension
 */
    const LDAP_SCOPE_CHILDREN = LDAP_SCOPE_CHILDREN;
//! limits the search scope to the object's immediate children
    const LDAP_SCOPE_ONELEVEL = LDAP_SCOPE_ONELEVEL;
//! limits the search scope to the object and all its descendants
    const LDAP_SCOPE_SUBTREE = LDAP_SCOPE_SUBTREE;
/** @} */
/** @defgroup ldap_modify_constants LDAP Modify Constants
//@{
 */

//! for adding an entry
    const LDAP_MOD_ADD = "add";
//! for deleting an entry
    const LDAP_MOD_DELETE = "delete";
//! for replacing an entry
    const LDAP_MOD_REPLACE = "replace";
/** @} */
/** @defgroup ldap_constants LDAP Constants
//@{
 */

//! the default timeout value
    const DefaultTimeout = 60s;
/** @} */
}
