/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RAAuthExec.h"
#import "RCMStepController.h"
#import "NSArray(CreateArgv).h"

#include <Security/AuthorizationTags.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "argcargv.h"
#include "authtools.h"

extern int		errno;
RCMStepController	*lcontroller;
RAAuthExec              *rae;

@implementation RAAuthExec

    void
argfilter( NSArray *args )
{
    /* log command line arguments */
    int			i;
 
    for ( i = 0; i < [ args count ]; i++ ) {
        [ lcontroller addTextToLog: [ NSString stringWithFormat: @"%@ ", [ args objectAtIndex: i ]]
                        color: [ NSColor blackColor ]
                        displayInPane: NO ];
    }
}

+ ( void )connectWithPorts: ( NSArray * )ports
{
    NSAutoreleasePool		*pool = [[ NSAutoreleasePool alloc ] init ];
    NSConnection		*cnctnToController;
    RAAuthExec		*serverObject;
    
    cnctnToController = [ NSConnection connectionWithReceivePort:
                            [ ports objectAtIndex: 0 ]
                            sendPort: [ ports objectAtIndex: 1 ]];
                            
    serverObject = [[ self alloc ] init ];
   
    [ (( RCMStepController * )[ cnctnToController rootProxy ] ) setServer: serverObject ];
    [ serverObject release ];
    
    [[ NSRunLoop currentRunLoop ] run ];
    
    [ pool release ];
}

- ( id )init
{
    self = [ super init ];
    lcontroller = nil;
    rae = self;
                                            
    return( self );
}

- ( void )reportError: ( NSString * )fmt, ...
{
    va_list                 vl;
    NSString                *error = nil;
    
#ifdef __STDC__
    va_start( vl, fmt );
#else /* __STDC__ */
    va_start( vl );
#endif /* __STDC__ */

    error = [[ NSString alloc ] initWithFormat: fmt
                                arguments: vl ];
                                
    [ lcontroller addTextToLog: error
                    color: [ NSColor redColor ]
                    displayInPane: NO ];
                    
    [ error release ];
}

- ( oneway void )onewayExecuteTool: ( int )tool
	    withArgs: ( NSArray * )args
	    controller: ( id )controller
{
    int				rc;
    
    rc = [ self executeTool: tool
	    withArgs: args
	    controller: controller ];
    
    if ( lcontroller != nil && rc != -2 ) {
	[ lcontroller tool: tool finishedWithStatus: rc ];
    }
}

- ( int )executeTool: ( int )tool withArgs: ( NSArray * )args
            controller: ( id )controller
{
    AuthorizationItem		right = { "edu.umich.rmauthexec", 0, 
                                           NULL, 0 };
    int				rc;
    NSArray			*av = nil;
    
    if ( args != nil ) {
        av = [[ NSArray alloc ] initWithArray: args ];
    }
    
    lcontroller = nil;
    lcontroller = controller;

    if ( tool != RADMIND && lcontroller != nil ) {
        [ lcontroller showProgUIForTool: tool ];
    }
    
    /* in log, show actual command being executed */
    if ( lcontroller != nil ) {
	[ lcontroller addTextToLog: [ NSString stringWithFormat: @"root# cd /\nroot# %s ", nameforcmd( tool ) ]
			color: [ NSColor blackColor ]
			displayInPane: NO ];
	argfilter( av );
	[ lcontroller addTextToLog: @"\n" color: [ NSColor blackColor ]
			displayInPane: NO ];
    }
    
    /* execute command */
    rc = authorize_and_exec( right,
                                tool,
                                av );
    if ( av != nil ) {
        [ av release ];
    }
                                
    if ( rc == -2 && lcontroller != nil ) {
        [ lcontroller authorizationFailed ];
    }
    
    return( rc );
}

    int
exec_command( AuthorizationRef authRef,
                int command,
                NSArray *args )
{
    char			authpath[ MAXPATHLEN ], toolpath[ MAXPATHLEN ], buf[ MAXPATHLEN ];
    char			errbuf[ MAXPATHLEN ] = { 0 };
    NSArray			*argv = nil;
    NSString                    *rootPath;
    char			*toolname;
    char			**av;
    int				wfd[ 2 ], rfd[ 2 ], efd[ 2 ], rr = 0;
    int				status, rc, i, ac, had_error = 0;
    pid_t			pid;
    fd_set			readmask;
    FILE			*r;
    
    AuthorizationExternalForm	extAuth;
    
    /* get path to setuid tool */
    if ( pathfortool( "rmauthexec", authpath ) != 0 ) {
        NSLog( @"rmauthexec couldn't be found" );
        [ lcontroller toolError: "rmauthexec couldn't be found." ];
        return( -1 );
    }

    argv = [ NSArray arrayWithObject:
                [ NSString stringWithUTF8String: authpath ]];
		
    if ( lcontroller != nil && [ lcontroller preApply ] ) {
	argv = [ argv arrayByAddingObject: @"-p" ];
    }
    if ( lcontroller != nil && [ lcontroller postApply ] ) {
	argv = [ argv arrayByAddingObject: @"-P" ];
    }
                
    if ( command >= 0 ) { 		/* get path for radmind tool to execute */
        toolname = nameforcmd( command );
        if ( pathfortool( toolname, toolpath ) != 0 ) {
            NSLog( @"%s couldn't be found.", toolname );
            return( -1 );
        }
        argv = [ argv arrayByAddingObjectsFromArray:
                    [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand", @"--",
                        [ NSString stringWithUTF8String: toolpath ], nil ]];
    }

    argv = [ argv arrayByAddingObjectsFromArray: args ];

    if (( ac = [ argv argv: &av ] ) <= 0 && lcontroller != nil ) {
        [ lcontroller toolError: "Malformed argument list. Cannot continue." ];
        return( 1 );
    }

    if ( av == NULL && lcontroller != nil ) {
        [ lcontroller toolError: "No arguments to execute!" ];
        return( 1 );
    }
    
    if ( AuthorizationMakeExternalForm( authRef, &extAuth ) != 0 ) {
        NSLog( @"Failed to make external auth form" );
        return( 0 );
    }
    
    /* set up pipes for interprocess communication */
    if ( pipe( wfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( rfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( efd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    
    if (( rootPath = [[ NSUserDefaults standardUserDefaults ]
                        objectForKey: @"RARootDirectory" ] ) == nil ) {
        rootPath = @"/";
    }
    
    if ( chdir( [ rootPath UTF8String ] ) != 0 ) {
        [ rae reportError: @"chdir / failed: %s\n", strerror( errno ) ];
        return( -1 );
    }
        
    switch ( fork()) {
    case 0:
        ( void )close( wfd[ 1 ] );
        if ( dup2( wfd[ 0 ], 0 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        
        ( void )close( rfd[ 0 ] );
        if ( dup2( rfd[ 1 ], 1 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        setvbuf( stdout, NULL, _IONBF, 0 );
        ( void )close( efd[ 0 ] );
        if ( dup2( efd[ 1 ], 2 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( efd[ 1 ] );
        ( void )close( wfd[ 0 ] );
        ( void )close( rfd[ 1 ] );
        
        execve( av[ 0 ], av, NULL );
        NSLog( @"execve: %s: %s", av[ 0 ], strerror( errno ));
        _exit( 2 );
        
    case -1:
        NSLog( @"fork() failed: %s", strerror( errno ));
        ( void )close( wfd[ 0 ] );
        ( void )close( wfd[ 1 ] );
        ( void )close( efd[ 0 ] );
        ( void )close( efd[ 1 ] );
        ( void )close( rfd[ 0 ] );
        ( void )close( rfd[ 1 ] );
        
        return( -1 );
        
    default:
        break;
    }
    
    /* free the char array */
    for ( i = 0; i <= ac; i++ ) {
        free( av[ i ] );
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    ( void )close( wfd[ 0 ] );
    ( void )close( rfd[ 1 ] );
    ( void )close( efd[ 1 ] );

    /* write authorization reference to setuid tool, to verify user */
    if ( write( wfd[ 1 ], &extAuth, sizeof( extAuth )) != sizeof( extAuth )) {
        NSLog( @"write failed: %s", strerror( errno ));
        ( void )close( wfd[ 1 ] );
        
        return( -1 );
    }
    
    if ( fcntl( rfd[ 0 ], F_SETFL, O_NONBLOCK ) < 0 ) {
        [ rae reportError: @"fnctl O_NONBLOCK: %s\n", strerror( errno ) ];
        return( -1 );
    }

    r = fdopen( rfd[ 0 ], "r" );
    setvbuf( r, NULL, _IONBF, 0 );
    
    /* save some cpu cycles by using select rather than pure non-blocking reads */
    for ( ;; ) {
        struct timeval		tv;

        tv.tv_sec = 0;
        tv.tv_usec = 1;
        
        memset( buf, '\0', sizeof( buf ));
        
        FD_ZERO( &readmask );
        FD_SET( rfd[ 0 ], &readmask );
        FD_SET( efd[ 0 ], &readmask );
        
        switch ( select(( MAX( efd[ 0 ], rfd[ 0 ] ) + 1 ), &readmask, NULL, NULL, &tv )) {
        case -1:
	    if ( lcontroller != nil ) {
		[ lcontroller toolError: "select failed\n" ];
		[ lcontroller addTextToLog: @"select failed\n" color: [ NSColor redColor ]
				    displayInPane: YES ];
	    }
            return( -1 );
            
        case 0:
            continue;
            
        default:
            break;
        }
        
        if ( ! FD_ISSET( rfd[ 0 ], &readmask )
                        && ! FD_ISSET( efd[ 0 ], &readmask )) {
            break;
        }
        
        if ( FD_ISSET( rfd[ 0 ], &readmask )) {
            BOOL		display = NO;

            while ( fgets(( char * )buf, MAXPATHLEN, r ) != NULL ) {
                if ( *buf == '%' && lcontroller != nil ) {
                    [ lcontroller updateProgress:
                            [ NSString stringWithUTF8String: ( char * )buf ]];
                } else if ( command == KTCHECK ) {
                    display = YES;
                }
                
                if ( strlen( buf ) > 0 && lcontroller != nil ) {
                    [ lcontroller addTextToLog: [ NSString stringWithFormat: @"%s", buf ]
                                    color: [ NSColor blackColor ]
                                    displayInPane: display ];
                }
            }
        }
        if ( FD_ISSET( efd[ 0 ], &readmask )) {
            /* if anything's printed to stderr, open the log drawer */
            if (( rr = read( efd[ 0 ], ( char * )buf, MAXPATHLEN )) <= 0 ) {
                break;
            }
            buf[ rr ] = '\0';
	    
            /* getpass(3) puts the prompt on stderr if /dev/tty isn't available */
            if ( command == LCREATE
		    && strcmp( "password:", buf ) == 0
		    && lcontroller != nil ) {
                /* write the password to the tool's stdin */
                NSString        *pass = [ lcontroller retrievePassword ];
                const char      *c_pass = [ pass UTF8String ];

                if ( pass && strlen( c_pass ) > 0 ) {
                    if ( write( wfd[ 1 ], c_pass, strlen( c_pass )) != strlen( c_pass )) {
                        [ rae reportError: @"write: %s", strerror( errno ) ];
                        break;
                    }
                }
                if ( write( wfd[ 1 ], "\n", strlen( "\n" )) != strlen( "\n" )) {
                    [ rae reportError: @"write: %s", strerror( errno ) ];
                    break;
                }
                
                [ lcontroller addTextToLog: [ NSString stringWithUTF8String: buf ]
                                color: [ NSColor blackColor ]
                                displayInPane: NO ];
            } else {
                /* display error in progress pane */
		if ( lcontroller != nil ) {
		    [ lcontroller addTextToLog: [ NSString stringWithUTF8String: buf ]
				    color: [ NSColor redColor ]
				    displayInPane: YES ];
		    [ lcontroller openDrawer ];
		}
            
                memset( errbuf, '\0', MAXPATHLEN );
                if ( strlen( buf ) >= MAXPATHLEN ) {
                    [ rae reportError: @"%s: too long", buf ];
                    break;
                }
                strcpy( errbuf, buf );
                
                had_error = 1;
            }
        }
    }
    if ( rr < 0 ) {
        NSLog( @"read returned < 0 (%d): %s\n", rr, strerror( errno ));
        exit( 2 );
    }

    ( void )close( wfd[ 1 ] );
    ( void )close( efd[ 0 ] );
    ( void )close( rfd[ 0 ] );
    
    pid = wait( &status );
    
    if ( av != NULL ) {
	free( av );
    }
    
    rc = WEXITSTATUS( status );
    
    switch ( command ) {
    case KTCHECK:
        if ( rc > 1 && had_error ) {
	    if ( lcontroller != nil ) {
		[ lcontroller toolError: errbuf ];
	    }
	    NSLog( @"error: %s", errbuf );
        }
        break;
        
    default:
        if ( rc != 0 && had_error ) {
	    if ( lcontroller != nil ) {
		[ lcontroller toolError: errbuf ];
	    }
	    NSLog( @"error: %s", errbuf );
        }
        break;
    }
    
    return( rc );
}

    int
authorize_and_exec( AuthorizationItem right,
                    int command,
                    NSArray *args )
{
    int				err;
    AuthorizationRef		authRef;
    AuthorizationRights		rights = { 1, &right };
    OSStatus			status;
    AuthorizationFlags		flags =
                                    kAuthorizationFlagDefaults |
                                    kAuthorizationFlagPreAuthorize |
                                    kAuthorizationFlagInteractionAllowed |
                                    kAuthorizationFlagExtendRights;
    status = AuthorizationCreate( 
                NULL,
                kAuthorizationEmptyEnvironment,
                kAuthorizationFlagDefaults,
                &authRef );
                
    if ( status != errAuthorizationSuccess ) {
        [ rae reportError: @"AuthorizationCreate failed: error %d\n", ( int )status ];
        return( -1 );
    }
    
    /* make user authenticate as an administrator */
    status = AuthorizationCopyRights(
                authRef,
                &rights,
                kAuthorizationEmptyEnvironment,
                flags,
                NULL );
                
    if ( status != errAuthorizationSuccess ) {
        [ rae reportError: @"AuthorizationCopyRights failed: %d\n", ( int )status ];
        AuthorizationFree( authRef, kAuthorizationFlagDefaults );
        return( -2 );
    }

    if (( err = exec_command( authRef, command, args )) != 0 ) {
        if ( command != KTCHECK ) {
            [ rae reportError: @"Executing command returned %d\n", err ];
        }
    }
    
    if (( status = AuthorizationFree( authRef, kAuthorizationFlagDefaults ))
                                                != errAuthorizationSuccess ) {
        [ rae reportError: @"AuthorizationFree failed: error %d\n", ( int )status ];
    }
    
    return( err );
}

@end
