/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RAAuthKiller.h"
#import "RCMStepController.h"
#import <Security/AuthorizationTags.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "authtools.h"

extern int		errno;
id			rcontroller;

@implementation RAAuthKiller

- ( id )init
{
    self = [ super init ];
                                            
    return( self );
}

- ( void )killTool: ( int )tool forController: ( id )controller
{
    AuthorizationItem		right = { "edu.umich.rmauthexec", 0, NULL, 0 };
    int				rc;
    
    rcontroller = nil;
    rcontroller = controller;
    
    rc = authorize_and_kill( right, tool );
                                
    if ( rc < 0 ) {
        [ rcontroller authorizationFailed ];
        return;
    }
                            
    [ rcontroller killFinishedWithStatus: rc ];
}

    int
killtool( AuthorizationRef authRef, int command )
{
    char			authpath[ MAXPATHLEN ], buf[ MAXPATHLEN ];
    char			*toolname;
    char			*argv[ 7 ] = { NULL, NULL, NULL,
                                                NULL, NULL, NULL, NULL };
    int				wfd[ 2 ], efd[ 2 ], rr, wr;
    int				status;
    pid_t			pid;
    
    AuthorizationExternalForm	extAuth;
    
    if ( pathfortool( "rmauthexec", authpath ) != 0 ) {
        syslog( LOG_ERR, "rmauthexec couldn't be found.");
        return( -1 );
    }

    toolname = nameforcmd( command );
    
    argv[ 0 ] = authpath;
    argv[ 1 ] = "-A";
    argv[ 2 ] = "KillProcess";
    argv[ 3 ] = "--";
    argv[ 4 ] = toolname;
    argv[ 5 ] = NULL;

    if ( AuthorizationMakeExternalForm( authRef, &extAuth ) != 0 ) {
        syslog( LOG_ERR, "Failed to make external auth form" );
        return( -1 );
    }
    
    if ( pipe( wfd ) != 0 ) {
        syslog( LOG_ERR, "pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( efd ) != 0 ) {
        syslog( LOG_ERR, "pipe failed: %s", strerror( errno ));
        return( -1 );
    }
        
    switch ( fork()) {
    case 0:
        ( void )close( wfd[ 1 ] );
        if ( dup2( wfd[ 0 ], 0 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( efd[ 0 ] );
        if ( dup2( efd[ 1 ], 2 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( efd[ 1 ] );
        ( void )close( wfd[ 0 ] );
        
        execve( argv[ 0 ], argv, NULL );
        syslog( LOG_ERR, "execve: %s: %s", argv[ 0 ], strerror( errno ));
        _exit( 2 );
        
    case -1:
        syslog( LOG_ERR, "fork() failed: %s", strerror( errno ));
        ( void )close( wfd[ 0 ] );
        ( void )close( wfd[ 1 ] );
        return( -1 );
        
    default:
        break;
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    ( void )close( wfd[ 0 ] );
    ( void )close( efd[ 1 ] );
    
    if (( wr = write( wfd[ 1 ], &extAuth, sizeof( extAuth ))) != sizeof( extAuth )) {
        syslog( LOG_ERR, "write failed: %s", strerror( errno ));
        if ( close( wfd[ 1 ] ) < 0 ) {
            syslog( LOG_ERR, "close failed: %s", strerror( errno ));
            return( -1 );
        }
        
        return( -1 );
    }

    for ( ;; ) {
        if (( rr = read( efd[ 0 ], buf, MAXPATHLEN )) <= 0 ) break;
        buf[ rr ] = '\0';
        [ rcontroller toolError: buf ];
        [ rcontroller addTextToLog: [ NSString stringWithUTF8String: buf ]
                        color: [ NSColor redColor ]
                        displayInPane: YES ];
        memset( buf, '\0', strlen( buf ));
    }
    if ( rr < 0 ) {
        fprintf( stderr, "read returned < 0 (%d): %s\n", rr, strerror( errno ));
        return( -1 );
    }
    ( void )close( wfd[ 1 ] );
    ( void )close( efd[ 0 ] );

    pid = wait( &status );
    return( WEXITSTATUS( status ));
}

    int
authorize_and_kill( AuthorizationItem right,
                    int command )
{
    int				err;
    AuthorizationRef		authRef;
    AuthorizationRights		rights = { 1, &right };
    OSStatus			status;
    AuthorizationFlags		flags =
                                    kAuthorizationFlagDefaults |
                                    kAuthorizationFlagPreAuthorize |
                                    kAuthorizationFlagInteractionAllowed |
                                    kAuthorizationFlagExtendRights;
                                
    status = AuthorizationCreate( 
                NULL,
                kAuthorizationEmptyEnvironment,
                kAuthorizationFlagDefaults,
                &authRef );
                
    if ( status != errAuthorizationSuccess ) {
        syslog( LOG_ERR, "AuthorizationCreate failed." );
        return( -1 );
    }
    
    status = AuthorizationCopyRights(
                authRef,
                &rights,
                kAuthorizationEmptyEnvironment,
                flags,
                NULL );
                
    if ( status != errAuthorizationSuccess ) {
        syslog( LOG_ERR, "AuthorizationCopyRights failed." );
        return( -1 );
    }
    
    if (( err = killtool( authRef, command )) != 0 ) {
        syslog( LOG_ERR, "attempting to execute command returned %d", err );
        return( err );
    }
    
    return( err );
}
@end
