/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#include <CoreFoundation/CoreFoundation.h>
#include <Security/Authorization.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <errno.h>
#include <dirent.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <utime.h>

#include "rmauthexec.h"
#include "automator.h"
#include "copy.h"
#include "defaults.h"
#include "kvalid.h"
#include "makedir.h"
#include "pidforname.h"
#include "prepost.h"
#include "rummgmt.h"
#include "selfrepair.h"

#define RADMIND			"/usr/local/sbin/radmind"
#define RADMIND_DIR		"/var/radmind"
#define RADMIND_CONFIG_FILE	"/var/radmind/config"
#define RADMIND_CLIENT_DIR	"/var/radmind/client"
#define RADMIND_CLIENT_KFILE	"/var/radmind/client/command.K"
#define RADMIND_KFILE_DIR	"/var/radmind/command"
#define RADMIND_TUTORIAL_KFILE	"/var/radmind/command/base.K"
#define RADMIND_TRANS_DIR	"/var/radmind/transcript"
#define RADMIND_FILE_DIR	"/var/radmind/file"
#define RADMIND_TMP_DIR		"/var/radmind/tmp"
#define RADMIND_TMP_TRANS_DIR	"/var/radmind/tmp/transcript"
#define RADMIND_TMP_FILE_DIR	"/var/radmind/tmp/file"
#define RADMIND_CERT_DIR	"/var/radmind/cert"
#define RADASSIST_TMPDIR	"/var/radmind/server_file_backup"
#define	FILEDIR			"/var/radmind/file"
#define FILE_TMPDIR		"/var/radmind/tmp/file"
#define LOCAL_SBIN		"/usr/local/sbin"
#define RADMIND_STARTUP_ITEM	"/Library/StartupItems/RadmindServer"
#define RADMIND_STARTUP_SCRIPT	"/Library/StartupItems/RadmindServer/RadmindServer"
#define RADMIND_STARTUP_PLIST	"/Library/StartupItems/RadmindServer/StartupParameters.plist"

#define AUTHMODE ( S_ISUID | S_IRUSR | S_IXUSR | \
			 S_IRGRP | S_IXGRP | \
			 S_IROTH | S_IXOTH )

extern int	errno;
char		*radmind_path = "/var/radmind";
char		*tmpdir = "/tmp";
uid_t		uid;

/* actions */
int		ra_setdefaults( int, char *[] );
int		ra_installhooks( int, char *[] );
int		ra_setloginhook( int, char *[] );
int		ra_rmloginhook( int, char *[] );
int		ra_kill( int, char *[] );
int		ra_installscripts( int, char *[] );
int		ra_setlogouthook( int, char *[] );
int		ra_rmlogouthook( int, char *[] );
int		ra_installperiodic( int, char *[] );
int		ra_serversetup( int, char *[] );
int		ra_exec( int, char *[] );
int		ra_copycerts( int, char *[] );
int		ra_rmcert( int, char *[] );
int		ra_installcert( int, char *[] );
int		ra_installprepost( int, char *[] );
int		ra_rmprepost( int, char *[] );
int		ra_copyprepost( int, char *[] );
int		ra_enablerum( int, char *[] );
int		ra_disablerum( int, char *[] );

struct ra_action	actions[] = {
    { RA_SETDEFAULTS,		ra_setdefaults },
    { RA_INSTALLHOOKS, 		ra_installhooks },
    { RA_SETLOGINHOOK, 		ra_setloginhook },
    { RA_RMLOGINHOOK, 		ra_rmloginhook },
    { RA_KILLPG, 		ra_kill },
    { RA_INSTALLSCRIPTS, 	ra_installscripts },
    { RA_SETLOGOUTHOOK, 	ra_setlogouthook },
    { RA_RMLOGOUTHOOK,		ra_rmlogouthook },
    { RA_INSTALLPERIODIC,	ra_installperiodic },
    { RA_SERVERSETUP,		ra_serversetup },
    { RA_EXEC,			ra_exec },
    { RA_COPYCERTS,		ra_copycerts },
    { RA_RMCERT,		ra_rmcert },
    { RA_INSTALLCERT,		ra_installcert },
    { RA_INSTALLPREPOST,	ra_installprepost },
    { RA_RMPREPOST,		ra_rmprepost },
    { RA_COPYPREPOST,		ra_copyprepost },
    { RA_ENABLERUM,		ra_enablerum },
    { RA_DISABLERUM,		ra_disablerum },
};

    int
ra_setdefaults( int ac, char *av[] )
{
    if ( seteuid( uid ) != 0 ) { /* become GUI user temporarily */
	fprintf( stderr, "seteuid %d: %s\n", uid, strerror( errno ));
	return( 2 );
    }
    copydefaults(); /* copy the defaults to root domain */
    setuid( 0 );

    return( 0 );
}

    int
ra_installhooks( int ac, char *av[] )
{
    installhooks( av[ 0 ] );
    return( 0 );
}

    int
ra_setloginhook( int ac, char *av[] )
{
    sethook( "LoginHook", "/etc/login.hook" );
    return( 0 );
}

    int
ra_rmloginhook( int ac, char *av[] )
{
    sethook( "LoginHook", NULL );
    return( 0 );
}

    int
ra_kill( int ac, char *av[] )
{
    pid_t		pid;
    int			rc;

    if (( pid = pidforname( av[ 0 ] )) > 1 ) {
	if (( rc = kill( pid, SIGINT )) < 0 ) {
	    fprintf( stderr, "kill %d: %s\n", pid, strerror( errno ));
	    return( 2 );
	}
	fprintf( stderr, "Cancelled %s (pid %d)\n", av[ 0 ], pid );
    }

    return( 2 );
}

    int
ra_installscripts( int ac, char *av[] )
{
    installumscripts( av[ 0 ] );
    return( 0 );
}

    int
ra_setlogouthook( int ac, char *av[] )
{
    sethook( "LogoutHook", "/etc/logout.hook" );
    return( 0 );
}

    int
ra_rmlogouthook( int ac, char *av[] )
{
    sethook( "LogoutHook", NULL );
    return( 0 );
}

    int
ra_installperiodic( int ac, char *av[] )
{
    int		ptype = strtol( av[ 0 ], NULL, 10 );
    
    if ( ac != 2 ) {
	fprintf( stderr, "Incorrect arguments\n" );
	return( 2 );
    }

    installperiodic( ptype, av[ 1 ] );
    return( 0 );
}

    int
ra_serversetup( int ac, char *av[] )
{
    int		cfd;
    char	*sscriptpath, *splistpath; 

    if ( ac != 2 ) {
	fprintf( stderr, "Incorrect arguments\n" );
	return( 2 );
    }
    sscriptpath = av[ 0 ];
    splistpath = av[ 1 ];

    makedir( RADMIND_KFILE_DIR );
    makedir( RADMIND_FILE_DIR );
    makedir( RADMIND_TRANS_DIR );
    makedir( RADMIND_TMP_DIR );
    makedir( RADMIND_TMP_TRANS_DIR );
    makedir( RADMIND_TMP_FILE_DIR );
    makedir( "/Library/StartupItems" );

    if (( cfd = open( RADMIND_CONFIG_FILE, O_CREAT | O_RDWR | O_EXCL, 0600 ))
		< 0 ) {
	if ( errno != EEXIST ) {
	    fprintf( stderr, "create %s: %s\n", RADMIND_CONFIG_FILE,
						strerror( errno ));
	    exit( errno );
	}
    } else {
	if ( write( cfd, "*\t\tbase.K\n",
		strlen( "*\t\tbase.K\n" )) != strlen( "*\t\tbase.K\n" )) {
	    fprintf( stderr, "write: %s\n", strerror( errno ));
	    exit( errno );
	}
	( void )close( cfd );
    }

    if (( cfd = open( RADMIND_TUTORIAL_KFILE,
			O_CREAT | O_RDWR | O_EXCL, 0600 )) < 0 ) {
	if ( errno != EEXIST ) {
	    fprintf( stderr, "create %s: %s\n", RADMIND_TUTORIAL_KFILE,
						strerror( errno ));
	    exit( errno );
	}
    } 
    ( void )close( cfd );

    makedir( RADMIND_STARTUP_ITEM ); 
    if ( chown( RADMIND_STARTUP_ITEM, 0, 0 ) < 0 ) {
	fprintf( stderr, "chown %s: %s\n", sscriptpath, strerror( errno ));
	exit( errno );
    }
    if ( chmod( RADMIND_STARTUP_ITEM, 0755 ) < 0 ) {
	fprintf( stderr, "chmod %s: %s\n", sscriptpath, strerror( errno ));
	exit( errno );
    }

    copy( sscriptpath, RADMIND_STARTUP_SCRIPT, 0755 );
    if ( chown( sscriptpath, 0, 0 ) < 0 ) {
	fprintf( stderr, "chown %s: %s\n", sscriptpath, strerror( errno ));
	exit( errno );
    }
    if ( chmod( sscriptpath, 0755 ) < 0 ) {
	fprintf( stderr, "chmod %s: %s\n", sscriptpath, strerror( errno ));
	exit( errno );
    }

    copy( splistpath, RADMIND_STARTUP_PLIST, 0644 );
    if ( chown( splistpath, 0, 0 ) < 0 ) {
	fprintf( stderr, "chown %s: %s\n", splistpath, strerror( errno ));
	exit( errno );
    }
    if ( chmod( splistpath, 0644 ) < 0 ) {
	fprintf( stderr, "chmod %s: %s\n", splistpath, strerror( errno ));
	exit( errno );
    }

    return( 0 );
}

    int
ra_exec( int ac, char *av[] )
{
    int			rr, rc, status;
    pid_t		pid;
    struct rlimit	rl;

    for ( rr = 0; rr < ( ac - 2 ); rr++ ) {
	if ( strcmp( av[ rr ], "-C" ) == 0 ) {
	    if ( access( RADMIND_CLIENT_KFILE, F_OK ) < 0 ) {
		fprintf( stderr, "No command file on client. "
		    "Please run the Radmind Assistant's Update feature. "
		    "Once you have finished updating, you may create new "
		    "loadsets." );
		exit( 2 );
	    } else if ( ntcheck( RADMIND_CLIENT_KFILE ) == 0 ) {
		fprintf( stderr, "The command file does not contain "
			    "a negative transcript. You cannot create a "
			    "new loadset without a negative transcript "
			    "in this client's command file. Please add a "
			    "negative transcript to this client's command "
			    "file on the server, and run the Updater "
			    "to get the new command file.\n" );
		exit( 2 );
	    }
	    break;
	} else if ( strcmp( av[ rr ], "-A" ) == 0 ) {
	    if (( rc = ptcheck( RADMIND_CLIENT_KFILE )) == 0 ) {
		fprintf( stderr, "The command file does not contain "
		    "a positive transcript. You cannot update this machine "
		    "without at least one postive transcript in your "
		    "loadset. Please verify that this client's command "
		    "file contains a postive transcript on the server, "
		    "then try updating again.\n" );
		exit( 2 );
	    } else if ( ntcheck( RADMIND_CLIENT_KFILE ) == 0 ) {
		fprintf( stderr, "The command file does not contain "
			    "a negative transcript. You cannot create a "
			    "new loadset without a negative transcript "
			    "in this client's command file. Please add a "
			    "negative transcript to this client's command "
			    "file on the server, and run the Updater "
			    "to get the new command file.\n" );
		exit( 2 );
	    }
	}
    }

    /* increase maximum number of open fds for long command files */
    rl.rlim_max = rl.rlim_cur = 1024;

    if ( setrlimit( RLIMIT_NOFILE, &rl ) < 0 ) {
	fprintf( stderr, "Using setrlimit to increase number of open files "
			"returned an error: %s\n", strerror( errno ));
	exit( 2 );
    }

    switch ( fork()) {
    case 0:
	execve( av[ 0 ], av, NULL );
	syslog( LOG_ERR, "execve %s failed: %s\n", av[0], strerror( errno ));
	fprintf( stderr, "execve %s failed: %s\n", av[0], strerror( errno ));
	fflush( stderr );
	_exit( 2 );

    case -1:
	fprintf( stderr, "fork failed: %s\n", strerror( errno ));
	exit( 2 );  

    default:
	break;
    }

    pid = wait( &status );
    return( WEXITSTATUS( status ));
}

    int
ra_copycerts( int ac, char *av[] )
{
    DIR			*d;
    struct dirent	*de;
    struct stat		st;
    struct utimbuf	ut;
    char		certdir[ MAXPATHLEN ];
    char		src[ MAXPATHLEN ], dst[ MAXPATHLEN ];

    if ( ac != 1 ) {
	fprintf( stderr, "Incorrect arguments\n" );
	return( 2 );
    }

    if ( snprintf( certdir, MAXPATHLEN, "%s/cert",
		radmind_path ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/cert: path too long\n", radmind_path );
	return( 2 );
    }
    
    if (( d = opendir( certdir )) == NULL ) {
	if ( errno == ENOENT ) {
	    /* valid case. make the dir */
	    makedir( certdir );
	    return( 0 );
	}
	
	fprintf( stderr, "opendir %s: %s\n", certdir, strerror( errno ));
	return( 2 );
    }

    while (( de = readdir( d )) != NULL ) {
	if ( de->d_name[ 0 ] == '.' ) {
	    /* don't copy any hidden files */
	    continue;
	}

	if ( snprintf( src, MAXPATHLEN, "%s/%s",
		certdir, de->d_name ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", certdir, de->d_name );
	    return( 2 );
	}
	if ( snprintf( dst, MAXPATHLEN, "%s/%s",
		av[ 0 ], de->d_name ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", av[ 0 ], de->d_name );
	    return( 2 );
	}

	if ( lstat( src, &st ) != 0 ) {
	    fprintf( stderr, "lstat %s: %s\n", src, strerror( errno ));
	    return( 2 );
	}
	if ( ! S_ISREG( st.st_mode )) {
	    /* only copy files */
	    continue;
	}

	copy( src, dst, 0600 );

	if ( chown( dst, uid, 0 ) != 0 ) {
	    fprintf( stderr, "chown %s %d: %s\n", dst, uid, strerror( errno ));
	    return( 2 );
	}

	ut.actime = st.st_atime;
	ut.modtime = st.st_mtime;
	if ( utime( dst, &ut ) != 0 ) {
	    fprintf( stderr, "utime %s: %s\n", dst, strerror( errno ));
	    return( 2 );
	}
    }

    return( 0 );
}

    int
ra_rmcert( int ac, char *av[] )
{
    if ( ac != 1 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 2 );
    }

    if ( unlink( av[ 0 ] ) != 0 ) {
	fprintf( stderr, "unlink %s: %s\n", av[ 0 ], strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

    int
ra_installcert( int ac, char *av[] )
{
    char	dst[ MAXPATHLEN ];
    char	*p;

    if ( ac != 1 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 2 );
    }

    if (( p = strrchr( av[ 0 ], '/' )) == NULL ) {
	fprintf( stderr, "%s: not a full path\n", av[ 0 ] );
	return( 2 );
    }
    p++;

    if ( snprintf( dst, MAXPATHLEN, "%s/cert/%s",
		radmind_path, p ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/cert/%s: path too long\n", radmind_path, p );
	return( 2 );
    }

    copy( av[ 0 ], dst, 0600 );
    if ( chown( dst, 0, 0 ) != 0 ) {
	fprintf( stderr, "chown %s root: %s\n", dst, strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

    int
ra_installprepost( int ac, char *av[] )
{
    char	dst[ MAXPATHLEN ];
    char	*p;

    if ( ac != 2 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 2 );
    }

    if (( p = strrchr( av[ 0 ], '/' )) == NULL ) {
	fprintf( stderr, "%s: not a full path\n", av[ 0 ] );
	return( 2 );
    }
    p++;

    if ( snprintf( dst, MAXPATHLEN, "%s/%s/%s",
		radmind_path, av[ 1 ], p ) >= MAXPATHLEN ) {
	fprintf( stderr, "%s/%s/%s: path too long\n",
		radmind_path, av[ 1 ], p );
	return( 2 );
    }

    copy( av[ 0 ], dst, 0600 );
    if ( chown( dst, 0, 0 ) != 0 ) {
	fprintf( stderr, "chown %s root: %s\n", dst, strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

    int
ra_rmprepost( int ac, char *av[] )
{
    if ( ac != 1 ) {
	fprintf( stderr, "Wrong arguments\n" );
	return( 2 );
    }

    if ( unlink( av[ 0 ] ) != 0 ) {
	fprintf( stderr, "unlink %s: %s\n", av[ 0 ], strerror( errno ));
	return( 2 );
    }

    return( 0 );
}

    int
ra_copyprepost( int ac, char *av[] )
{
    DIR			*d;
    struct dirent	*de;
    struct stat		st;
    struct utimbuf	ut;
    char		*types[ 3 ] = { "preapply", "postapply", NULL };
    char		*type;
    char		prepostdir[ MAXPATHLEN ];
    char		dstdir[ MAXPATHLEN ];
    char		src[ MAXPATHLEN ], dst[ MAXPATHLEN ];
    int			i;

    if ( ac != 1 ) {
	fprintf( stderr, "Incorrect arguments\n" );
	return( 2 );
    }

    for ( i = 0; types[ i ] != NULL; i++ ) {
	type = types[ i ];

	if ( snprintf( prepostdir, MAXPATHLEN, "%s/%s",
		    radmind_path, type  ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", radmind_path, type );
	    return( 2 );
	}
	if ( snprintf( dstdir, MAXPATHLEN, "%s/%s",
		    av[ 0 ], type ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: path too long\n", av[ 0 ], type );
	    return( 2 );
	}

	makedir( dstdir );
	if ( chown( dstdir, uid, 0 ) != 0 ) {
	    fprintf( stderr, "chown %d %s: %s\n",
		    uid, dstdir, strerror( errno ));
	}
	
	if (( d = opendir( prepostdir )) == NULL ) {
	    if ( errno == ENOENT ) {
		/* no scripts or script dir is valid. create the dir. */
		makedir( prepostdir );
		continue;
	    }

	    fprintf( stderr, "opendir %s: %s\n", prepostdir, strerror( errno ));
	    return( 2 );
	}

	while (( de = readdir( d )) != NULL ) {
	    if ( de->d_name[ 0 ] == '.' ) {
		/* don't copy any hidden files */
		continue;
	    }

	    if ( snprintf( src, MAXPATHLEN, "%s/%s",
		    prepostdir, de->d_name ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s/%s: path too long\n",
			prepostdir, de->d_name );
		return( 2 );
	    }
	    if ( snprintf( dst, MAXPATHLEN, "%s/%s/%s",
		    av[ 0 ], type, de->d_name ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s/%s/%s: path too long\n",
			av[ 0 ], type, de->d_name );
		return( 2 );
	    }

	    if ( lstat( src, &st ) != 0 ) {
		fprintf( stderr, "lstat %s: %s\n", src, strerror( errno ));
		return( 2 );
	    }
	    if ( ! S_ISREG( st.st_mode )) {
		/* only copy files */
		continue;
	    }

	    copy( src, dst, 0600 );

	    if ( chown( dst, uid, 0 ) != 0 ) {
		fprintf( stderr, "chown %s %d: %s\n",
			dst, uid, strerror( errno ));
		return( 2 );
	    }

	    ut.actime = st.st_atime;
	    ut.modtime = st.st_mtime;
	    if ( utime( dst, &ut ) != 0 ) {
		fprintf( stderr, "utime %s: %s\n", dst, strerror( errno ));
		return( 2 );
	    }
	}
    }

    return( 0 );
}

    int
ra_enablerum( int ac, char *av[] )
{
    int		rumtype = 0;
    int		rc;

    if ( ac != 2 ) {
	fprintf( stderr, "%s: wrong arguments\n", __PRETTY_FUNCTION__ );
	return( 2 );
    }

    if ( strcmp( av[ 0 ], LAUNCHD ) == 0 ) {
	rumtype = LAUNCHD_SYSTEM;
    } else if ( strcmp( av[ 0 ], STARTUPITEM ) == 0 ) {
	rumtype = PRELAUNCHD_SYSTEM;
    } else {
	fprintf( stderr, "%s: wrong arguments\n", __PRETTY_FUNCTION__ );
	return( 2 );
    }
	
    rc = rum_enable( rumtype, av[ 1 ] );

    return( rc );
}

    int
ra_disablerum( int ac, char *av[] )
{
    int		rumtype = 0;
    int		rc;

    if ( ac != 2 ) {
	fprintf( stderr, "%s: wrong arguments\n", __PRETTY_FUNCTION__ );
	return( 2 );
    }

    if ( strcmp( av[ 0 ], LAUNCHD ) == 0 ) {
	rumtype = LAUNCHD_SYSTEM;
    } else if ( strcmp( av[ 0 ], STARTUPITEM ) == 0 ) {
	rumtype = PRELAUNCHD_SYSTEM;
    } else {
	fprintf( stderr, "%s: wrong arguments\n", __PRETTY_FUNCTION__ );
	return( 2 );
    }
	
    rc = rum_disable( rumtype );

    return( rc );
}

    int
ptcheck( char *kfile )
{
    return( kfile_type_present( kfile, 'p' ));
}

    int
ntcheck( char *kfile )
{
    return( kfile_type_present( kfile, 'n' ));
}

    int
main( int argc, char *argv[] )
{
    const char			*rightname = "edu.umich.rmauthexec";
    AuthorizationRef		authref;
    AuthorizationExternalForm	extauth;
    AuthorizationItem   	right = { rightname, 0, NULL, 0 };
    AuthorizationRights 	rights = { 1, &right };
    AuthorizationFlags  	flags = kAuthorizationFlagDefaults |
                                        kAuthorizationFlagInteractionAllowed |
                                        kAuthorizationFlagExtendRights;

    struct stat			st;
    char			*action = NULL;
    extern char			*optarg;
    extern int			optind;
    int				rc = 0, err = 0;
    int				c, i, nactions;
    int				ra_preapply = 0, ra_postapply = 0;

    if ( argc < 2 ) {
        fprintf( stderr, "Usage: %s args\n", argv[ 0 ] );
        exit( 1 );
    }

    if (( rc = read( 0, &extauth, sizeof( extauth ))) != sizeof( extauth )) {
        syslog( LOG_ERR, "read %d bytes: %s", &extauth );
	exit( 2 );
    }

    if ( AuthorizationCreateFromExternalForm( &extauth, &authref ) != 0 ) {
	exit( 2 );
    }

    /* store the realuid for use when synchronizing defaults */
    uid = getuid();

    if ( geteuid() != 0 ) {
	exit( selfrepair( argc, argv, rightname, authref, extauth ));
    }
    if ( setuid( 0 ) != 0 ) {
	fprintf( stderr, "setuid root failed.\n" );
	exit( 2 );
    }

    if ( stat( argv[ 0 ], &st ) < 0 ) {
	fprintf( stderr, "stat %s failed: %s\n", argv[ 0 ], strerror( errno ));
	exit( errno );
    }
    if (( st.st_mode & S_IWUSR) || ( st.st_mode & S_IWGRP )
		|| ( st.st_mode & S_IWOTH )) {
	if ( chmod( argv[ 0 ], AUTHMODE ) < 0 ) {
	    fprintf( stderr, "chmod 4555 %s failed: %s\n", argv[ 0 ],
		strerror( errno ));
	}
    }

    if ( AuthorizationCopyRights( authref, &rights,
	    kAuthorizationEmptyEnvironment,
	    flags, NULL ) != 0 ) {
	fprintf( stderr, "AuthCopyRights failed.\n" );
	exit( 1 );
    }

    makedir( RADMIND_DIR );
    makedir( RADMIND_CLIENT_DIR );

    while (( c = getopt( argc, argv, "A:D:d:Pp" )) != EOF ) {
	switch ( c ) {
	case 'A':		/* action */
	    if ( strlen( optarg ) >= MAXPATHLEN ) {
		fprintf( stderr, "%s: too long\n", optarg );
		exit( 2 );
	    }
	    if (( action = strdup( optarg )) == NULL ) {
		fprintf( stderr, "strdup %s: %s\n", optarg, strerror( errno ));
		exit( 2 );
	    }
	    break;
	    
	case 'D':		/* radmind path */
	    radmind_path = optarg;
	    break;

	case 'd':		/* tmpdir for session */
	    tmpdir = optarg;
	    break;

	case 'P':		/* run post-apply scripts */
	    ra_postapply++;
	    break;

	case 'p':		/* run pre-apply scripts */
	    ra_preapply++;
	    break;

	default:
	    err++;
	    break;
	}
    }

    if ( err || action == NULL ) {
	fprintf( stderr, "Usage: %s args\n", argv[ 0 ] );
	exit( 1 );
    }

    if (( argc = ( argc - optind )) == 0 ) {
	fprintf( stderr, "Missing arguments\n" );
	exit( 1 );
    }
    argv += optind;

    openlog( argv[ 0 ], LOG_NDELAY | LOG_CONS, LOG_USER );

    if ( ra_preapply ) {
	if ( preapply( radmind_path, argv[ argc - 1 ] ) != 0 ) {
	    syslog( LOG_ERR, "Some pre-apply scripts failed.\n" );
	}
    }

    nactions = ( sizeof( actions ) / sizeof( actions[ 0 ] ));
    for ( i = 0; i < nactions; i++ ) {
	if ( strcmp( action, actions[ i ].ra_name ) == 0 ) {
	    break;
	}
    }
    free( action );
    if ( i >= nactions ) {
	fprintf( stderr, "%s unrecognized\n", action );
	exit( 1 );
    }

    rc = (*(actions[ i ].ra_func))( argc, argv );

    if ( ra_postapply ) {
	if ( postapply( radmind_path, argv[ argc - 1 ] ) != 0 ) {
	    syslog( LOG_ERR, "Some post-apply scripts failed.\n" );
	}
    }

    return( rc );

}
