/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#import "RAServerAuth.h"
#import "RAServerController.h"
#import "NSArray(CreateArgv).h"

#include <Security/AuthorizationTags.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "authtools.h"

@class RSMLoadsetManager;

extern int		errno;
static int              threadID = 0;
static AuthorizationRef rsmAuthRef = NULL;

@implementation RAServerAuth

+ ( void )connectWithPorts: ( NSArray * )ports
{
    NSAutoreleasePool		*pool = [[ NSAutoreleasePool alloc ] init ];
    NSConnection		*cnctnToController;
    RAServerAuth		*serverObject;
    NSLock                      *lock;
    
    cnctnToController = [ NSConnection connectionWithReceivePort:
                            [ ports objectAtIndex: 0 ]
                            sendPort: [ ports objectAtIndex: 1 ]];
                            
    serverObject = [[ self alloc ] init ];
    [ serverObject setThreadID: threadID ];
    
    lock = [[ NSLock alloc ] init ];
    [ lock tryLock ];
    threadID++;
    [ lock release ];
   
    [ cnctnToController setRootObject: serverObject ];
    [ (( id )[ cnctnToController rootProxy ] ) setServer: serverObject
                                    andThreadID: [ serverObject threadID ]];
    [ serverObject release ];
    
    if ( [ (( id )[ cnctnToController rootProxy ] )
	    respondsToSelector: @selector( firstServerInfoLoad ) ] ) {
	[ (( id )[ cnctnToController rootProxy ] ) firstServerInfoLoad ];
    }
    
    [[ NSRunLoop currentRunLoop ] run ];
    
    [ pool release ];
    [ NSThread exit ];
}

- ( void )setThreadID: ( int )ID
{
    _rsmThreadID = ID;
}

- ( int )threadID
{
    return( _rsmThreadID );
}

- ( id )init
{
    self = [ super init ];
    rascontroller = nil;
                                            
    return( self );
}

- ( oneway void )disconnect
{
    NSArray                     *connections = [ NSConnection allConnections ];
    NSConnection                *conn = nil;
    NSEnumerator                *en = [ connections objectEnumerator ];
    
    while (( conn = [ en nextObject ] ) != nil ) {
        if ( [[ conn rootObject ] isEqual: self ] ) {
            [ conn invalidate ];
            break;
        }
    }
}

- ( oneway void )executeCommand: ( int )cmd
            withArguments: ( NSArray * )args
            controller: ( id )controller
{
    AuthorizationItem		right = { "edu.umich.radmind.generic", 0, 
                                           NULL, 0 };
    int				rc;

    if ( rascontroller ) {
        rascontroller = nil;
    }
    rascontroller = controller;

    rc = [ self prepareAuthorizedCommand: cmd withRight: right
                    andArguments: args ];
    
    if ( rc == -2 ) {
        [ rascontroller authorizationFailedInThreadWithID: [ self threadID ]];
        return;
    }
                            
    [ rascontroller command: cmd finishedWithStatus: rc
                    inThread: [ self threadID ]];
}

- ( int )executeAuthorizedCommand: ( int )command
            withRef: ( AuthorizationRef )authRef
            andArguments: ( NSArray * )args;
{
    char			authpath[ MAXPATHLEN ], buf[ MAXPATHLEN ];
    char			**execargs;
    NSMutableArray		*av = nil;
    int				wfd[ 2 ], rfd[ 2 ], efd[ 2 ];
    int				status, ac, i;
    pid_t			pid;
    fd_set			readmask;
    FILE			*rf = NULL, *ef = NULL;
    
    AuthorizationExternalForm	extAuth;

    if ( pathfortool( "rsmauthtool", authpath ) != 0 ) {
        NSLog( @"rsmauthtool couldn't be found." );
        return( -1 );
    }
    
    av = [[ NSMutableArray alloc ] init ];
    [ av addObject: [ NSString stringWithUTF8String: authpath ]]; // XXXX
    [ av addObjectsFromArray: args ];

    if ( AuthorizationMakeExternalForm( authRef, &extAuth ) != 0 ) {
        NSLog( @"Failed to make external auth form" );
        return( -1 );
    }
    
    if ( pipe( wfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( rfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( efd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    
    if ( chdir( "/" ) != 0 ) {
        NSLog( @"chdir to / failed: %s\n", strerror( errno ));
        return( -1 );
    }
    
    if (( ac = [ av argv: &execargs ] ) == 0 ) {
	[ rascontroller toolError: "Nothing to execute."
                        fromThreadWithID: [ self threadID ]];
        /* XXXX cleanup */
	return( -1 );
    }
    [ av release ];
        
    switch (( pid = fork())) {
    case 0:
        ( void )close( wfd[ 1 ] );
        if ( dup2( wfd[ 0 ], 0 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            _exit( 2 );
        }
        
        ( void )close( rfd[ 0 ] );
        if ( dup2( rfd[ 1 ], 1 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            _exit( 2 );
        }
        setvbuf( stdout, NULL, _IONBF, 0 );
        ( void )close( efd[ 0 ] );
        if ( dup2( efd[ 1 ], 2 ) < 0 ) {
            syslog( LOG_ERR, "dup2 failed: %s", strerror( errno ));
            _exit( 2 );
        }
        ( void )close( efd[ 1 ] );
        ( void )close( wfd[ 0 ] );
        ( void )close( rfd[ 1 ] );
        
        if ( setpgid( getpid(), getpid()) < 0 ) {
            syslog( LOG_ERR, "setpgid failed: %s", strerror( errno ));
            _exit( 2 );
        }
        
        execve( execargs[ 0 ], execargs, NULL );
        syslog( LOG_ERR, "execve: %s: %s", execargs[ 0 ], strerror( errno ));
        _exit( 2 );
        
    case -1:
        syslog( LOG_ERR, "fork() failed: %s", strerror( errno ));
        ( void )close( wfd[ 0 ] );
        ( void )close( wfd[ 1 ] );
        ( void )close( efd[ 0 ] );
        ( void )close( efd[ 1 ] );
        ( void )close( rfd[ 0 ] );
        ( void )close( rfd[ 1 ] );
        
        return( -1 );
        
    default:
        break;
    }

    for ( i = 0; i <= ac; i++ ) {
        free( execargs[ i ] );
    }
    free( execargs );
    
    signal( SIGPIPE, SIG_IGN );
    
    [ rascontroller setCurrentCommandPID: pid threadID: [ self threadID ]];
    
    ( void )close( wfd[ 0 ] );
    ( void )close( rfd[ 1 ] );
    ( void )close( efd[ 1 ] );

    if ( write( wfd[ 1 ], &extAuth, sizeof( extAuth )) != sizeof( extAuth )) {
        syslog( LOG_ERR, "write failed: %s", strerror( errno ));
        ( void )close( wfd[ 1 ] );
        return( -1 );
    }
    
    ( void )close( wfd[ 1 ] );
    
    if ( fcntl( rfd[ 0 ], F_SETFL, O_NONBLOCK ) < 0 ) {
        [ rascontroller toolError:
            "Failed to set non-blocking I/O on read descriptor."
                        fromThreadWithID: [ self threadID ]];
        return( -1 );
    }
    if ( fcntl( efd[ 0 ], F_SETFL, O_NONBLOCK ) < 0 ) {
        NSLog( @"non-block: %s", strerror( errno ));
        return( -1 );
    }
    if (( rf = fdopen( rfd[ 0 ], "r" )) == NULL ) {
        NSLog( @"fdopen rfd[ 0 ]: %s", strerror( errno ));
        return( -1 );
    }
    if (( ef = fdopen( efd[ 0 ], "r" )) == NULL ) {
        NSLog( @"fdopen efd[ 0 ]: %s", strerror( errno ));
        return( -1 );
    }
    
    setvbuf( rf, NULL, _IONBF, 0 );
    setvbuf( ef, NULL, _IONBF, 0 );

    /* save some cpu cycles by using select rather than non-blocking reads */
    for ( ;; ) {
        struct timeval		tv;

        tv.tv_sec = 0;
        tv.tv_usec = 1;
        
        memset( buf, 0, sizeof( buf ));
        
        FD_ZERO( &readmask );
        FD_SET( rfd[ 0 ], &readmask );
        FD_SET( efd[ 0 ], &readmask );
        
        switch ( select(( MAX( rfd[ 0 ], efd[ 0 ] ) + 1 ),
                        &readmask, NULL, NULL, &tv )) {
        case -1:
            NSLog( @"select: %s", strerror( errno ));
            return( -1 );
            
        case 0:
            continue;
            
        default:
            break;
        }
        
        if ( ! FD_ISSET( rfd[ 0 ], &readmask ) && ! FD_ISSET( efd[ 0 ], &readmask )) {
            break;
        }
        
        if ( FD_ISSET( rfd[ 0 ], &readmask )) {
            NSString                *s = nil;
            
            
            while ( fgets(( char * )buf, MAXPATHLEN, rf ) != NULL ) {
                buf[ strlen( buf ) - 1 ] = '\0';
                
                s = [ NSString stringWithUTF8String: buf ];
                
                if ( [ s characterAtIndex: 0 ] == '%' ) {
                    [ rascontroller updateToolProgressWithString: s
                                threadID: [ self threadID ]];
                } else {
                    [ rascontroller updateDisplayWithString: s
                                threadID: [ self threadID ]];
                }
            }
            if ( feof( rf )) {
                break;
            }
        } else if ( FD_ISSET( efd[ 0 ], &readmask )) {
            if ( fgets(( char * )buf, MAXPATHLEN, ef ) != NULL ) {
                [ rascontroller toolError: buf fromThreadWithID: [ self threadID ]];
            }
        }
    }

    ( void )close( efd[ 0 ] );
    ( void )close( rfd[ 0 ] );

    pid = wait( &status );
    
    return( WEXITSTATUS( status ));
}

            
- ( int )prepareAuthorizedCommand: ( int )command
            withRight: ( AuthorizationItem )right
            andArguments: ( NSArray * )args
{
    int				err;
    AuthorizationRights		rights = { 1, &right };
    OSStatus			status;
    AuthorizationFlags		flags =
                                    kAuthorizationFlagDefaults |
                                    kAuthorizationFlagPreAuthorize |
                                    kAuthorizationFlagInteractionAllowed |
                                    kAuthorizationFlagExtendRights;
          
    if ( rsmAuthRef == NULL ) {
	status = AuthorizationCreate( 
		    NULL,
		    kAuthorizationEmptyEnvironment,
		    kAuthorizationFlagDefaults,
		    &rsmAuthRef );
		    
	if ( status != errAuthorizationSuccess ) {
	    NSLog( @"AuthorizationCreate failed: error %d", (int)status );
	    return( -1 );
	}
    }
    
    status = AuthorizationCopyRights(
		    rsmAuthRef,
		    &rights,
		    kAuthorizationEmptyEnvironment,
		    flags,
		    NULL );
		    
    if ( status != errAuthorizationSuccess ) {
	NSLog( @"AuthorizationCopyRights failed: error %d", (int)status );
	AuthorizationFree( rsmAuthRef, kAuthorizationFlagDefaults );
	return( -2 );
    }

    err = [ self executeAuthorizedCommand: command withRef: rsmAuthRef
                    andArguments: args ];

    //AuthorizationFree( authRef, kAuthorizationFlagDefaults );
    
    return( err );
}

@end
