#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "genplist.h"
#include "llist.h"

extern int			errno;

#define RSMItemName		CFSTR( "RadmindServerItemName" )
#define RSMItemPath		CFSTR( "RadmindServerItemPath" )
#define RSMItemRelativePath	CFSTR( "RadmindServerItemRelativePath" )
#define RSMItemType		CFSTR( "RadmindServerItemType" )
#define RSMItemSize		CFSTR( "RadmindServerItemSize" )
#define RSMItemMTime		CFSTR( "RadmindServerItemMTime" )
#define RSMItemHRMTime		CFSTR( "RadmindServerItemHRMTime" )
#define RSMItemParent		CFSTR( "RadmindServerItemParent" )

    void
readdirs( char *dirpath, CFMutableArrayRef array, char *base )
{
    CFMutableDictionaryRef	dict = NULL;
    CFStringRef			str;

    DIR				*d;
    struct dirent		*de;
    struct llist		*head = NULL;
    struct llist		*cur, *new;
    struct stat			st;
    struct tm			*tm;
    char			tmp[ MAXPATHLEN ];
    char			*p;
    int				len = strlen( dirpath );
    int				type;
    
    if (( d = opendir( dirpath )) == NULL ) {
	perror( dirpath );
	exit( 2 );
    }

    while (( de = readdir( d )) != NULL ) {
	if ( strcmp( de->d_name, "." ) == 0
		|| strcmp( de->d_name, ".." ) == 0 ) {
	    continue;
	}

	if ( de->d_name[ 0 ] == '.' ) {
	    continue;
	}

	if ( snprintf( tmp, MAXPATHLEN, "%s/%s", dirpath, de->d_name )
		>= MAXPATHLEN ) {
	    fprintf( stderr, "%s/%s: too long\n", dirpath, de->d_name );
	    exit( 2 );
	}

	new = ll_allocate( tmp );
	ll_insert( &head, new );
    }

    if ( closedir( d ) != 0 ) {
	perror( "closedir" );
	exit( 2 );
    }

    for ( cur = head; cur != NULL; cur = cur->ll_next ) {
	if ( lstat( cur->ll_name, &st ) != 0 ) {
	    fprintf( stderr, "stat %s: %s\n", cur->ll_name, strerror( errno ));
	    exit( 2 );
	}
	switch ( st.st_mode & S_IFMT ) {
	case S_IFREG:
	    str = CFSTR( "file" );
	    type = 'f';
	    break;

	case S_IFDIR:
	    str = CFSTR( "directory" );
	    type = 'd';
	    break;

	case S_IFLNK:
	    str = CFSTR( "symlink" );
	    type = 'l';
	    break;

	default:
	    str = CFSTR( "unknown" );
	    type = '?';
	    break;
	}

	dict = CFDictionaryCreateMutable( kCFAllocatorDefault, 0,
			&kCFTypeDictionaryKeyCallBacks,
			&kCFTypeDictionaryValueCallBacks );
	if ( dict == NULL ) {
	    fprintf( stderr, "CFDictionaryCreateMutable failed\n" );
	    exit( 2 );
	}

	/* add type */
	CFDictionaryAddValue( dict, RSMItemType, str );

	/* add item name */
	if (( p = strrchr( cur->ll_name, '/' )) == NULL ) {
	    p = cur->ll_name;
	} else {
	    p++;
	}
	str = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )p, ( CFIndex )strlen( p ),
			kCFStringEncodingUTF8, false );
	if ( str == NULL ) {
	    fprintf( stderr, "CFStringCreateWithBytes failed\n" );
	    exit( 2 );
	}
	CFDictionaryAddValue( dict, RSMItemName, str );
	CFRelease( str );

	/* add path relative to radmind_path */
	if ( strcmp( cur->ll_name, base ) == 0 ) {
	    str = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )p, ( CFIndex )strlen( p ),
			kCFStringEncodingUTF8, false );
	    if ( str == NULL ) {
		fprintf( stderr, "CFStringCreateWithBytes failed\n" );
		exit( 2 );
	    }
	} else {
	    len = strlen( base );
	    p = &cur->ll_name[ len ];
	    if ( ++p == NULL ) {
		p = cur->ll_name;
	    }

	    str = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )p, ( CFIndex )strlen( p ),
			kCFStringEncodingUTF8, false );
	    if ( str == NULL ) {
		fprintf( stderr, "CFStringCreateWithBytes failed\n" );
		exit( 2 );
	    }
	}
	CFDictionaryAddValue( dict, RSMItemRelativePath, str );
	CFRelease( str );

	/* convert path to CFStringRef, add to dictionary */
	str = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )cur->ll_name,
			( CFIndex )strlen( cur->ll_name ),
			kCFStringEncodingUTF8, false );
	if ( str == NULL ) {
	    fprintf( stderr, "CFStringCreateWithBytes %s failed\n",
			cur->ll_name );
	    exit( 2 );
	}
	CFDictionaryAddValue( dict, RSMItemPath, str );
	CFRelease( str );

	/* add numeric timestamps */
	str = CFStringCreateWithFormat( kCFAllocatorDefault, NULL,
			CFSTR( "%ld" ), st.st_mtime );
	if ( str == NULL ) {
	    fprintf( stderr, "CFStringCreateWithFormat %ld failed\n",
			st.st_mtime );
	    exit( 2 );
	}
	CFDictionaryAddValue( dict, RSMItemMTime, str );
	CFRelease( str );

	/* add human readable date */
	tm = localtime( &st.st_mtime );
	if ( strftime( tmp, MAXPATHLEN, "%b %e %Y %H:%M", tm ) == 0 ) {
	    fprintf( stderr, "strftime failed: %s\n", strerror( errno ));
	    exit( 2 );
	}
	str = CFStringCreateWithBytes( kCFAllocatorDefault,
			( const UInt8 * )tmp, ( CFIndex )strlen( tmp ),
			kCFStringEncodingUTF8, false );
	if ( str == NULL ) {
	    fprintf( stderr, "CFStringCreateWithBytes %s failed\n", tmp );
	    exit( 2 );
	}
	CFDictionaryAddValue( dict, RSMItemHRMTime, str );
	CFRelease( str );

	/* add parent, if necessary */
	str = CFStringCreateWithBytes( kCFAllocatorDefault,
		    ( const UInt8 * )dirpath, ( CFIndex )strlen( dirpath ),
		    kCFStringEncodingUTF8, false );
	if ( str == NULL ) {
	    fprintf( stderr, "CFStringCreateWithBytes %s failed\n",
		    dirpath );
	    exit( 2 );
	}
	CFDictionaryAddValue( dict, RSMItemParent, str );
	CFRelease( str );

	/* add size */
	if ( snprintf( tmp, MAXPATHLEN, "%lld", st.st_size ) >= MAXPATHLEN ) {
	    fprintf( stderr, "%lld: too long\n", st.st_size );
	    exit( 2 );
	}
	str = CFStringCreateWithBytes( kCFAllocatorDefault,
                        ( const UInt8 * )tmp, ( CFIndex )strlen( tmp ),
                        kCFStringEncodingUTF8, false );
        if ( str == NULL ) {
            fprintf( stderr, "CFStringCreateWithBytes %s failed\n", tmp );
            exit( 2 );
        }
	CFDictionaryAddValue( dict, RSMItemSize, str );
        CFRelease( str );

	/* add dictionary to the array */
	CFArrayAppendValue( array, dict );
	CFRelease( dict );

	if ( type == 'd' ) {
	    readdirs( cur->ll_name, array, base );
	}
    }

    ll_free( head );

    return;
}

    void
genplist( char *dirpath, char *plist, char *base )
{
    CFMutableArrayRef		array = NULL;
    CFWriteStreamRef		stream;
    CFStringRef			errorString;
    CFStringRef			pathRef;
    CFURLRef			url;

    if (( array = CFArrayCreateMutable( kCFAllocatorDefault,
			0, &kCFTypeArrayCallBacks )) == NULL ) {
	fprintf( stderr, "CFArrayCreateMutable failed\n" );
	exit( 2 );
    }

    readdirs( dirpath, array, base );

    pathRef = CFStringCreateWithCString( kCFAllocatorDefault,
		plist, kCFStringEncodingUTF8 );
    if ( pathRef == NULL ) {
	fprintf( stderr, "CFStringCreateWithCString failed\n" );
	exit( 2 );
    }

    url = CFURLCreateWithFileSystemPath( kCFAllocatorDefault, pathRef,
		kCFURLPOSIXPathStyle, false );
    if ( url == NULL ) {
	fprintf( stderr, "CFURLCreateWithFileSystemPath failed\n" );
	exit( 2 );
    }
    CFRelease( pathRef );

    if (( stream = CFWriteStreamCreateWithFile( kCFAllocatorDefault,
		url )) == false ) {
	fprintf( stderr, "CFWriteStreamCreateWithFile failed\n" );
	exit( 2 );
    }
    CFRelease( url );

    if ( CFWriteStreamOpen( stream ) == false ) {
	fprintf( stderr, "CFWriteStreamOpen failed\n" );
	exit( 2 );
    }	    

    if ( CFPropertyListWriteToStream(( CFPropertyListRef )array, stream,
		kCFPropertyListXMLFormat_v1_0, &errorString ) == 0 ) {
	fprintf( stderr, "CFPropertyListWriteToStream failed: " );
	CFShow( errorString );
	exit( 2 );
    }
    CFRelease( array );

    CFWriteStreamClose( stream );
    CFRelease( stream );
}
