#include <CoreFoundation/CoreFoundation.h>
#include <Security/Security.h>

#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>

#include "sessionrights.h"

#define AUTH_ADMIN		CFSTR( kAuthorizationRuleAuthenticateAsAdmin )
#define AUTH_ADMINGROUP		CFSTR( "admin" )
#define AUTH_RULE		CFSTR( "rule" )
#define AUTH_TIMEOUT		CFSTR( "timeout" )
#define AUTH_CLASS		CFSTR( "class" )
#define AUTH_USER		CFSTR( "user" )
#define AUTH_GROUP		CFSTR( "group" )
#define AUTH_COMMENT		CFSTR( "comment" )
#define AUTH_ALLOW_ROOT		CFSTR( "allow-root" )
#define AUTH_SHARED		CFSTR( "shared" )
#define RMCOMMENT		CFSTR( "Rights for working with Radmind" )

extern int		errno;

    int
getauthref( AuthorizationRef *ref )
{
    OSStatus		err;
    AuthorizationItem	right = { "edu.umich.radmind.generic", 0, NULL, 0 };
    AuthorizationRights	rights = { 1, &right };
    AuthorizationFlags	flags = ( kAuthorizationFlagInteractionAllowed |
				kAuthorizationFlagExtendRights );

    if ( *ref != NULL ) {
	return( 0 );
    }

    if (( err = AuthorizationCreate( NULL, NULL, kAuthorizationFlagDefaults,
	    &( *ref ))) != errAuthorizationSuccess ) {
	CFShow( CFSTR( "AuthorizationCreate failed" ));
	return( 0 );
    }
    if (( err = AuthorizationCopyRights( *ref, &rights, NULL, flags, NULL ))
	    != errAuthorizationSuccess ) {
	CFShow( CFSTR( "AuthorizationCopyRights failed" ));
	return( 0 );
    }

    return( 1 );
}

    int
getright( const char *rightname, CFDictionaryRef *dict )
{
    OSStatus		err;
    CFDictionaryRef	d = NULL;

    if (( err = AuthorizationRightGet( rightname, &d ))
		!= errAuthorizationSuccess && err != errAuthorizationDenied ) { 
	CFShow( CFSTR( "AuthorizationRightGet failed" ));
	return( 0 );
    }
    
    *dict = d;

    return( 1 );
}

/*
 * set the radmind rights in the authorization database.
 * includes workaround for rdar://problem/3446163,
 * in which AuthorizationRightSet releases the main
 * bundle too many times, causing crashes.
 * see http://developer.apple.com/technotes/tn2002/tn2095.html
 */
    int
setright( const char *rightname, int timeout )
{
    int			rc = 1;
    OSStatus		err;
    CFBooleanRef	allowroot = kCFBooleanTrue;
    CFBooleanRef	shared = kCFBooleanTrue;
    CFBundleRef		bundle;
    CFIndex		retaincount;
    CFNumberRef		num = CFNumberCreate( kCFAllocatorDefault,
					kCFNumberIntType, &timeout );
    CFStringRef         keys[ 6 ] = { AUTH_TIMEOUT, AUTH_CLASS, AUTH_GROUP,
					AUTH_COMMENT, AUTH_ALLOW_ROOT,
					AUTH_SHARED };
    CFTypeRef		values[ 6 ] = { num, AUTH_USER, AUTH_ADMINGROUP,
					RMCOMMENT, allowroot,
					shared }; 
    CFDictionaryRef	dict;
    AuthorizationRef	authref = NULL;

    if ( getauthref( &authref ) == 0 ) {
	CFRelease( num );
	return( 0 );
    }

    dict = CFDictionaryCreate( kCFAllocatorDefault, ( void * )keys,
		( void * )values, 6, &kCFCopyStringDictionaryKeyCallBacks,
		&kCFTypeDictionaryValueCallBacks );
    if ( dict == NULL ) {
	CFShow( CFSTR( "CFDictionaryCreate failed" ));
	AuthorizationFree( authref, kAuthorizationFlagDefaults );
	CFRelease( num );
	return( 0 );
    }

    /* workaround */
    bundle = CFBundleGetMainBundle();
    retaincount = CFGetRetainCount( bundle );
    CFRetain( bundle );

    if (( err = AuthorizationRightSet( authref, rightname, dict,
					NULL, bundle, NULL ))
					!= errAuthorizationSuccess ) {
	CFShow( CFSTR( "AuthorizationRightSet failed" ));
	rc = 0;
    }

    CFRelease( dict );
    CFRelease( num );
    AuthorizationFree( authref, kAuthorizationFlagDefaults );

    /* workaround */
    if ( CFGetRetainCount( bundle ) == retaincount ) {
	CFShow( CFSTR( "Working around rdar://problems/3446163" ));
    } else {
	CFRelease( bundle );
    }

    return( rc );
}

    int
removeright( const char *rightname )
{
    int			rc = 1;
    OSStatus            err;
    AuthorizationRef	authref = NULL;

    if ( getauthref( &authref ) == 0 ) {
	return( 0 );
    }

    if (( err = AuthorizationRightRemove( authref, rightname )) != noErr ) {
	CFShow( CFSTR( "AuthorizationRightRemove" ));
	rc = 0;
    }

    AuthorizationFree( authref, kAuthorizationFlagDefaults );

    return( rc );
}
