/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.
 */

#import "RTESearchField.h"

/* offsets for cell drawing */
#define CELL_START_OFFSET       25.0
#define CELL_END_OFFSET         55.0

static NSImage              *_left = nil, *_right = nil, *_contents = nil;

@implementation RTESearchField

- ( NSImage * )backgroundImage
{
    NSRect                  rect;
    float                   width, start, end;
    NSImage                 *img = nil;
    NSSize                  size;
    
    rect = [ self bounds ];
    width = rect.size.width;
    
    img = [[ NSImage alloc ] initWithSize: rect.size ];
    
    if ( img == nil ) {
        return( img );
    }
    
    [ img setFlipped: NO ];
    
    /* draw search field */
    [ img lockFocus ];
    
        /* left end cap */
    size = [ _leftEndCap size ];
    start = size.width;
    
    width -= size.width;
    
    [ _leftEndCap compositeToPoint: NSZeroPoint operation: NSCompositeSourceOver ];
    
        /* right end cap */
    size = [ _rightEndCap size ];
    width -= size.width;
    end = ( rect.size.width - size.width );
    
    [ _rightEndCap compositeToPoint: NSMakePoint( end, 0.0 )
                    operation: NSCompositeSourceOver ];
                    
        /* contents */
    size = [ _leftEndCap size ];
    [ _fieldContents setScalesWhenResized: YES ];
    
    /* set size to fill gap. -1 from height so the image is drawn correctly */
    [ _fieldContents setSize: NSMakeSize( width, NSHeight( rect ) - 1 ) ]; 
    [ _fieldContents compositeToPoint: NSMakePoint( size.width, 0 )
                    operation: NSCompositeSourceOver ];
    
        /* magnifying glass, offset slightly from start of image */
    [ _magnifyingGlass compositeToPoint: NSMakePoint( 4, 3 )
                    operation: NSCompositeSourceOver ];
    
    [ img unlockFocus ];
    
    return( img );
}

- ( void )setFocused: ( BOOL )focused
{
    _isFocused = focused;
}

- ( BOOL )isFocused
{
    return( _isFocused );
}

- ( id )init
{
    _completedField = nil;
    
    self = [ super init ];
    
    return( self );
}

- ( void )awakeFromNib
{
    if ( _left == nil ) {
        _left = [[ NSImage imageNamed: @"LeftEndCap.tiff" ] retain ];
    }
    if ( _right == nil ) {
        _right = [[ NSImage imageNamed: @"RightEndCap.tiff" ] retain ];
    }
    if ( _contents == nil ) {
        _contents = [[ NSImage imageNamed: @"SearchFieldContent.tiff" ] retain ];
    }
    
    _leftEndCap = _left;
    _rightEndCap = _right;
    _fieldContents = _contents;
    
    [ self setDrawsBackground: NO ];
    [ self setBordered: NO ];
    [ self setBezeled: NO ];
    [ self setFocused: NO ];
    
    _magnifyingGlass = [[ NSImage imageNamed: @"MagnifyingGlass.tiff" ] retain ];
    _completedField = [ self backgroundImage ];
    _cancelSearchImage = [[ NSImage imageNamed: @"cancelsearch.tiff" ] retain ];
}

- ( void )dealloc
{
    if ( _completedField != nil ) {
        [ _completedField release ];
        _completedField = nil;
    }
    if ( _left != nil ) {
        [ _left release ];
        _left = nil;
    }
    if ( _right != nil ) {
        [ _right release ];
        _right = nil;
    }
    if ( _contents != nil ) {
        [ _contents release ];
        _contents = nil;
    }
    if ( _magnifyingGlass != nil ) {
        [ _magnifyingGlass release ];
        _magnifyingGlass = nil;
    }
    if ( _cancelSearchImage != nil ) {
        [ _cancelSearchImage release ];
        _cancelSearchImage = nil;
    }
    
    [ super dealloc ];
}

- ( BOOL )becomeFirstResponder
{
    if ( [[ self textColor ] isEqual: [ NSColor disabledControlTextColor ]] ) {
        [ self setStringValue: @"" ];
        [ self setTextColor: [ NSColor controlTextColor ]];
    }

    [ self setFocused: YES ];
    [ self setNeedsDisplay: YES ];
    
    return( [ super becomeFirstResponder ] );
}

- ( BOOL )resignFirstResponder
{
    return( [ super resignFirstResponder ] );
}

- ( void )drawFocusRing
{
    NSRect              bounds = [ self bounds ];
    NSBezierPath        *bp = [ NSBezierPath bezierPath ];
    float               radius = (( NSHeight( bounds ) - 1 ) / 2 );
    
    /* make path around view */
    [ bp moveToPoint: NSMakePoint( radius, NSHeight( bounds )) ];
    [ bp lineToPoint: NSMakePoint( NSWidth( bounds ) - radius, NSHeight( bounds )) ];
    [ bp appendBezierPathWithArcWithCenter: NSMakePoint( NSWidth( bounds ) - radius, radius + 1 )
                                radius: radius
                                startAngle: 90.0
                                endAngle: -90.0
                                clockwise: YES ];
    [ bp lineToPoint: NSMakePoint( radius, 1 ) ];
    [ bp appendBezierPathWithArcWithCenter: NSMakePoint( radius, radius + 1 )
                                radius: radius
                                startAngle: -90.0
                                endAngle: 90.0
                                clockwise: YES ];
    
    /* draw ring on path */
    [ NSGraphicsContext saveGraphicsState ];
    NSSetFocusRingStyle( NSFocusRingOnly );
    [ bp fill ];
    [ bp removeAllPoints ];
    [ NSGraphicsContext restoreGraphicsState ];
}

- ( void )drawRect: ( NSRect )rect
{
    NSRect              bounds = [ self bounds ];
    
    if ( _completedField != nil ) {
        [ _completedField compositeToPoint: NSMakePoint( 0, NSHeight( bounds ))
                            operation: NSCompositeSourceOver ];
    }
    
    if ( [[ self target ] isSearching ] ) {
        [ _cancelSearchImage compositeToPoint:
                                NSMakePoint( NSWidth( bounds ) - 20, NSHeight( bounds ) - 3 )
                                operation: NSCompositeSourceOver ];
    }
    
    if ( [[ self window ] isKeyWindow ] && [ self isFocused ] ) {
        [ self drawFocusRing ];
    }
    
    [[ self cell ] drawWithFrame: NSMakeRect( CELL_START_OFFSET, 4,
                                        NSWidth( bounds ) - CELL_END_OFFSET,
                                        NSHeight( bounds ) - 6 )
                    inView: self ];
}

- ( void )setFrameSize: ( NSSize )size
{
    [ super setFrameSize: size ];
    
    [[ self cell ] setControlSize: NSSmallControlSize ];
    
    if ( _completedField != nil ) {
        [ _completedField release ];
    }
    
    _completedField = [ self backgroundImage ];
}

- ( void )selectText: ( id )sender
{
    NSText              *t = [[ self window ] fieldEditor: YES forObject: self ];
    NSText              *textObj;
    NSRect              bounds = [ self bounds ];
    float               offset;
    
    if ( [ t superview ] != nil ) {
        return;
    }
    
    [ self setFocused: YES ];

    textObj = [[ self cell ] setUpFieldEditorAttributes: t ];
    
    offset = 10;
    
    [[ self cell ] setDrawsBackground: YES ];
    [[ self cell ] selectWithFrame: NSMakeRect( CELL_START_OFFSET, 4,
                                                NSWidth( bounds ) - CELL_END_OFFSET,
                                                NSHeight( bounds ) - 6 )
                    inView: self
                    editor: textObj
                    delegate: self
                    start: 0
                    length: [[ self stringValue ] length ]];
    [[ self cell ] setDrawsBackground: NO ];
}

- ( void )keyDown: ( NSEvent * )e
{
    unichar             key = [[ e charactersIgnoringModifiers ] characterAtIndex: 0 ];
    
    if ( key == '\033' ) {  /* escape */
        if ( [[ self target ] isSearching ] ) {
            [[ self target ] cancelSearchMode ];
            return;
        }
    }
    
    [ super keyDown: e ];
}

- ( void )mouseDown: ( NSEvent * )e
{
    NSPoint             location = [ self convertPoint: [ e locationInWindow ] fromView: nil ];
    NSRect              magrect, cancelrect, bounds = [ self bounds ];
    BOOL                flipped = [ self isFlipped ];
    
    cancelrect = NSMakeRect( NSWidth( bounds ) - 20, 0, 20, NSHeight( bounds ));
    /* slight offset for magnifying glass image */
    magrect = NSMakeRect( 4, 3, 20, NSHeight( bounds ));
                        
    if ( NSMouseInRect( location, magrect, flipped )) {
        /* always make the menu appear in the same location */
        float           ox = ( NSMinX( bounds ) + 4 ), oy = ( NSHeight( bounds ) + 3 );
        NSPoint         pt = [ self convertPoint: NSMakePoint( ox, oy ) toView: nil ];
        NSEvent         *modEvent = [ NSEvent mouseEventWithType: [ e type ]
                                                location: pt
                                                modifierFlags: [ e modifierFlags ]
                                                timestamp: [ e timestamp ]
                                                windowNumber: [ e windowNumber ]
                                                context: [ e context ]
                                                eventNumber: [ e eventNumber ]
                                                clickCount: [ e clickCount ]
                                                pressure: [ e pressure ]];
                                                
        [ NSMenu popUpContextMenu: [ self menu ]
                    withEvent: modEvent
                    forView: self ];
    } else if ( NSMouseInRect( location, cancelrect, flipped ) && [[ self target ] isSearching  ] ) {
        [[ self target ] cancelSearchMode ];
        return;
    } else {
        NSText          *t = [[ self window ] fieldEditor: YES forObject: self ];
        NSText          *textObj;

        [ self setFocused: YES ];
        
        if ( [ t superview ] != nil ) {
            return;
        }
        
        textObj = [[ self cell ] setUpFieldEditorAttributes: t ];
        [[ self cell ] setDrawsBackground: YES ];
        [[ self cell ] editWithFrame: NSMakeRect( CELL_START_OFFSET, 4,
                                                    NSWidth( bounds ) - CELL_END_OFFSET,
                                                    NSHeight( bounds ) - 6 )
                        inView: self
                        editor: textObj
                        delegate: self
                        event: e ];
        [[ self cell ] setDrawsBackground: NO ];
    }
}

- ( void )textDidEndEditing: ( NSNotification * )notification
{
    [ self setFocused: NO ];
    
    if ( [[ self stringValue ] length ] == 0 ) {
        [ self setTextColor: [ NSColor disabledControlTextColor ]];
        [ self setStringValue: @"Search" ];
    }
    
    [ super textDidEndEditing: notification ];
}

/* set which cursor should be displayed in portions of the field */
- ( void )resetCursorRects
{
    NSRect              bounds = [ self bounds ];
       
    /* magnifying glass rect */
    [ self addCursorRect: NSMakeRect( 0, 0, CELL_START_OFFSET, NSHeight( bounds ))
                 cursor: [ NSCursor arrowCursor ]];
    
    if ( ! [[ self target ] isSearching ] ) {
        [ self addCursorRect: NSMakeRect( 0, 0, NSWidth( bounds ) + 2, NSHeight( bounds ))
                     cursor: [ NSCursor IBeamCursor ]];
    } else {
        [ self addCursorRect: NSMakeRect( 0, 0, NSWidth( bounds ) - 35, NSHeight( bounds ))
                     cursor: [ NSCursor IBeamCursor ]];
        
        [ self addCursorRect: NSMakeRect( NSWidth( bounds ) - 20, 0, 20, NSHeight( bounds ))
                     cursor: [ NSCursor arrowCursor ]];
    }
}

@end
