/*
Copyright (c) 2002 Regents of The University of Michigan.
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appears in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of The University
of Michigan not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission. This software is supplied as is without expressed or
implied warranties of any kind.

Research Systems Unix Group
The University of Michigan
c/o Wesley Craig
4251 Plymouth Road B1F2, #2600
Ann Arbor, MI 48105-2785
radmind@umich.edu
http://rsug.itd.umich.edu/software/radmind/
*/

#import <Security/AuthorizationTags.h>
#import "RXAuthRW.h"
#import "RXTranscript.h"
#import "NSMutableDictionary(RXAdditions).h"
#import "NSArray(CreateArgv).h"

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include "authtools.h"

#define TREAD	0
#define TWRITE	1
#define FSDIFF	2
#define TWHICH	3

extern int		errno;
static unsigned int     threadID = 0;

@implementation RXAuthRW

+ ( void )connectWithPorts: ( NSArray * )ports
{
    NSAutoreleasePool		*pool = [[ NSAutoreleasePool alloc ] init ];
    NSConnection		*cnctnToController;
    RXAuthRW			*serverObject;
    int                         tID;
    
    cnctnToController = [ NSConnection connectionWithReceivePort:
                            [ ports objectAtIndex: 0 ]
                            sendPort: [ ports objectAtIndex: 1 ]];
                            
    serverObject = [[ self alloc ] init ];
    tID = [ serverObject threadID ];
   
    [ (( RXTranscript * )[ cnctnToController rootProxy ] ) setServer: serverObject
                            threadID: tID ];
    [ cnctnToController setRootObject: serverObject ];
    //[ serverObject release ];
    
    [ (( RXTranscript * )[ cnctnToController rootProxy ] )
                            DOSetupCompleteInThreadWithID: tID ];

    [[ NSRunLoop currentRunLoop ] run ];
    
    [ serverObject release ];
    [ pool release ];
}

- ( void )setThreadID: ( int )ID
{
    _rteThreadID = ID;
}

- ( int )threadID
{
    return( _rteThreadID );
}

- ( id )init
{
    NSLock          *idLock;
    
    idLock = [[ NSLock alloc ] init ];
    [ idLock lock ];
    [ self setThreadID: threadID ];
    threadID++;
    [ idLock unlock ];
    [ idLock release ];
    idLock = nil;
    
    self = [ super init ];
    
    return( self );
}

/* invalidate connection from the server-side */
- ( void )disconnect
{
    NSArray                     *connections = [ NSConnection allConnections ];
    NSConnection                *conn = nil;
    int                         i;

    for ( i = 0; i < [ connections count ]; i++ ) {
        if (( conn = [ connections objectAtIndex: i ] ) == nil ) {
            continue;
        }

        if ( [[ conn rootObject ] isEqual: self ] ) {
            [ conn invalidate ];
            break;
        }
    }
}

- ( oneway void )executeAction: ( int )action arguments: ( NSArray * )args
                controller: ( RXTranscript * )rxt
{
    int				rc;
    NSArray                     *arguments = [ args copy ];
    
    _rteController = rxt;

    rc = [ self prepareAuthorizedTask: action withArguments: arguments
                    forTranscript: _rteController ];
    
    if ( rc == -2 ) {
        [ rxt authorizationFailedInThreadWithID: [ self threadID ]];
        return;
    }
    
    switch ( action ) {
    case TREAD:
    case FSDIFF:
        [ _rteController readFinishedFromThreadWithID: [ self threadID ]];
        break;
        
    case TWRITE:
    case TWHICH:
        [ _rteController writeFinishedFromThreadWithID: [ self threadID ]];
        break;
        
    default:
        break;
    }

    [ arguments release ];
}

    
- ( int )executeAuthorizedTask: ( int )action withArguments: ( NSArray * )args
            authRef: ( AuthorizationRef )authRef forTranscript: ( RXTranscript * )rxt
{
    FILE			*rfp;
    fd_set			readmask;
    char			authpath[ MAXPATHLEN ], buf[ MAXPATHLEN * 2 ];
    char			**execargs = NULL;
    int				wfd[ 2 ], rfd[ 2 ], efd[ 2 ], rr, ac, i;
    int				status;
    pid_t			pid;
    AuthorizationExternalForm	extAuth;
    NSArray                     *av;
    NSMutableArray              *transcriptLines = nil;
    NSMutableString		*twhichOutput = nil;
    NSString                    *path = nil;
    
    if ( pathfortool( CFSTR( "rteauthexec" ), authpath ) != 0 ) {
        syslog( LOG_ERR, "rteauthexec couldn't be found.");
        return( -1 );
    }
    
    av = [ NSArray arrayWithObject: [ NSString stringWithUTF8String: authpath ]];
    av = [ av arrayByAddingObjectsFromArray: args ];

    if (( ac = [ av argv: &execargs ] ) <= 0 ) {
        NSLog( @"invalid argument vector" );
        return( -1 );
    }

    if ( AuthorizationMakeExternalForm( authRef, &extAuth ) != 0 ) {
        NSLog( @"Failed to make external auth form" );
        return( -1 );
    }
    
    if ( pipe( wfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( rfd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    if ( pipe( efd ) != 0 ) {
        NSLog( @"pipe failed: %s", strerror( errno ));
        return( -1 );
    }
    
    /* move to rootpath */
    chdir( "/" );

    switch ( fork()) {
    case 0:
        ( void )close( wfd[ 1 ] );
        if ( dup2( wfd[ 0 ], 0 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( rfd[ 0 ] );
        if ( dup2( rfd[ 1 ], 1 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( efd[ 0 ] );
        if ( dup2( efd[ 1 ], 2 ) < 0 ) {
            NSLog( @"dup2 failed: %s", strerror( errno ));
            return( -1 );
        }
        ( void )close( efd[ 1 ] );
        ( void )close( wfd[ 0 ] );
        ( void )close( rfd[ 1 ] );
        
        execve( execargs[ 0 ], execargs, NULL );
        fprintf( stderr, "execve: %s: %s", execargs[ 0 ], strerror( errno ));
        fflush( stderr );
        _exit( 2 );
        
    case -1:
        NSLog( @"fork() failed: %s", strerror( errno ));
        ( void )close( wfd[ 0 ] );
        ( void )close( wfd[ 1 ] );
        
        return( -1 );
        
    default:
        break;
    }

    for ( i = 0; i <= ac; i++ ) {
        free( execargs[ i ] );
    }
    free( execargs );
    
    signal( SIGPIPE, SIG_IGN );
    
    ( void )close( wfd[ 0 ] );
    ( void )close( rfd[ 1 ] );
    ( void )close( efd[ 1 ] );

    if ( write( wfd[ 1 ], &extAuth, sizeof( extAuth )) != sizeof( extAuth )) {
        NSLog( @"write failed: %s", strerror( errno ));
        ( void )close( wfd[ 1 ] );
        return( -1 );
    }
    
    if ( action == FSDIFF || action == TWHICH ) {
        if (( rfp = fdopen( rfd[ 0 ], "r" )) == NULL ) {
            NSLog( @"fdopen %d failed: %s\n", rfd[ 0 ], strerror( errno ));
            return( -1 );
        }
        setbuf( rfp, NULL );
	
        FD_ZERO( &readmask );
	
        if ( transcriptLines == nil && action == FSDIFF ) {
            transcriptLines = [[ NSMutableArray alloc ] init ];
        } else if ( twhichOutput == nil && action == TWHICH ) {
	    twhichOutput = [[ NSMutableString alloc ]
				initWithCapacity: MAXPATHLEN ];
	}
        
        for ( ;; ) {
            FD_SET( rfd[ 0 ], &readmask );

            if ( select( rfd[ 0 ] + 1, &readmask, NULL, NULL, NULL ) < 0 ) {
                NSLog( @"select failed: %s\n", strerror( errno ));
                return( -1 );
            }

	    if ( ! FD_ISSET( rfd[ 0 ], &readmask )) {
		break;
	    }
	    
	    if ( fgets( buf, ( MAXPATHLEN * 2 ), rfp ) == NULL ) {
		if ( ferror( rfp )) {
		    NSLog( @"fgets: %s", strerror( errno ));
		    clearerr( rfp );
		}
		break;
	    }
	    if ( action == FSDIFF ) {
		[ transcriptLines addObject: [ NSMutableDictionary dictionaryForLine:
					[ NSString stringWithUTF8String: buf ]]];
		
		if ( [ transcriptLines count ] % 100 == 0 ) {
		    /* sending mutable objects between threads is risky */
		    [ rxt addLines: [ NSArray arrayWithArray: transcriptLines ]
			    fromThreadWithID: [ self threadID ]];
		    [ transcriptLines removeAllObjects ];
		}
	    } else if ( action == TWHICH ) {
		[ twhichOutput appendFormat: @"%s", buf ];
	    }
        }
        if ( action == FSDIFF && [ transcriptLines count ] ) {
            [ rxt addLines: [ NSArray arrayWithArray: transcriptLines ]
                            fromThreadWithID: [ self threadID ]];
            [ transcriptLines removeAllObjects ];
        }
    }
    
    for ( ;; ) {
        if (( rr = read( efd[ 0 ], buf, ( MAXPATHLEN * 2 ))) <= 0 ) break;
        buf[ rr ] = '\0';
	
#ifdef notdef
	/* getpass(3) puts the prompt on stderr if /dev/tty isn't available */
	if ( command == LCREATE && strcmp( "password:", buf ) == 0 ) {
	    NSString		*pass = [ rxt retrievePassword ];
	    const unsigned char	*c_pass = [ pass UTF8String ];
	    
	    if ( pass && ( strlen( c_pass ) > 0 )) {
		if ( write( wfd[ 1 ], c_pass, strlen( c_pass )) != strlen( c_pass )) {
		    [ rxt readError: "wrote wrong number of bytes"
			    fromThreadWithID: [ self threadID ]];
		    break;
		}
	    }
	    if ( write( wfd[ 1 ], "\n", strlen( "\n" )) != strlen( "\n" )) {
		[ rxt readError: "wrote wrong number of bytes"
			    fromThreadWithID: [ self threadID ]];
		break;
	    }
	    
	    /* XXX logging */
	} else {
	    [ rxt readError: buf fromThreadWithID: [ self threadID ]];
	    memset( buf, '\0', strlen( buf ));
	}
#endif notdef
	[ rxt readError: buf fromThreadWithID: [ self threadID ]];
	memset( buf, '\0', strlen( buf ));
    }
    if ( rr < 0 ) {
        NSLog( @"read returned < 0 (%d): %s\n", rr, strerror( errno ));
        return( -1 );
    }
    ( void )close( wfd[ 1 ] );
    ( void )close( rfd[ 0 ] );
    ( void )close( efd[ 0 ] );
    
    pid = wait( &status );
    
    switch ( action ) {
    case TREAD:
        path = [ av objectAtIndex: 5 ];
        [ rxt readTranscriptCopyAtPath: path threadID: [ self threadID ]];
	break;
    
    case FSDIFF:
        [ rxt addLines: transcriptLines fromThreadWithID: [ self threadID ]];
	break;
	
    case TWHICH:
	[ rxt displayTwhichOutput: [ NSString stringWithString: twhichOutput ]
		fromThreadWithID: [ self threadID ]];
	break;
    
    default:
	break;
    }
    
    if ( transcriptLines != nil ) {
        [ transcriptLines release ];
    }
    
    if ( twhichOutput != nil ) {
	[ twhichOutput release ];
    }
    
    return( WEXITSTATUS( status ));
}

- ( int )prepareAuthorizedTask: ( int )action withArguments: ( NSArray * )args
            forTranscript: ( RXTranscript * )rxt
{
    int				err;
    AuthorizationRef		authRef = NULL;
    AuthorizationItem		right = { "edu.umich.rteauthexec", 0, NULL, 0 };
    AuthorizationRights		rights = { 1, &right };
    OSStatus			status = 0;
    AuthorizationFlags		flags =
                                    kAuthorizationFlagDefaults |
                                    kAuthorizationFlagInteractionAllowed |
                                    kAuthorizationFlagExtendRights;
                   
    status = AuthorizationCreate( 
                NULL,
                kAuthorizationEmptyEnvironment,
                kAuthorizationFlagDefaults,
                &authRef );
                
    if ( status != errAuthorizationSuccess ) {
        NSLog( @"AuthorizationCreate failed: error %d.", status );
        goto AUTHORIZATION_FAILED;
    }
    
    status = AuthorizationCopyRights(
                authRef,
                &rights,
                kAuthorizationEmptyEnvironment,
                flags,
                NULL );
                
    if ( status != errAuthorizationSuccess ) {
        NSLog( @"AuthorizationCopyRights failed: error %d.", status );
        goto AUTHORIZATION_FAILED;
    }
    
    err = [ self executeAuthorizedTask: action withArguments: args
                    authRef: authRef forTranscript: rxt ];

    if (( status = AuthorizationFree( authRef, kAuthorizationFlagDefaults ))
                    != errAuthorizationSuccess ) {
        NSLog( @"AuthorizationFree failed: error %d", status );
    }
    
    if ( err != 0 ) {
        NSLog( @"attempting to execute command returned %d", err );
        return( err );
    }
    
    return( err );
    
AUTHORIZATION_FAILED:
    if ( authRef != NULL ) {
        status = AuthorizationFree( authRef, kAuthorizationFlagDefaults );
        if ( status != errAuthorizationSuccess ) {
            NSLog( @"AuthorizationFree failed: error %d.", status );
        }
    }
    
    return( -2 );
}

@end
