/*
Copyright (c) 2004 Regents of The University of Michigan.
All Rights Reserved.

Permission to use, copy, modify, and distribute this software and
its documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appears in all copies and
that both that copyright notice and this permission notice appear
in supporting documentation, and that the name of The University
of Michigan not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission. This software is supplied as is without expressed or
implied warranties of any kind.

Research Systems Unix Group
The University of Michigan
c/o Wesley Craig
4251 Plymouth Road B1F2, #2600
Ann Arbor, MI 48105-2785

http://rsug.itd.umich.edu/software/radmind
radmind@umich.edu
*/

#import "RXTranscript.h"
#import "RTEAuditor.h"
#import "RXPrefs.h"
#import "RTEInfoPanelController.h"

#import "NSString(RXAdditions).h"
#import "NSMutableDictionary(RXAdditions).h"
#import "RXTranscript-ConversionAdditions.h"
#import "NSImage-RTEExtensions.h"
#import "NSAttributedString-Ellipsis.h"
#import "UMWildcard.h"

#include <sys/param.h>
#include <fcntl.h>
#include <unistd.h>
#include "argcargv.h"
#include "code.h"
#include "ischild.h"
#include "pathcmp.h"

#define	TREAD	0
#define TWRITE	1
#define FSDIFF	2
#define TWHICH	3

@implementation RXTranscript

int		INITIAL_READ = 0;
int		searchtype = PATHSEARCH;
float           fontsize = 12.0;

    int
pathcompare( id ob1, id ob2, void *context )
{
    const char	    *c1 = [[ ob1 objectForKey: @"path" ] UTF8String ];
    const char	    *c2 = [[ ob2 objectForKey: @"path" ] UTF8String ];
    
    return( pathcmp(( const unsigned char * )c1,
		    ( const unsigned char * )c2 ));
}

- ( void )setTranscriptContents: ( NSMutableArray * )contents
{
    if ( parsedLines != nil ) {
        [[[ self undoManager ] prepareWithInvocationTarget: self ]
                setTranscriptContents: parsedLines ];
        [ parsedLines release ];
        parsedLines = nil;
    }
    
    if ( contents != nil ) {
        parsedLines = [ contents mutableCopy ];
    }
    
    [ tContentsTableView reloadData ];
}

- ( NSMutableArray * )transcriptContents
{
    return( parsedLines );
}

- ( void )setSearching: ( BOOL )searching
{
    _isSearching = searching;
}

- ( BOOL )isSearching
{
    return( _isSearching );
}

- ( void )setDelegate: ( id )delegate
{
    if ( _rteTranscriptDelegate != nil ) {
	[ _rteTranscriptDelegate release ];
	_rteTranscriptDelegate = nil;
    }
    
    if ( delegate ) {
	_rteTranscriptDelegate = [ delegate retain ];
    }
}

- ( id )delegate
{
    return( _rteTranscriptDelegate );
}

- ( void )setHelperThreadID: ( int )ID
{
    _rteThreadID = ID;
}

- ( int )helperThreadID
{
    return( _rteThreadID );
}

- ( void )setupThreadCommunication
{
    NSPort              *sendPort = nil, *recPort = nil;
    NSArray             *portArray = nil;
    
    recPort = [ NSPort port ];
    sendPort = [ NSPort port ];
    connectionToAuthRW = [[ NSConnection alloc ] initWithReceivePort: recPort
                                                sendPort: sendPort ];
    [ connectionToAuthRW setRootObject: self ];
    portArray = [ NSArray arrayWithObjects: sendPort, recPort, nil ];

    [ NSThread detachNewThreadSelector: @selector( connectWithPorts: )
                                        toTarget: [ RXAuthRW class ]
                                        withObject: portArray ];
}

- ( void )fontSizeChanged: ( id )sender
{
    fontsize = [[ NSUserDefaults standardUserDefaults ] floatForKey: @"RTETextFontSize" ];
    if ( fontsize > 18.0 || fontsize < 8.0 ) {
        fontsize = 12.0;
    }
    
    [ tContentsTableView reloadData ];
}

- ( id )init
{
    /* column view is buggy and useless in this version */
#ifdef HAS_COLUMN_VIEW
    NSNumber		*view = [[ NSUserDefaults standardUserDefaults ] objectForKey: @"TView" ];

    if ( view == nil )
        viewtype = 0;
    else
        viewtype = [ view intValue ];
#endif /* HAS_COLUMN_VIEW */
    
    _rteThreadID = -1;
    self = [ super init ];
    arw = nil;
    
    if ( self == nil ) {
        NSLog( @"self was nil!" );
        exit( 2 );
    }
    
    viewtype = RXLISTVIEW;
    mode = NORMAL;
    _isSearching = NO;
    
    [[ NSNotificationCenter defaultCenter ] addObserver: self
                                            selector: @selector( controlTextDidChange: )
                                            name:  NSControlTextDidChangeNotification
                                            object: nil ];
    [[ NSDistributedNotificationCenter defaultCenter ] addObserver: self
                                            selector: @selector( fontSizeChanged: )
                                            name:  RTEFontSizeChangedNotification
                                            object: nil ];
    
    lindex = 0;

    parsedLines = nil;
    _rteTranscriptDelegate = nil;
    searchResults = [[ NSMutableArray alloc ] init ];
    
    [ self setupThreadCommunication ];
    
    return( self );
}

- ( void )setServer: ( id )serverObject threadID: ( int )threadID
{
    [ serverObject setProtocolForProxy: @protocol( RXTranscriptAuthInterface ) ];
    [ serverObject retain ];
    
    arw = ( RXAuthRW <RXTranscriptAuthInterface> * )serverObject;
    [ self setHelperThreadID: threadID ];
}
            
#ifdef HAS_COLUMN_VIEW
- ( int )browser: ( NSBrowser * )sender numberOfRowsInColumn: ( int )column
{
    int		count = 0;

    if ( [ columnContents count ] > 0 ) {
        if ( [ columnContents count ] >= ( column + 1 )) {
//NSLog( @"%@", [ columnContents objectAtIndex: column ] );
            count = [[ columnContents objectAtIndex: column ] count ];
        }
    }
//NSLog( @"%d", count );

    return( count );
}

- ( void )browser: ( NSBrowser * )browser willDisplayCell: ( NSBrowserCell * )cell atRow: ( int )row column: ( int )column
{
    NSArray	*curColumnItems;
    NSImage	*cellIcon;
    NSString	*path;

    if ( [[ columnContents objectAtIndex: column ] count ] <= 0 ) {
        return;
    }
    
    curColumnItems = [ NSArray arrayWithArray: [ columnContents objectAtIndex: column ]];
    
    path = [[ curColumnItems objectAtIndex: row ] objectForKey: @"path" ];
    
    if ( path != nil ) {
        [ cell setStringValue: [ path lastPathComponent ]];
        
        if ( [[[ curColumnItems objectAtIndex: row ] objectForKey: @"type" ]
                    characterAtIndex: 0 ] != 'd' ) {
            [ cell setLeaf: YES ];
            cellIcon = [[[ NSWorkspace sharedWorkspace ]
                        iconForFileType:[ path pathExtension ]] retain ];
                        
            if ( !cellIcon ) {
                cellIcon = [[ self imageForType: [[[ curColumnItems objectAtIndex: row ]
                                                    objectForKey: @"type" ]
                                                    characterAtIndex: 0 ]] retain ];
            }
        } else {
            cellIcon = [[[ NSWorkspace sharedWorkspace ] iconForFile: @"/private" ] retain ];
        }
        
        [ cellIcon setScalesWhenResized: YES ];
        [ cellIcon setSize: NSMakeSize( 16.0, 16.0 ) ];
        [ cell setImage: cellIcon ];
        if ( cellIcon != nil ) [ cellIcon release ];
    } else {
        /* really if we've got a nil path we should die */
        [ cell setStringValue: @"err" ];
    }
}
#endif /* HAS_COLUMN_VIEW */

- ( void )textChanged: ( NSNotification * )aNotification
{
    if ( [[ addItemPathField stringValue ] length ] == 0 ) {
	[ addItemButton setEnabled: NO ];
    } else {
	[ addItemButton setEnabled: YES ];
    }
}

- ( IBAction )getAddItemSheet: ( id )sender
{
    if ( [ tWindow childWindows ] != nil ) {
	NSBeep();
	return;
    }
    
    [ addItemButton setEnabled: NO ];
    [ addItemPathField setStringValue: @"" ];
    
    [ NSApp beginSheet: addItemPanel
	    modalForWindow: tWindow
	    modalDelegate: self
	    didEndSelector: NULL
	    contextInfo: nil ];
}

/* choose item to add to transcript */
- ( IBAction )chooseItem: ( id )sender
{
    int			rc;
    NSOpenPanel		*op = [ NSOpenPanel openPanel ];
    
    [ op setTitle: @"Choose item to add to transcript" ];
    [ op setPrompt: @"Choose" ];
    [ op setAllowsMultipleSelection: NO ];
    [ op setCanChooseDirectories: YES ];
    
    /* undocumented method to display hidden files */
    if ( [ op respondsToSelector: @selector( _navView ) ] ) {
	/* this causes compiler warnings, but works if _navView is available */
	[[ op _navView ] setShowsHiddenFiles: YES ];
    }
    
    rc = [ op runModalForDirectory: nil file: nil types: nil ];
    
    switch ( rc ) {
    case NSOKButton:
	break;
	
    case NSCancelButton:
    default:
	return;
    }
    
    [ addItemPathField setStringValue: [[ op filenames ] objectAtIndex: 0 ]];
    [ addItemButton setEnabled: YES ];
}

- ( IBAction )addItem: ( id )sender
{
    NSString		*fsdiffOrigin = nil;
    NSString		*itemPath;
    BOOL		isDir;
    int			rc;
    
    if ( ! [[ NSFileManager defaultManager ] fileExistsAtPath:
					    [ addItemPathField stringValue ]
					    isDirectory: &isDir ] ) {
	NSRunAlertPanel( @"Error", @"%@: no such file or directory.",
		@"OK", @"", @"", [ addItemPathField stringValue ] );
	return;
    }
    
    if ( isDir ) {
	[ NSApp activateIgnoringOtherApps: YES ];
	rc = NSRunAlertPanel( @"What would you like to add to this transcript?",
	    @"You can add only the directory, or the directory and its contents",
	    @"Add Directory", @"Cancel", @"Add Directory and Contents" );
	
	switch ( rc ) {
	default:
	case NSAlertDefaultReturn:
	    [[ NSUserDefaults standardUserDefaults ] setObject:
					    [ NSNumber numberWithInt: 0 ]
						    forKey: @"FsdiffType" ];
	    break;
	    
	case NSAlertAlternateReturn:
	    return;
	
	case NSAlertOtherReturn:
	    [[ NSUserDefaults standardUserDefaults ] setObject:
					    [ NSNumber numberWithInt: 1 ]
						    forKey: @"FsdiffType" ];
	    break;
	}
    }
	    
    [ addItemPanel orderOut: nil ];
    [ NSApp endSheet: addItemPanel ];
    
    if (( fsdiffOrigin = [[ NSUserDefaults standardUserDefaults ]
	    objectForKey: @"RTEBasePath" ] ) == nil ||
	    [ fsdiffOrigin isEqualToString: @"/" ] ) {
	fsdiffOrigin = @"";
    }
    
    itemPath = [ fsdiffOrigin stringByAppendingString:
		    [ addItemPathField stringValue ]];

    [ self readFile: [ NSDictionary dictionaryWithObjectsAndKeys:
				[ NSNumber numberWithInt: FSDIFF ], @"action",
				itemPath, @"filea",
				@"", @"fileb", nil ]];
				
    [ tWindow setDocumentEdited: YES ];
}

- ( IBAction )cancelAddItem: ( id )sender
{
    [ addItemPanel orderOut: nil ];
    [ NSApp endSheet: addItemPanel ];
}

- ( void )deleteSelectedLines: ( id )sender
{
    NSMutableArray      *contents = [[ self transcriptContents ] mutableCopy ];
    
    [[ self undoManager ] setActionName: NSLocalizedString( @"Delete", @"Delete" ) ];
    
    if ( viewtype == RXCOLUMNVIEW ) {
#ifdef HAS_COLUMN_VIEW
        NSMutableArray	*tmp;
        NSNumber	*warn = [[ NSUserDefaults standardUserDefaults ]
                                    objectForKey: @"Warn" ];
        NSDictionary	*dict, *next;
        char		ftype;
        int		index, endpt;
        int		row = [ transcriptBrowser selectedRowInColumn:
                                [ transcriptBrowser selectedColumn ]];
                                
        if ( row < 0 ) return;
        
        tmp = [ columnContents objectAtIndex: [ transcriptBrowser selectedColumn ]];
        dict = [ tmp objectAtIndex: row ];
        next = [ tmp objectAtIndex: ( row + 1 ) ];
        ftype = [[ dict objectForKey: @"type" ] characterAtIndex: 0 ];
        index = atoi(( char * )[[ dict objectForKey: @"tindex" ] UTF8String ] );
        endpt = atoi(( char * )[[ next objectForKey: @"tindex" ] UTF8String ] );
        
        if ( ftype == 'd' && ( [ warn intValue ] || warn == nil )) {
            int		rc;
            
            rc = NSRunAlertPanel( @"Warning",
                    [ NSString stringWithFormat:
                    @"You are about to delete %d lines from the transcript. Do you want to continue?",
                        ( endpt - index ) ],
                    @"Delete", @"Don't Delete", @"" );
                    
            switch ( rc ) {
                case NSAlertDefaultReturn:
                    break;
                    
                default:
                case NSAlertAlternateReturn:
                    return;
            }
        }
        
        for ( ;; ) {
            if ( atoi(( char * )[[[ parsedLines objectAtIndex: index ]
                        objectForKey: @"tindex" ] UTF8String ] ) == endpt ) break;
                        
            [ parsedLines removeObjectAtIndex: index ];
        }
        
        [ columnContents removeAllObjects ];
        [ self updateIndices ];
        [ self arrangeColumn: 0 withOffset: 0 toOffset: 0 ];
        [ transcriptBrowser reloadColumn: 0 ];
#endif /* HAS_COLUMN_VIEW */
    } else {
        NSEnumerator		*lenum = [ tContentsTableView selectedRowEnumerator ];
        unsigned int		firstline = -1;
        id			linenum;
        int			selcount = [ tContentsTableView numberOfSelectedRows ];
        int			*sellines = NULL, *ssellines = NULL, i = 0;
        
        if ( lenum == nil ) return;
        
        sellines = ( int * )calloc( selcount, sizeof( int ));
        if ( sellines == NULL ) {
            perror( "malloc" );
            exit( 1 );
        }
        
        if ( mode == SEARCH ) {
            ssellines = ( int * )calloc( selcount, sizeof( int ));
            if ( ssellines == NULL ) {
                perror( "malloc" );
                exit( 1 );
            }
        }
        
        while (( linenum = [ lenum nextObject ] ) != nil ) {
            if ( mode == SEARCH ) {
                sellines[ i ] = strtoll(( char * )[[[ searchResults objectAtIndex:
                                    [ linenum intValue ]] objectForKey: @"tindex" ]
                                        UTF8String ], NULL, 10 );
                ssellines[ i ] = [ linenum intValue ];
            } else {
                sellines[ i ] = [ linenum intValue ];
            }
            if ( firstline == -1 ) firstline = [ linenum intValue ];
            i++;
        }

        [ tContentsTableView selectRow: *sellines byExtendingSelection: NO ];
        if ( mode == SEARCH ) {
            [ searchResults removeObjectsFromIndices: ( unsigned int * )ssellines
			    numIndices: selcount ];
        }
        [ contents removeObjectsFromIndices: ( unsigned int * )sellines
			    numIndices: selcount ];
        
        free( sellines );
        if ( ssellines != NULL ) {
            free( ssellines );
        }
        
        [ self setTranscriptContents: contents ];
        [ contents release ];
        [ self updateIndices ];

        if ( firstline >= [ ( mode == SEARCH ? searchResults : [ self transcriptContents ] ) count ] ) {
            firstline = ( [ ( mode == SEARCH ? searchResults : [ self transcriptContents ] ) count ] - 1 );
        }
        if ( firstline >= 0 ) {
            [ tContentsTableView selectRow: firstline
                            byExtendingSelection: NO ];
	    [ tContentsTableView scrollRowToVisible: firstline ];
        }
    }
    [ lineCountField setStringValue: [ NSString stringWithFormat: @"%d lines",
                [ ( mode == SEARCH ? searchResults : [ self transcriptContents ] ) count ]]];
                
    [ tWindow setDocumentEdited: YES ];
}

- ( IBAction )copy: ( id )sender
{
    if ( [[ tWindow firstResponder ] isEqual: tContentsTableView ] ) {
	NSPasteboard	*pb;
	NSMutableArray	*copiedlines = [[ NSMutableArray alloc ] init ];
	NSMutableArray	*modelines = nil;
	NSEnumerator	*en = [ tContentsTableView selectedRowEnumerator ];
	id			linenum = nil;
	
	while (( linenum = [ en nextObject ] ) != nil ) {
	    switch ( mode ) {
	    case SEARCH:
		modelines = searchResults;
		break;
	    case NORMAL:
		modelines = parsedLines;
		break;
	    }
	    [ copiedlines addObject:
			[ modelines objectAtIndex: [ linenum intValue ]]];
	}
	pb = [ NSPasteboard generalPasteboard ];
	[ pb declareTypes: [ NSArray arrayWithObject: RTETranscriptContentsPboardType ] owner: self ];
	[ pb setPropertyList: copiedlines forType: RTETranscriptContentsPboardType ];
	[ copiedlines release ];
    } else {
        if ( [[ tWindow firstResponder ] respondsToSelector: @selector( copy: ) ] ) {
            [[ tWindow firstResponder ] performSelector: @selector( copy: )
                                        withObject: sender
                                        afterDelay: 0.0 ];
        }
    }
}

- ( IBAction )cut: ( id )sender
{    
    if ( [[ tWindow firstResponder ] isEqual: tContentsTableView ] ) {
        [ self copy: nil ];
        [ self deleteSelectedLines: nil ];
        [[ self undoManager ] setActionName: NSLocalizedString( @"Cut", @"Cut" ) ];
    } else {
        if ( [[ tWindow firstResponder ] respondsToSelector: @selector( cut: ) ] ) {
            [[ tWindow firstResponder ] performSelector: @selector( cut: )
                                            withObject: sender
                                            afterDelay: 0.0 ];
        }
    }
    
    [ tContentsTableView reloadData ];
    [ tWindow setDocumentEdited: YES ];
}

- ( IBAction )paste: ( id )sender
{
    NSPasteboard            *pb = [ NSPasteboard generalPasteboard ];
    NSArray                 *types = [ pb types ];
    
    if ( [ types containsObject: RTETranscriptContentsPboardType ] ) {
	if ( [[ tWindow firstResponder ] isEqual: tContentsTableView ] ) {
	    NSArray         *pastedlines = nil;
            NSMutableArray  *lines = [[ self transcriptContents ] mutableCopy ];
	    
	    if ( mode == SEARCH ) {
		NSBeep();
		return;
	    }
	    
	    pastedlines = [ pb propertyListForType: RTETranscriptContentsPboardType ];
            
            if ( lines == nil ) {
                lines = [[ NSMutableArray alloc ] init ];
            }
            
            [ lines addObjectsFromArray: pastedlines ];
            [ lines sortUsingFunction: &pathcompare context: nil ];
            [ self setTranscriptContents: lines ];
            [ lines release ];
            [[ self undoManager ] setActionName: NSLocalizedString( @"Paste", @"Paste" ) ];
	} else {
	    NSBeep();
	    return;
	}
    } else if ( ! [[ tWindow firstResponder ] isEqual: tContentsTableView ] ) {
        if ( [[ tWindow firstResponder ] respondsToSelector: @selector( paste: ) ] ) {
            [[ tWindow firstResponder ] performSelector: @selector( paste: )
                                            withObject: sender
                                            afterDelay: 0.0 ];
        }
    } else {
	NSBeep();
	return;
    }
	
    [ tContentsTableView reloadData ];
    [ tWindow setDocumentEdited: YES ];
}

#ifdef HAS_COLUMN_VIEW
- ( void )clearTranscriptBrowser
{
    [ columnContents removeAllObjects ];
    [ transcriptBrowser loadColumnZero ];
}
#endif /* HAS_COLUMN_VIEW */

- ( IBAction )transSingleClick: ( id )browser
{
    int 		row = 0; //column = 0, next_off = 0; 
    //NSArray		*curColumnItems;
    
    if ( [ transcriptLineDrawer state ] == NSDrawerOpenState ) {
        row = [ tContentsTableView selectedRow ];
        if ( row < 0 ) row = 0;
        [ transcriptLineView setEditable: YES ];
        [ transcriptLineView setString:
            [ NSString transcriptLineFromDictionary:
            [ ( mode == SEARCH ? searchResults : parsedLines ) objectAtIndex: row ]]];
        [ transcriptLineView setEditable: NO ];
    }
    
    if ( viewtype == RXLISTVIEW ) return;
    
#ifdef HAS_COLUMN_VIEW
    column = [ transcriptBrowser selectedColumn ];
    
    curColumnItems = [ NSArray arrayWithArray: [ columnContents objectAtIndex:
                                                                column ]];
    
    while ( column < ( [ columnContents count ] - 1 )) {
#ifdef notdef
NSLog( @"\n\ncount: %d\ncount - 1:%d", [ columnContents count ], ( [ columnContents count ] - 1 ));
NSLog( @"last object: %@", [ columnContents lastObject ] );
#endif notdef
            [ columnContents removeLastObject ];
    }
    
    row = [ transcriptBrowser selectedRowInColumn: [ transcriptBrowser selectedColumn ]];
/*
    [ typeImage setImage: [ self imageForType: [[[ curColumnItems objectAtIndex: row ]
                                objectForKey: @"type" ] characterAtIndex: 0 ]]];
*/
    [ pathField setStringValue: [[ curColumnItems objectAtIndex: row ]
        objectForKey: @"path" ]];
    
    switch ( [[[ curColumnItems objectAtIndex: row ]
                objectForKey: @"type" ] characterAtIndex: 0 ] ) {
    case 'a':
    case 'f':
        [ transcriptBrowser setLastColumn: [ transcriptBrowser selectedColumn ]];
        break;
    case 'D':
    case 's':
    case 'p':
        break;
    case 'd':
        if ( row == ( [ curColumnItems count ] - 1 )) {
            next_off = 0;
        } else {
            next_off = [[[ curColumnItems objectAtIndex: ( row + 1 )] objectForKey: @"tindex" ] intValue ];
        }
                
        [ transcriptBrowser setLastColumn: [ transcriptBrowser selectedColumn ]];
        [ self arrangeColumn: ( [ transcriptBrowser selectedColumn ] + 1 )
                withOffset: [[[ curColumnItems objectAtIndex: row ] objectForKey: @"tindex" ] intValue ]
                toOffset: next_off ];
        [ transcriptBrowser addColumn ];
        [ transcriptBrowser reloadColumn: ( [ transcriptBrowser selectedColumn ] + 1 ) ];
        break;
    case 'l':
    case 'h':
        break;
    default:
        break;
    }
#endif /* HAS_COLUMN_VIEW */
}

- ( IBAction )transDoubleClick: ( id )browser
{
    [ self showInfo: nil ];
}

- ( NSString * )windowNibName
{
    // Override returning the nib file name of the document
    // If you need to use a subclass of NSWindowController or if your document supports multiple NSWindowControllers, you should remove this method and override -makeWindowControllers instead.
    return( @"RXTranscript" );
}

- ( void )windowControllerDidLoadNib: ( NSWindowController * )aController
{    
    [ super windowControllerDidLoadNib: aController ];
    
    /* set document delegate */
    [ self setDelegate: [ NSApp delegate ]];
    
    /* hide load progress bar initially */
    [ loadProgressIndicator setUsesThreadedAnimation: YES ];
    [ loadProgressIndicator retain ];
    [ loadProgressIndicator removeFromSuperview ];
    
    [ tbSearchField setEditable: YES ];
    [ tbSearchField setTextColor: [ NSColor disabledControlTextColor ]];
    [ tbSearchField setStringValue: @"Search" ];
    [ tbSearchField setTarget: self ];
    [ tbSearchField setDelegate: self ];
    [ tbSearchField retain ];
    [ tbSearchField removeFromSuperview ];
    
    /* set up toolbar */
    [ self toolbarSetup ];
    
    /* set browser behaviours */
#ifdef HAS_COLUMN_VIEW
    [ transcriptBrowser setTarget: self ];
    [ transcriptBrowser setAcceptsArrowKeys: YES ];
    [ transcriptBrowser setSendsActionOnArrowKeys: NO ];
    [ transcriptBrowser setMaxVisibleColumns: 4 ];
    [ transcriptBrowser setMinColumnWidth:
            NSWidth( [ transcriptBrowser bounds ] ) / 3.0 ];
    [ transcriptBrowser setAction: @selector( transSingleClick: ) ];
    [ transcriptBrowser setDoubleAction: @selector( transDoubleClick: ) ];
#endif /* HAS_COLUMN_VIEW */
    
    [ tContentsTableView setAction: @selector( transSingleClick: ) ];
    [ tContentsTableView setDataSource: self ];
    [ tContentsTableView setDelegate: self ];
    [ tContentsTableView setDoubleAction: @selector( showInfo: ) ];
    [ tContentsTableView registerForDraggedTypes:
            [ NSArray arrayWithObjects: NSFilenamesPboardType,
                                        NSStringPboardType, 
                                        RTETranscriptContentsPboardType, nil ]];
                                        
    if ( [ tContentsTableView respondsToSelector:
                @selector( setUsesAlternatingRowBackgroundColors: ) ] ) {
        [ tContentsTableView setUsesAlternatingRowBackgroundColors: YES ];
    }
    
    fontsize = [[ NSUserDefaults standardUserDefaults ] floatForKey: @"RTETextFontSize" ];
    if ( fontsize == 0.0 ) {
        fontsize = 12.0;
    }
    
    if ( viewtype == RXCOLUMNVIEW ) {
        viewtype = RXLISTVIEW;
#ifdef HAS_COLUMN_VIEW
        [ box setContentView: browserView ];
        [ browserView setNeedsDisplay: YES ];
        
        if ( [ parsedLines count ] ) {
            [ self arrangeColumn: 0 withOffset: 0 toOffset: 0 ];
            [ transcriptBrowser reloadColumn: 0 ];
        }
#endif /* HAS_COLUMN_VIEW */
    } else {
        [ box setContentView: tableView ];
    }

    [ tContentsTableView setAllowsColumnSelection: NO ];
    [[ tContentsTableView tableColumnWithIdentifier: @"Type" ] sizeToFit ];
    [ tContentsTableView reloadData ];
    [ tbSearchField setNextKeyView: tContentsTableView ];
    [ tContentsTableView setNextKeyView: tbSearchField ];
    [ tContentsTableView sizeLastColumnToFit ];
    
    [[ NSNotificationCenter defaultCenter ] addObserver: self
				    selector: @selector( textChanged: )
				    name: NSControlTextDidChangeNotification
				    object: addItemPathField ];
                                    
    [ transcriptSplitView setDelegate: self ];
    /* hide the info pane */
    [ transcriptSplitView collapseSubView: infoTabView animate: NO ];
}

- ( void )toolbarSetup
{
    NSToolbar *rxtbar = [[[ NSToolbar alloc ] initWithIdentifier:  @"RTETranscriptToolbar" ]
                                                autorelease ];
    
    [ rxtbar setAllowsUserCustomization: YES ];
    [ rxtbar setAutosavesConfiguration: YES ];
    [ rxtbar setDisplayMode: NSToolbarDisplayModeIconAndLabel ];
    [ rxtbar setDelegate: self ];
    
    [ tWindow setToolbar: rxtbar ];
}

/**/
/* required toolbar delegate methods */
/**/

- ( NSToolbarItem * )toolbar: ( NSToolbar * )toolbar itemForItemIdentifier: ( NSString * )itemIdent willBeInsertedIntoToolbar: ( BOOL )flag
{
    NSToolbarItem *rxtbarItem = [[[ NSToolbarItem alloc ]
                                    initWithItemIdentifier: itemIdent ] autorelease ];
                                      
    if ( [ itemIdent isEqualToString: RTEToolbarInfoIdentifier ] ) {
        [ rxtbarItem setLabel: @"Info" ];
        [ rxtbarItem setPaletteLabel: @"Info" ];
        [ rxtbarItem setToolTip: @"Show item information" ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"info.png" ]];
        [ rxtbarItem setAction: @selector( showInfo: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarConvertIdentifier ] ) {
        [ rxtbarItem setLabel: NSLocalizedString( @"Convert", @"Convert" ) ];
        [ rxtbarItem setPaletteLabel: NSLocalizedString( @"Convert", @"Convert" ) ];
        [ rxtbarItem setToolTip: NSLocalizedString( @"Convert base path of transcript",
                                        @"Convert base path of transcript" ) ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"convert.png" ]];
        [ rxtbarItem setAction: @selector( convertTranscriptBasePath: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarSaveIdentifier ] ) {
        [ rxtbarItem setLabel: @"Save" ];
        [ rxtbarItem setPaletteLabel: @"Save" ];
        [ rxtbarItem setToolTip: @"Save Transcript" ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"save.png" ]];
        [ rxtbarItem setAction: @selector( saveDocument: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarDeleteIdentifier ] ) {
        [ rxtbarItem setLabel: @"Delete" ];
        [ rxtbarItem setPaletteLabel: @"Delete" ];
        [ rxtbarItem setToolTip: @"Delete from Transcript" ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"trash.png" ]];
        [ rxtbarItem setAction: @selector( deleteSelectedLines: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarSearchIdentifier ] ) {
        [ rxtbarItem setLabel: @"Search" ];
        [ rxtbarItem setPaletteLabel: @"Search" ];
        [ rxtbarItem setToolTip: @"Search for Items in Transcript" ];
        [ rxtbarItem setView: tbSearchField ];
        [ rxtbarItem setAction: @selector( search: ) ];
        [ rxtbarItem setTarget: self ];
        [ rxtbarItem setMinSize: NSMakeSize( 30, NSHeight( [ tbSearchField frame ] )) ];
	[ rxtbarItem setMaxSize: NSMakeSize( 160, NSHeight( [ tbSearchField frame ] )) ];
        [ tbSearchField setEnabled: YES ];
        if ( viewtype == RXCOLUMNVIEW ) [ tbSearchField setEnabled: NO ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarToggleCommentIdentifier ] ) {
        [ rxtbarItem setLabel: NSLocalizedString( @"Toggle Comment", @"Toggle Comment" ) ];
        [ rxtbarItem setPaletteLabel: NSLocalizedString( @"Toggle Comment", @"Toggle Comment" ) ];
        [ rxtbarItem setToolTip: NSLocalizedString( @"Toggle comment on selected line",
                                                    @"Toggle comment on selected line" ) ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"comment.png" ]];
        [ rxtbarItem setAction: @selector( toggleCommentOnSelectedLine: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarAuditIdentifier ] ) {
        [ rxtbarItem setLabel: NSLocalizedString( @"Audit", @"Audit" ) ];
        [ rxtbarItem setPaletteLabel: NSLocalizedString( @"Audit", @"Audit" ) ];
        [ rxtbarItem setToolTip: NSLocalizedString(
		@"Audit transcript for potential security problems",
		@"Audit transcript for potential security problems" ) ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"comment.png" ]];
        [ rxtbarItem setAction: @selector( auditTranscript: ) ];
        [ rxtbarItem setTarget: self ];
    } else if ( [ itemIdent isEqualToString: RTEToolbarTwhichIdentifier ] ) {
	[ rxtbarItem setLabel: NSLocalizedString( @"Source Transcript", @"Source Transcript" ) ];
        [ rxtbarItem setPaletteLabel: NSLocalizedString( @"Source Transcript", @"Source Transcript" ) ];
        [ rxtbarItem setToolTip: NSLocalizedString(
		@"Show transcripts containing the selected path",
		@"Show transcripts containing the selected path" ) ];
        [ rxtbarItem setImage: [ NSImage imageNamed: @"comment.png" ]];
        [ rxtbarItem setAction: @selector( showSourceTranscripts: ) ];
        [ rxtbarItem setTarget: self ];
    }
            
    return( rxtbarItem );
}

- ( NSArray * )toolbarDefaultItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RTEToolbarSaveIdentifier,
                            RTEToolbarConvertIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RTEToolbarInfoIdentifier,
			    RTEToolbarAuditIdentifier,
                            RTEToolbarToggleCommentIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RTEToolbarDeleteIdentifier,
                            NSToolbarSeparatorItemIdentifier,
                            RTEToolbarSearchIdentifier, nil ];
                            
    return( tmp );
}

- ( NSArray * )toolbarAllowedItemIdentifiers: ( NSToolbar * )toolbar
{
    NSArray	*tmp = [ NSArray arrayWithObjects:
                            RTEToolbarSaveIdentifier,
                            RTEToolbarInfoIdentifier,
                            RTEToolbarToggleCommentIdentifier,
                            NSToolbarFlexibleSpaceItemIdentifier,
                            RTEToolbarDeleteIdentifier, 
                            NSToolbarSeparatorItemIdentifier,
                            RTEToolbarSearchIdentifier,
                            RTEToolbarConvertIdentifier, 
			    RTEToolbarAuditIdentifier, nil ];
                            
    return( tmp );
}

- ( BOOL )validateToolbarItem: ( NSToolbarItem * )item
{
    BOOL        validate = YES;
    
    if ( [[ item itemIdentifier ] isEqualToString: RTEToolbarSaveIdentifier ] ) {
        validate = [ self isDocumentEdited ];
    } else if ( [[ item itemIdentifier ] isEqualToString: RTEToolbarDeleteIdentifier ] ) {
        if ( [[ self transcriptContents ] count ] == 0 ) {
            validate = NO;
        }
    } else if ( [[ item itemIdentifier ] isEqualToString: RTEToolbarInfoIdentifier ] ) {
        if ( [[ self transcriptContents ] count ] == 0 ) {
            validate = NO;
        } else if ( [ transcriptSplitView isSubviewCollapsed: infoTabView ] ) {
            [ item setAction: @selector( showInfo: ) ];
            [ item setLabel: NSLocalizedString( @"Show Info", @"Show Info" ) ];
        } else {
            [ item setAction: @selector( hideInfo: ) ];
            [ item setLabel: NSLocalizedString( @"Hide Info", @"Hide Info" ) ];
        }
    } else if ( [[ item itemIdentifier ] isEqualToString: RTEToolbarConvertIdentifier ] ) {
        if ( [[ self transcriptContents ] count ] == 0 ) {
            validate = NO;
        }
    }
    
    return( validate );
}
/* end required toolbar delegate methods */

- ( BOOL )validateMenuItem: ( NSMenuItem * )anItem
{
    if ( [[ anItem title ] isEqualToString: NSLocalizedString( @"Toggle Comment On Selected Line",
                                                    @"Toggle Comment On Selected Line" ) ] ) {
        if ( [ tContentsTableView numberOfSelectedRows ] != 1 ) {
            return( NO );
        }
    } else if ( [[ anItem title ] isEqualToString:
		NSLocalizedString( @"Show Text Transcript Line",
		@"Show Text Transcript Line" ) ] ) {
	if ( [ ( mode == SEARCH ? searchResults : parsedLines ) count ] <= 0 ) {
	    return( NO );
	}
    } else if ( [[ anItem title ] isEqualToString:
		NSLocalizedString( @"Show Info", @"Show Info" ) ] ) {
	if ( [ ( mode == SEARCH ? searchResults : parsedLines ) count ] <= 0 ) {
	    return( NO );
	}
    }
    
    return( [ super validateMenuItem: anItem ] );
}

- ( IBAction )toggleDrawer: ( id )sender
{
    if ( [ transcriptLineDrawer state ] == NSDrawerOpenState ) {
        [ transcriptLineDrawer close ];
    } else if ( [ transcriptLineDrawer state ] == NSDrawerClosedState ) {
        [ transcriptLineDrawer setContentSize: NSMakeSize( 0, 80.0 ) ];
        [ transcriptLineView setEditable: YES ];
        [ transcriptLineView setString:
            [ NSString transcriptLineFromDictionary:
            [ ( mode == SEARCH ? searchResults : parsedLines ) objectAtIndex:
                                        [ tContentsTableView selectedRow ]]]];
        [ transcriptLineView setEditable: NO ];
        [ transcriptLineDrawer open ];
    }
}

- ( IBAction )convertTranscriptBasePath: ( id )sender
{
    NSEvent                     *e = [ NSApp currentEvent ], *modEvent;
    NSPoint                     pt = [ e locationInWindow ];
                                                        
    pt.y = NSHeight( [[ tWindow contentView ] frame ] );
    
    modEvent = [ NSEvent mouseEventWithType: [ e type ]
                            location: pt
                            modifierFlags: [ e modifierFlags ]
                            timestamp: [ e timestamp ]
                            windowNumber: [ e windowNumber ]
                            context: [ e context ]
                            eventNumber: [ e eventNumber ]
                            clickCount: [ e clickCount ]
                            pressure: [ e pressure ]];
    [ NSMenu popUpContextMenu: conversionMenu
                withEvent: modEvent
                forView: [ tWindow contentView ]];
}

- ( IBAction )convertBasePath: ( id )sender
{
    NSString                    *path;
    int                         rc;
    
    if ( ! [ sender isKindOfClass: [ NSMenuItem class ]] ) {
        NSBeep();
        return;
    }
    
    path = [ sender title ];
    
    rc = [ self convertTranscriptBaseToPath: path ];
    
    if ( rc < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"Couldn't convert transcript base path to %@",
                            @"Couldn't convert transcript base path to %@" ),
                NSLocalizedString( @"OK", @"OK" ), @"", @"", path );
    }
}

- ( void )auditTranscript: ( id )sender
{
    RTEAuditor			*auditor = [[ RTEAuditor alloc ] init ];
    NSMutableArray		*auditResults = nil;
    
    auditResults = [[ auditor auditTranscriptContents:
				[ self transcriptContents ]] retain ];
				
    [ auditor release ];
    
    if ( auditorDisplay == nil ) {
	auditorDisplay = [[ RTEAuditorDisplay alloc ] init ];
	[ NSBundle loadNibNamed: @"AuditResults" owner: auditorDisplay ];
    }
    
    [ auditorDisplay displayAuditResults: auditResults
	    parentWindow: [ tContentsTableView window ]];
    
    [ auditResults release ];
}

- ( void )showSourceTranscripts: ( id )sender
{
    NSArray			*args, *lines;
    NSString			*path;
    int				row;

    if (( row = [ tContentsTableView selectedRow ] ) < 0 ) {
	return;
    }
    
    lines = ( mode == SEARCH ? searchResults : parsedLines );
    
    if (( path = [[ lines objectAtIndex: row ] objectForKey: @"path" ] ) == nil ) {
	return;
    }
    
    args = [ NSArray arrayWithObjects: @"-A", @"ExecuteCommand",
					[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]],
					@"--", @"/usr/local/bin/twhich",
                                        @"-a", path, nil ];

    [ self executeAuthorizedAction: TWHICH arguments: args ];
}

- ( oneway void )displayTwhichOutput: ( NSString * )output fromThreadWithID: ( int )ID
{
    NSString			*string;
    
    if ( ID != [ self helperThreadID ] ) {
	return;
    }
    
    string = [ output retain ];

    if ( twhichDisplay == nil ) {
	twhichDisplay = [[ RTETwhichDisplay alloc ] init ];
	[ NSBundle loadNibNamed: @"TwhichResults" owner: twhichDisplay ];
    }
    
    [ twhichDisplay parseAndDisplayTwhichString: string ];
    [ string release ];
}

- ( IBAction )toggleCommentOnSelectedLine: ( id )sender
{
    char			c, **tav, ln[ LINE_MAX ];
    int                         row, tac, index;
    NSMutableDictionary		*dict, *newdict;
    NSString			*line;
    NSMutableArray              *lines = nil;
    NSEnumerator                *en = nil;
    NSNumber                    *n = nil;
    CFRange                     range = { 0, 0 };
    
    if ( [ tContentsTableView numberOfSelectedRows ] == 0 ) {
        return;
    }
    
    if ( mode == SEARCH ) {
	lines = [ searchResults mutableCopy ];
    } else {
	lines = [[ self transcriptContents ] mutableCopy ];
    }
    en = [ tContentsTableView selectedRowEnumerator ];
    
    while (( n = [ en nextObject ] ) != nil ) {
        row = [ n intValue ];
        dict = [ lines objectAtIndex: row ];
        c = [[ dict objectForKey: @"path" ] characterAtIndex: 0 ];
                    
        if ( c == '#' ) {   /* uncomment line */
            line = [ dict objectForKey: @"path" ];
            
            if ( [ line length ] < 2 ) {
                continue;
            }
            
            line = [ line substringFromIndex: 1 ];
            range.length = ( CFIndex )[ line length ];
            
            if ( CFStringGetBytes(( CFStringRef )line, range,
                    CFStringGetSystemEncoding(), 0,
                    FALSE, ( UInt8 * )ln, LINE_MAX, NULL ) == 0 ) {
                NSLog( @"CFStringGetBytes for %@ failed", line );
                continue;
            }
            tac = argcargv( ln, &tav );
            
            if ( tac < 2 ) {
                continue;
            }
            
            newdict = [ NSMutableDictionary dictionaryForLine: line ];
        } else {            /* comment line */
            line = [ NSString transcriptLineFromDictionary: dict ];
            line = [ NSString stringWithFormat: @"# %@", line ];
            newdict = [ NSMutableDictionary dictionaryForLine: line ];
        }
        
	index = [[[ lines objectAtIndex: row ] objectForKey: @"tindex" ] intValue ];
	
        [ lines replaceObjectAtIndex: row withObject: newdict ];
	if ( mode == SEARCH ) {
	    [[ self transcriptContents ] replaceObjectAtIndex: index withObject: newdict ];
	}
    }
    
    if ( mode == SEARCH ) {
	[ searchResults release ];
	searchResults = [ lines retain ];
    } else {
	[ self setTranscriptContents: lines ];
    }
    [ lines release ];
    
    [ self updateIndices ];
    [ tContentsTableView reloadData ];
}

#ifdef notdef
- ( IBAction )newComment: ( id )sender
{
    int                 row = [ tContentsTableView selectedRow ];
    NSMutableArray      *lines = [[ self transcriptContents ] mutableCopy ];
    
    if ( row < 0 ) {
        row = 0;
    }
    
    [ lines insertObject: [ NSMutableDictionary dictionaryWithObjectsAndKeys:
                                @"# New Comment", @"host", @"", @"kfile", nil ]
                            atIndex: row ];
    [ tContentsTableView reloadData ];
    [ tContentsTableView selectRow: row byExtendingSelection: NO ];
    [ tContentsTableView editColumn: 0 row: row withEvent: nil select: YES ];
}
#endif notdef

- ( IBAction )lineNumberToggle: ( id )sender
{
    if ( [[ NSUserDefaults standardUserDefaults ] boolForKey: @"ShowLineNumbers" ] ) {
        if ( [ tContentsTableView tableColumnWithIdentifier: @"linenumbers" ] != nil ) {
            [ lineNumberColumn retain ];
            [ tContentsTableView removeTableColumn:
                        [ tContentsTableView tableColumnWithIdentifier: @"linenumbers" ]];
        }
        [[ NSUserDefaults standardUserDefaults ] setBool: NO
                                                    forKey: @"ShowLineNumbers" ];
#ifdef notdef
        if ( [ tContentsTableView tableColumnWithIdentifier: @"linenumbers" ] == nil ) {
            column = [[ NSTableColumn alloc ] initWithIdentifier: @"linenumbers" ];
            [[ column headerCell ] setStringValue: @"Line #" ];
            [ column sizeToFit ];
        }
#endif notdef
    } else {
        if ( [ tContentsTableView tableColumnWithIdentifier: @"linenumbers" ] == nil ) {
            [ tContentsTableView addTableColumn: lineNumberColumn ];
            [ tContentsTableView moveColumn: 3 toColumn: 0 ];
        }
        [[ NSUserDefaults standardUserDefaults ] setBool: YES
                                                    forKey: @"ShowLineNumbers" ];
    }
}

- ( IBAction )showGotoLinePanel: ( id )sender
{
    [ gotoLinePanel makeKeyAndOrderFront: nil ];
}

- ( IBAction )gotoLine: ( id )sender
{
    NSMutableArray	*lines = parsedLines;
    int			ln;
    
    if ( mode == SEARCH ) {
	lines = searchResults;
    }
    
    /* subtract one: rows are counted from 0, not 1 ) */
    ln = ( [ gotoLineField intValue ] - 1 );
    if (( ln < 0 ) || ( ln >= [ lines count ] )) {
	NSBeep();
	return;
    }
    
    [ tContentsTableView selectRow: ln byExtendingSelection: NO ];
    [ tContentsTableView scrollRowToVisible: ln ];
    [ tWindow makeFirstResponder: tContentsTableView ];
}

- ( void )controlTextDidChange: ( NSNotification * )aNotification
{
    if ( [ aNotification object ] != tbSearchField ) {
        return;
    }
    
    [ NSObject cancelPreviousPerformRequestsWithTarget: self
                selector: @selector( search: )
                object: nil ];
                
    if ( [[ tbSearchField stringValue ] length ] == 0 ) {
        [ self cancelSearchMode ];
        return;
    }
                
    [ self performSelector: @selector( search: )
            withObject: nil
            afterDelay: 0.5 ];
}

- ( void )searchFor: ( NSString * )searchTerm searchType: ( int )type
        caseInsensitive: ( BOOL )ci
{
    NSString		*searchkey = nil, *searchString = searchTerm;
    BOOL		wildcard;
    int			i;
    
    [ searchResults removeAllObjects ];
    
    wildcard = [[ NSUserDefaults standardUserDefaults ]
		    boolForKey: @"RTEWildcardSearches" ];
    
    mode = SEARCH;
    
    switch ( type ) {
    case PATHSEARCH:
    default:
        searchkey = @"path";
        break;
    case TYPESEARCH:
        searchkey = @"type";
        break;
    case OWNERSEARCH:
        searchkey = @"owner";
        break;
    case GROUPSEARCH:
        searchkey = @"group";
        break;
    case PERMSEARCH:
        searchkey = @"perm";
        break;
    case MINUSSEARCH:
        searchkey = @"pm";
        searchString = @"-";
        break;
    case PLUSSEARCH:
        searchkey = @"pm";
        searchString = @"+";
        break;
    }
    
    for ( i = 0; i < [ parsedLines count ]; i++ ) {
        NSDictionary	*dict = [ parsedLines objectAtIndex: i ];
        NSString	*tl = [ dict objectForKey: searchkey ];
	BOOL		match = NO;

        if ( tl == nil ) {
	    continue;
	}
        
	if ( wildcard ) {
	    match = [ tl matchesWildcard: searchString caseSensitive: !ci ];
	} else {
	    match = [ tl containsString: searchString
			    useCaseInsensitiveComparison: ci ];
	}
	
	if ( match ) {
	    [ searchResults addObject: dict ];
	}
    }
    
    [ tContentsTableView reloadData ];
    [ lineCountField setStringValue: [ NSString stringWithFormat: @"%d match%@",
                                [ searchResults count ],
                                ( [ searchResults count ] == 1 ? @"" : @"es" ) ]];
}

- ( void )search: ( id )sender
{
    [ searchResults removeAllObjects ];
    [ self setSearching: YES ];
    
    if ( [[ tbSearchField stringValue ] length ] == 0 ) {
        mode = NORMAL;
        [[ tbSearchField cell ] setImage: nil ];
        [ tbSearchField setNeedsDisplay: YES ];
        [ tContentsTableView reloadData ];
        [ lineCountField setStringValue: [ NSString stringWithFormat: @"%d line%@",
                                        [ parsedLines count ],
                                        ( [ parsedLines count ] == 1 ? @"" : @"s" ) ]];
        
        return;
    }
    
    [ self searchFor: [ tbSearchField stringValue ]
            searchType: [ self advancedSearchType ]
            caseInsensitive: YES ];
}

- ( IBAction )advancedSearch: ( id )sender
{
    BOOL		ci;
    int			st;
    
    st = [ advancedSearchTypeButton indexOfSelectedItem ];
    [ self setAdvancedSearchType: st ];
    
    if ( [[ advancedSearchField stringValue ] length ] == 0 &&
                [ self advancedSearchType ] != PLUSSEARCH &&
                [ self advancedSearchType ] != MINUSSEARCH ) {
        mode = NORMAL;
        [ tContentsTableView reloadData ];
        [ lineCountField setStringValue: [ NSString stringWithFormat: @"%d line%@",
                                        [ parsedLines count ],
                                        ( [ parsedLines count ] == 1 ? @"" : @"s" ) ]];
        
        return;
    }
    
    ci = ( BOOL )[ advancedSearchCaseSensitivity state ];
    [ self searchFor: [ advancedSearchField stringValue ]
        searchType: st caseInsensitive: ci ];
}

- ( void )setAdvancedSearchType: ( int )type
{
    searchtype = type;
}

- ( int )advancedSearchType
{
    return( searchtype );
}

- ( IBAction )setSearchType: ( id )sender
{
    NSMenu              *menu;
    NSArray             *itemArray;
    
    if ( ! [ sender isKindOfClass: [ NSMenuItem class ]] ) {
        return;
    }
    
    if (( menu = [ sender menu ] ) != nil ) {
        int             i;
        
        itemArray = [ menu itemArray ];
        
        for ( i = 0; i < [ itemArray count ]; i++ ) {
            [[ itemArray objectAtIndex: i ] setState: NSOffState ];
        }
    }
    
    [ sender setState: NSOnState ];
    [ self setAdvancedSearchType: [ sender tag ]];
}

- ( IBAction )displayAdvancedSearchPanel: ( id )sender
{
    [ advancedSearchTypeButton selectItemAtIndex: [ self advancedSearchType ]];
    [ advancedSearchPanel makeKeyAndOrderFront: nil ];
}

- ( IBAction )dismissAdvancedSearchPanel: ( id )sender
{
    [ advancedSearchPanel close ];
}

- ( void )cancelSearchMode
{
    int                 count = [[ self transcriptContents ] count ];
    
    [ searchResults removeAllObjects ];
    mode = NORMAL;
    [ tContentsTableView reloadData ];
    [ lineCountField setStringValue:
                [ NSString stringWithFormat: @"%d line%@", count,
                                        ( count == 1 ? @"" : @"s" ) ]];
    [ tbSearchField setStringValue: @"" ];
    [ tbSearchField selectText: nil ];
    [ self setSearching: NO ];
}

#ifdef HAS_COLUMN_VIEW
- ( void )switchViews
{
    if ( viewtype == RXLISTVIEW ) {
        viewtype = RXCOLUMNVIEW;
        [ box setContentView: browserView ];
        
        if ( [ tContentsTableView selectedRow ] >= 0 ) {
            [ self arrangeColumn: 0 withOffset: 0 toOffset: 0 ];
            [ transcriptBrowser reloadColumn: 0 ];
            [ transcriptBrowser selectRow: 0 inColumn: 0 ];
        }
    } else {
//#ifdef notdef
        int		index = atoi( [[[[ columnContents objectAtIndex:
                                    [ transcriptBrowser selectedColumn ]]
                                    objectAtIndex: [ transcriptBrowser selectedRowInColumn:
                                                    [ transcriptBrowser selectedColumn ]]]
                                    objectForKey: @"tindex" ] UTF8String ] );
//#endif notdef
                                    
        viewtype = RXLISTVIEW;
        if ( [ columnContents count ] > 0 ) {
            [ columnContents removeAllObjects ];
            [ transcriptBrowser reloadColumn: 0 ];
        }
        [ pathField setStringValue: @"" ];
        [ box setContentView: tableView ];
        
        if ( [ transcriptBrowser selectedRowInColumn: [ transcriptBrowser selectedColumn ]] >= 0 )
            [ tContentsTableView reloadData ];
        //[ tContentsTableView selectRow: index byExtendingSelection: NO ];
        //[ tContentsTableView scrollRowToVisible: index ];
    }
    
    [ self toolbarSetup ];
}
#endif /* HAS_COLUMN_VIEW */

- ( IBAction )showInfo: ( id )sender
{
    //NSArray		*curColumnItems;
    NSDictionary		*dict;
    NSMenu			*menu;
    id				menuItem;
    int                         row; //tcolumn;
    char                        type;
                        
#ifdef HAS_COLUMN_VIEW
    if ( viewtype == RXCOLUMNVIEW ) {
        if ( [ transcriptBrowser selectedColumn ] < 0 ) {
            NSRunAlertPanel( @"No item selected.", @"Please select an item and try again",
                                @"OK", @"", @"" );
            return;		/* if there's nothing selected, there's nothing to display */
        }

        tcolumn = [ transcriptBrowser selectedColumn ];
        row = [ transcriptBrowser selectedRowInColumn: [ transcriptBrowser selectedColumn ]];
        curColumnItems = [ NSArray arrayWithArray:
                                [ columnContents objectAtIndex: tcolumn ]];
        dict = [ curColumnItems objectAtIndex: row ];
    }
#endif /* HAS_COLUMN_VIEW */

    menu = [ NSApp mainMenu ];
    menu = [[ menu itemWithTitle: NSLocalizedString( @"File", @"File" ) ] submenu ];
    if (( menuItem = [ menu itemWithTitle:
	    NSLocalizedString( @"Show Info", @"Show Info" ) ] ) != nil ) {
	[ menuItem setTitle: NSLocalizedString( @"Hide Info", @"Hide Info" ) ];
	[ menuItem setAction: @selector( hideInfo: ) ];
    }
    
    if ( [ tContentsTableView numberOfSelectedRows ] > 1 ) {
        [ typeImage setImage: nil ];
        [ typeImage setImage: [ NSImage imageNamed: @"multiple.png" ]];
        [ kind setStringValue: @"multiple items" ];
        [ permissionsField setStringValue: @"-" ];
        [ size setStringValue: @"-" ];
        [ modified setStringValue: @"-" ];
        [ chksum setStringValue: @"-" ];
        [ owner setStringValue: @"-" ];
        [ group setStringValue: @"-" ];
        [ nameField setStringValue: @"Multiple items selected" ];
        [ where setStringValue:
            @"Multiple lines selected. Changes made will affect all selected items." ];
        [ self setupPermissionSwitchesForMode: nil ];
        [ editOwnerUIDField setStringValue: @"" ];
        [ editGroupGIDField setStringValue: @"" ];
	[ editedTranscriptLineField setStringValue: @"" ];

	[ transcriptSplitView expandSubView: infoTabView animate: NO ];
        return;
    }

    row = [ tContentsTableView selectedRow ];
    if ( row < 0 ) {
        NSBeep();
        return;		/* if there's nothing selected, there's nothing to display */
    }
    
    dict = [ ( mode == NORMAL ? parsedLines : searchResults ) objectAtIndex: row ];
    type = [[ dict objectForKey: @"type" ] characterAtIndex: 0 ];
    
    if ( type == '0' ) { NSBeep(); return; }
    [ dict retain ];

#ifdef rte_next_version
    infoPanelController = [[ RTEInfoPanelController alloc ] init ];
    [ infoPanelController setInfoDictionary: dict ];
    [ NSBundle loadNibNamed: @"InfoPanel" owner: infoPanelController ];
    [ infoPanelController showWindow: nil ];
    [ self addWindowController: infoPanelController ];
#endif /* rte_next_version */
    
    /* set its fields with the right information */
    [ typeImage setImage: [ NSImage imageForType: type ]];    
    [ nameField setStringValue: [[ dict objectForKey: @"path" ]
                    lastPathComponent ]];
    [ where setStringValue: [ dict objectForKey: @"path" ]];
    [ editedTranscriptLineField setStringValue:
            [ NSString transcriptLineFromDictionary: dict ]];
    
    if ( type != 'h' && type != 'l' && type != '0' ) {
        NSString	*o = [ dict objectForKey: @"owner" ],
                        *g = [ dict objectForKey: @"group" ],
                        *p = [ dict objectForKey: @"perm" ];
        [ self setupPermissionSwitchesForMode: p ];
        if ( o != nil )
            [ editOwnerUIDField setStringValue: o ];
            
        if ( g != nil )
            [ editGroupGIDField setStringValue: g ];
    }
    
    switch ( type ) {
    case 'a':
    case 'f':
        [ kind setStringValue: (( type == 'a' ) ? @"applefile" : @"file" ) ];
        [ permissionsField setStringValue: 
            [ NSString stringWithFormat: @"%@ (%@)",
                    [[ dict objectForKey: @"perm" ] stringFromOctal ],
                    [ dict objectForKey: @"perm" ]]];
        [ size setStringValue: [[ dict objectForKey: @"size" ] descriptiveSizeString ]];
        [ modified setStringValue:
                [ NSString formattedDateFromTimeInterval: [ dict objectForKey: @"mtime" ]]];
        [ chksum setStringValue: [ dict objectForKey: @"cksum" ]];
        [ owner setStringValue: [ dict objectForKey: @"owner" ]];
        [ group setStringValue: [ dict objectForKey: @"group" ]];
        break;
    case 'D':
    case 's':
    case 'p':
    case 'd':
        switch( type ) {
        case 'D':
            [ kind setStringValue: @"door (System V UNIX only)" ];
            break;
        case 's':
            [ kind setStringValue: @"socket" ];
            break;
        case 'p':
            [ kind setStringValue: @"named pipe" ];
            break;
        case 'd':
            [ kind setStringValue: @"directory" ];
        default:
            break;
        }
    
        [ owner setStringValue: [ dict objectForKey: @"owner" ]];
        [ group setStringValue: [ dict objectForKey: @"group" ]];
        [ permissionsField setStringValue: [ NSString stringWithFormat: @"%@ (%@)",
                    [[ dict objectForKey: @"perm" ] stringFromOctal ],
                    [ dict objectForKey: @"perm" ]]];
        [ size setStringValue: @"-" ];
        [ modified setStringValue: @"-" ];
        [ chksum setStringValue: @"-" ]; 
        break;
    case 'l':
    case 'h':
        [ where setStringValue: 
            [ NSString stringWithFormat: @"%@ -> %@",
                [ dict objectForKey: @"path" ], [ dict objectForKey: @"dest" ]]];
        [ kind setStringValue: [ NSString stringWithFormat: @"%s",
                                (( type == 'l' ) ? "symbolic link" : "hard link" ) ]];
        [ permissionsField setStringValue: @"-" ];
        [ size setStringValue: @"-" ];
        [ modified setStringValue: @"-" ];
        [ chksum setStringValue: @"-" ];
        [ owner setStringValue: @"-" ];
        [ group setStringValue: @"-" ];
        break;
    case 'b':
    case 'c':
    /* etc */
        [ owner setStringValue: [ dict objectForKey: @"owner" ]];
        [ group setStringValue: [ dict objectForKey: @"group" ]];
        [ permissionsField setStringValue: 
                [ NSString stringWithFormat: @"%@ (%@)",
                    [[ dict objectForKey: @"perm" ] stringFromOctal ],
                    [ dict objectForKey: @"perm" ]]];
    default:
        break;
    }
    [ dict release ];
    
    [ transcriptSplitView expandSubView: infoTabView animate: NO ];
}

- ( IBAction )hideInfo: ( id )sender
{
    NSMenu		*menu;
    id			menuItem;
    
    [ transcriptSplitView collapseSubView: infoTabView animate: NO ];
    
    menu = [ NSApp mainMenu ];
    menu = [[ menu itemWithTitle: NSLocalizedString( @"File", @"File" ) ] submenu ];
    if (( menuItem = [ menu itemWithTitle:
	    NSLocalizedString( @"Hide Info", @"Hide Info" ) ] ) != nil ) {
	[ menuItem setTitle: NSLocalizedString( @"Show Info", @"Show Info" ) ];
	[ menuItem setAction: @selector( showInfo: ) ];
    }
}

- ( IBAction )changeOwner: ( id )sender
{
    NSString			*newowner;
    NSMutableArray              *lines;
    NSEnumerator		*en;
    id				dobj;
    int				i, err = 0;
    
    newowner = [ editOwnerUIDField stringValue ];
    
    for ( i = 0; i < [ newowner length ]; i++ ) {
        if ( ! isdigit( [ newowner characterAtIndex: i ] )) {
            /* assume the entry in the field is a username */
	    struct passwd	*pw;
	    const char		*username = [ newowner UTF8String ];
	    
	    if (( pw = getpwnam( username )) == NULL ) {
		err++;
		break;
	    }
	    
	    newowner = nil;
	    newowner = [ NSString stringWithFormat: @"%ld", pw->pw_uid ];
            break;
        }
    }
    
    if ( err || ! [ newowner length ] || [ newowner isEqualToString: @"" ] ||
            newowner == nil ) {
        [ editOwnerUIDField setStringValue: [ owner stringValue ]];
        NSBeginAlertSheet( @"Error",
                          @"OK", @"", @"", tWindow,
                          self, NULL, nil, NULL,
                          @"Invalid owner." );
        return;
    }
    
    lines = [ self transcriptContents ];
    
    [[ self undoManager ] setActionName: NSLocalizedString( @"Owner Change",
                                                            @"Owner Change" ) ];
    
    en = [ tContentsTableView selectedRowEnumerator ];
    while (( dobj = [ en nextObject ] ) != nil ) {
        int                 index = [ dobj intValue ];
        
        if ( mode == SEARCH ) {
            [[ searchResults objectAtIndex: index ] setObject: newowner
                                                        forKey: @"owner" ];
            index = strtol(( char * )[[[ searchResults objectAtIndex: [ dobj intValue ]]
                                                objectForKey: @"tindex" ] UTF8String ],
                                                NULL, 10 );
        }
        
        [[ lines objectAtIndex: index ] setObject: newowner forKey: @"owner" ];
    }
    
    [ editOwnerUIDField setStringValue: newowner ];
    
    [ tWindow setDocumentEdited: YES ];
    if ( [ tContentsTableView numberOfSelectedRows ] == 1 ) {
	[ self showInfo: nil ];
    }
}

- ( IBAction )changeGroup: ( id )sender
{
    NSString			*newgroup = nil;
    NSMutableArray              *lines;
    NSEnumerator		*en;
    id				dobj;
    int				i, err = 0;
    
    newgroup = [ editGroupGIDField stringValue ];
    
    for ( i = 0; i < [ newgroup length ]; i++ ) {
        if ( ! isdigit( [ newgroup characterAtIndex: i ] )) {
	    /* assume it's a group name */
	    struct group	*gr;
	    const char		*groupname = [ newgroup UTF8String ];
	    
	    if (( gr = getgrnam( groupname )) == NULL ) {
		err++;
		break;
	    }
	    
            newgroup = nil;
	    newgroup = [ NSString stringWithFormat: @"%ld", gr->gr_gid ];
            break;
        }
    }
    
    if ( err || ! [ newgroup length ] || [ newgroup isEqualToString: @"" ] 
            || newgroup == nil ) {
        [ editGroupGIDField setStringValue: [ group stringValue ]];
        NSBeginAlertSheet( @"Error",
                          @"OK", @"", @"", tWindow,
                          self, NULL, nil, NULL,
                          @"Invalid group." );
        return;
    }
    
    lines = [[ self transcriptContents ] mutableCopy ];
    
    en = [ tContentsTableView selectedRowEnumerator ];
    while (( dobj = [ en nextObject ] ) != nil ) {
        int                 index = [ dobj intValue ];
        
        if ( mode == SEARCH ) {
            [[ searchResults objectAtIndex: index ] setObject: newgroup
                                                        forKey: @"group" ];
            index = strtol(( char * )[[[ searchResults objectAtIndex: [ dobj intValue ]]
                                                objectForKey: @"tindex" ] UTF8String ],
                                                NULL, 10 );
        }
        
        [[ lines objectAtIndex: index ] setObject: newgroup forKey: @"group" ];
    }
    
    [[ self undoManager ] setActionName: NSLocalizedString( @"Group Change",
                                                            @"Group Change" ) ];
    [ self setTranscriptContents: lines ];
    [ lines release ];
    
    [ editGroupGIDField setStringValue: newgroup ];
    [ tWindow setDocumentEdited: YES ];
    [ self updateChangeCount: NSChangeDone ];
    if ( [ tContentsTableView numberOfSelectedRows ] == 1 ) {
	[ self showInfo: nil ];
    }
}

- ( void )setupPermissionSwitchesForMode: ( NSString * )octalmode
{
    NSString		*modeString = octalmode;
    int			own, ow, grp, g, oth, ot, adv, a;
    
    /* switch all off first */
    [ owReadSwitch setState: NSOffState ];
    [ owWriteSwitch setState: NSOffState ];
    [ owExecSwitch setState: NSOffState ];
    [ grReadSwitch setState: NSOffState ];
    [ grWriteSwitch setState: NSOffState ];
    [ grExecSwitch setState: NSOffState ];
    [ otReadSwitch setState: NSOffState ];
    [ otWriteSwitch setState: NSOffState ];
    [ otExecSwitch setState: NSOffState ];
    [ setuidSwitch setState: NSOffState ];
    [ setgidSwitch setState: NSOffState ];
    [ stickySwitch setState: NSOffState ];
    
    [ octalModeField setStringValue: @"" ];
    if ( modeString == nil || [ modeString isEqualToString: @"" ] ) {
	modeString = @"0000";
    }
    
    a = adv = ( [ octalmode characterAtIndex: 0 ] - '0' );
    ow = own = ( [ octalmode characterAtIndex: 1 ] - '0' );
    g = grp = ( [ octalmode characterAtIndex: 2 ] - '0' );
    ot = oth = ( [ octalmode characterAtIndex: 3 ] - '0' );
    
    if (( adv = ( adv - 4 )) >= 0 ) [ setuidSwitch setState: NSOnState ];
    if (( adv = ( adv - 2 )) >= 0
            || ( adv == -2 && a == 2 )
            || ( adv == -3 && a == 3 )) [ setgidSwitch setState: NSOnState ];
    if (( adv = ( adv - 1 )) >= 0 
            || adv == -2 || adv == -4 || adv == -6 ) [ stickySwitch setState: NSOnState ];

    if (( own = ( own - 4 )) >= 0 ) [ owReadSwitch setState: NSOnState ];
    if (( own = ( own - 2 )) >= 0
            || ( own == -2 && ow == 2 )
            || ( own == -3 && ow == 3 )) [ owWriteSwitch setState: NSOnState ];
    if (( own = ( own - 1 )) >= 0 
            || own == -2 || own == -4 || own == -6 ) [ owExecSwitch setState: NSOnState ];

    if (( grp = ( grp - 4 )) >= 0 ) [ grReadSwitch setState: NSOnState ];
    if (( grp = ( grp - 2 )) >= 0
            || ( grp == -2 && g == 2 )
            || ( grp == -3 && g == 3 )) [ grWriteSwitch setState: NSOnState ];
    if (( grp = ( grp - 1 )) >= 0
            || grp == -2 || own == -4 || own == -6 ) [ grExecSwitch setState: NSOnState ];
   
    if (( oth = ( oth - 4 )) >= 0 ) [ otReadSwitch setState: NSOnState ];
    if (( oth = ( oth - 2 )) >= 0
            || ( oth == -2 && ot == 2 )
            || ( grp == -3 && ot == 3 )) [ otWriteSwitch setState: NSOnState ];
    if (( oth = ( oth - 1 )) >= 0
            || oth == -2 || oth == -4 || oth == -6 ) [ otExecSwitch setState: NSOnState ];

    [ octalModeField setStringValue: modeString ];
}

- ( IBAction )permissionSwitchClick: ( id )sender
{
    int			own = 0, grp = 0, oth = 0, bit = 0;
    
    if ( [ setuidSwitch state ] == NSOnState ) bit += 4;
    if ( [ setgidSwitch state ] == NSOnState ) bit += 2;
    if ( [ stickySwitch state ] == NSOnState ) bit += 1;
    
    if ( [ owReadSwitch state ] == NSOnState ) own += 4;
    if ( [ owWriteSwitch state ] == NSOnState ) own += 2;
    if ( [ owExecSwitch state ] == NSOnState ) own += 1;

    if ( [ grReadSwitch state ] == NSOnState ) grp += 4;
    if ( [ grWriteSwitch state ] == NSOnState ) grp += 2;
    if ( [ grExecSwitch state ] == NSOnState ) grp += 1;

    if ( [ otReadSwitch state ] == NSOnState ) oth += 4;
    if ( [ otWriteSwitch state ] == NSOnState ) oth += 2;
    if ( [ otExecSwitch state ] == NSOnState ) oth += 1;
    
    [ octalModeField setStringValue: [ NSString stringWithFormat: @"%d%d%d%d",
            bit, own, grp, oth ]];
	    
    [ self setPermissions: sender ];
}

- ( IBAction )setPermissions: ( id )sender
{
    NSString			*newmode = nil;
    NSMutableArray              *lines = nil;
    NSEnumerator		*en;
    id				dobj;
    int				i, err = 0;
    
    /* eventually NSFormatter should prevent users from entering alpha chars */
    newmode = [ octalModeField stringValue ];
    
    for ( i = 0; i < [ newmode length ]; i++ ) {
        if ( ! isdigit( [ newmode characterAtIndex: i ] )) {
            err = 1;
            break;
        }
    }
    
    if ( err || [ newmode length ] != 4 ) {
        [ octalModeField setStringValue: [ permissionsField stringValue ]];
        NSBeginAlertSheet( @"Error",
                          @"OK", @"", @"", tWindow,
                          self, NULL, nil, NULL,
                          NSLocalizedString( @"The permission set must be in octal",
					    @"The permission set must be in octal" ));
        return;
    }
    
    lines = [ self transcriptContents ];
    
    [[ self undoManager ] setActionName: NSLocalizedString( @"Permission Change",
                                                            @"Permission Change" ) ];
    
    en = [ tContentsTableView selectedRowEnumerator ];
    while (( dobj = [ en nextObject ] ) != nil ) {
        int                 index = [ dobj intValue ];
        
        if ( mode == SEARCH ) {
            [[ searchResults objectAtIndex: index ] setObject: newmode
                                                        forKey: @"perm" ];
            index = strtol(( char * )[[[ searchResults objectAtIndex: [ dobj intValue ]]
                                                objectForKey: @"tindex" ] UTF8String ],
                                                NULL, 10 );
        }
        
        [[ lines objectAtIndex: index ] setObject: newmode forKey: @"perm" ];
    }
    
    [ tWindow setDocumentEdited: YES ];
    if ( [ tContentsTableView numberOfSelectedRows ] == 1 ) {
	[ self showInfo: nil ];
    }
}

- ( IBAction )saveManuallyEditedTranscriptLine: ( id )sender
{
    NSString                    *line = [ editedTranscriptLineField stringValue ];
    NSMutableDictionary         *parsedLine = nil;
    int                         index = [ tContentsTableView selectedRow ];
    
    if ( ! line || [ line length ] == 0 ) {
        NSBeginAlertSheet( NSLocalizedString( @"Error", @"Error" ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"",
                            tWindow, self, NULL, nil, NULL,
                            NSLocalizedString( @"Invalid transcript line",
                                                @"Invalid transcript line" ));
        return;
    }
    
    parsedLine = [ NSMutableDictionary dictionaryForLine: line ];
    if ( parsedLine == nil ) {
        NSBeginAlertSheet( NSLocalizedString( @"Error", @"Error" ),
                            NSLocalizedString( @"OK", @"OK" ), @"", @"",
                            tWindow, self, NULL, nil, NULL,
                            NSLocalizedString( @"%@: Invalid transcript line",
                                        @"%@: Invalid transcript line"), line );
        return;
    }
    
    [ parsedLines replaceObjectAtIndex: index withObject: parsedLine ];
    [ tContentsTableView reloadData ];
}

- ( NSData * )dataRepresentationOfType: ( NSString * )aType
{
    return( nil );
}

- ( BOOL )loadDataRepresentation: ( NSData * )data ofType: ( NSString * )aType
{
    return( YES );
}

#ifdef HAS_COLUMN_VIEW
- ( BOOL )isVisibleObject: ( NSString * )object atIndex: ( int )index
        withParent: ( NSString * )parent
{
    /* sanity check */
    if ( index > [ parsedLines count ] || index < 0 ) {
        NSRunAlertPanel( @"Index out of range",
                        @"You're trying to have me access an invalid transcript line.",
                        @"Exit", @"", @"" );
        exit( 1 );
    }

    if ( ![ parent isEqualToString: @"./" ] ) {
        if ( [[ object stringByDeletingLastPathComponent ]
                isEqualToString: parent ] ) {
            return( YES );
        }
    } else {
        if ( [[ object stringByDeletingLastPathComponent ]
                isEqualToString: @"." ] ) {
            return( YES );
        }
    }

    return( NO );
}
#endif /* HAS_COLUMN_VIEW */

#ifdef HAS_COLUMN_VIEW
- ( void )arrangeColumn: ( int )column withOffset: ( int )offset toOffset: ( int )end
{
    int			i;
    NSMutableArray	*curColumnItems = [[ NSMutableArray alloc ] init ];
    NSString		*parent;

    parent = [[ parsedLines objectAtIndex: offset ] objectForKey: @"path" ];
   
    for ( i = 1; i < [ parsedLines count ]; i++ ) {
        if (( i == end ) && end != 0 ) {
            break;	/* we've come to the next object not a part of the branch we're interested in */
        }
        
        if ( [ self isVisibleObject: [[ parsedLines objectAtIndex: i ] objectForKey: @"path" ]
                    atIndex: i withParent: parent ] ) {
            [ curColumnItems addObject: [ parsedLines objectAtIndex: i ]];
        }
    }

    [ columnContents addObject: curColumnItems ];    
    [ curColumnItems release ];
    
    return;
}
#endif /* HAS_COLUMN_VIEW */

- ( void )calculateAndDisplayLoadsetSize
{
    NSArray             *contents = [ self transcriptContents ];
    NSDictionary        *entry = nil;
    NSString            *s = nil;
    CFRange             range = { 0, 0 };
    off_t               lsize = 0, itemsize = 0;
    unsigned int        i = 0;
    char                buf[ MAXPATHLEN ]  = { 0 };
    
    for ( i = 0; i < [ contents count ]; i++ ) {
        entry = [ contents objectAtIndex: i ];
        s = [ entry objectForKey: @"size" ];
        
        if ( s == nil ) {
            continue;
        }
        
        range.length = ( CFIndex )[ s length ];
        
        /*
         * this is faster than -UTF8String, since
         * -UTF8String might need to allocate a buffer
         * when called.
         */
        if ( CFStringGetBytes(( CFStringRef )s, range,
                kCFStringEncodingUTF8, 0,
                FALSE, ( UInt8 * )buf, MAXPATHLEN, NULL ) == 0 ) {
            NSLog( @"CFStringGetBytes for %@ failed", s );
            continue;
        }

        itemsize = strtoll(( const char * )buf, NULL, 10 );
        if ( errno == ERANGE || errno == EINVAL ) {
            NSLog( @"strtoll %s failed: %s", buf, strerror( errno ));
            continue;
        }
        
        if ( [[ entry objectForKey: @"pm" ] isEqualToString: @"-" ] ) {
            lsize -= itemsize;
        } else {
            lsize += itemsize;
        }
        
        memset( buf, '\0', sizeof( buf ));
    }
    
    [ loadsetSizeField setStringValue:
            [[ NSString stringWithFormat: @"%lld", lsize ] descriptiveSizeString ]];
}

- ( void )updateIndices
{
    int			i;
    NSAutoreleasePool   *p = [[ NSAutoreleasePool alloc ] init ];
    
    for ( i = 0; i < [[ self transcriptContents ] count ]; i++ ) {
        [[[ self transcriptContents ] objectAtIndex: i ] setObject:
                [ NSString stringWithFormat: @"%d", i ] forKey: @"tindex" ];
                                            
        if ( i % 100 == 0 ) {
            [ p release ];
            p = [[ NSAutoreleasePool alloc ] init ];
        }
    }
    [ p release ];
}

- ( oneway void )readError: ( char * )errmsg fromThreadWithID: ( int )ID
{
    if ( ID != [ self helperThreadID ] ) {
        return;
    }
    
    NSRunAlertPanel( @"Error reading transcript:",
        [ NSString stringWithUTF8String: errmsg ], @"OK", @"", @"" );
}

- ( void )setMaxLoadIndicatorValue: ( double )value
{
    [ loadProgressIndicator setUsesThreadedAnimation: YES ];
    [ loadProgressIndicator setIndeterminate: NO ];
    [ loadProgressIndicator setMaxValue: value ];
    [ loadProgressIndicator setMinValue: 0.0 ];
    [ loadProgressIndicator setDoubleValue: 0.0 ];
}

- ( void )setReadProgressMessage: ( NSString * )message
{
    NSString                    *progmsg = nil;
    double                      d = [ loadProgressIndicator doubleValue ];
    
    progmsg = [[ NSString alloc ] initWithString: message ];
    
    if ( [ loadProgressIndicator superview ] == nil ) {
        [ transcriptHeaderView addSubview: loadProgressIndicator ];
        [ loadProgressIndicator setFrameOrigin:
                    NSMakePoint(( [ transcriptHeaderView frame ].size.width - [ loadProgressIndicator frame ].size.width - 12 ),
                                    [ loadProgressIndicator frame ].origin.y ) ];
    }
    
    [ loadProgressIndicator setUsesThreadedAnimation: YES ];
                             
    [ lineCountField setStringValue: progmsg ];
    if (( d + 1000 ) > [ loadProgressIndicator maxValue ] ) {
        [ loadProgressIndicator setIndeterminate: YES ];
        [ loadProgressIndicator startAnimation: nil ];
    } else {
        [ loadProgressIndicator incrementBy: ( double )1000.0 ];
    }
    
    [ loadProgressIndicator displayIfNeeded ];
    [ lineCountField displayIfNeeded ];
    
    [ progmsg release ];
}

- ( oneway void )addLines: ( NSArray * )lines fromThreadWithID: ( int )ID
{
    if ( ID != [ self helperThreadID ] || [ lines count ] == 0 ) {
        return;
    }

    if ( parsedLines == nil ) {
        parsedLines = [[ NSMutableArray alloc ] initWithArray: lines ];
    } else {
        [ parsedLines addObjectsFromArray: lines ];
    }
    /* XXXX sort should occur only after all lines have been added */
    [ parsedLines sortUsingFunction: &pathcompare context: NULL ];
    [ tContentsTableView reloadData ];
    [ loadProgressIndicator setIndeterminate: YES ];
}

- ( void )readTranscriptCopyAtPath: ( NSString * )path threadID: ( int )ID
{
    NSData		*fileData = nil;
    NSArray             *lines = nil;
    NSMutableArray      *pLines = nil;
    NSMutableDictionary *dict = nil;
    NSString            *filecontents = nil;
    NSAutoreleasePool   *poole = nil;
    int                 i, total;

    if ( ID != [ self helperThreadID ] ) {
        return;
    }

    if ( ! [[ NSFileManager defaultManager ] fileExistsAtPath: path ] ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"%@: no such file or directory",
                                    @"%@: no such file or directory" ),
                NSLocalizedString( @"OK", @"OK" ), @"", @"", path );
        return;
    }

    pLines = [[ NSMutableArray alloc ] init ];
    fileData = [ NSData dataWithContentsOfFile: path ];
    filecontents = [[ NSString alloc ] initWithData: fileData
                        encoding: NSUTF8StringEncoding ];
    lines = [ filecontents componentsSeparatedByString: @"\n" ];

    [ self setMaxLoadIndicatorValue: ( double )( [ lines count ] - 1 ) ];

    total = [ lines count ];
    
    poole = [[ NSAutoreleasePool alloc ] init ];
    for ( i = 0; i < total; i++ ) {
        if ( [[ lines objectAtIndex: i ] isEqualToString: @"" ] ) {
            continue;
        }
        
        dict = [ NSMutableDictionary dictionaryForLine: [ lines objectAtIndex: i ]];
        [ pLines addObject: dict ];
        
        if ( i % 1000 == 0 ) {
            [ self setReadProgressMessage:
                    [ NSString stringWithFormat:
                        @"Parsed %d of %d lines...",
                        i, ( total - 1 ) ]];
            [ poole release ];
            dict = nil;
            poole = [[ NSAutoreleasePool alloc ] init ];
        }
    }
    [ poole release ];
    
    [ filecontents release ];
        
    [ self setReadProgressMessage: [ NSString stringWithFormat:
            NSLocalizedString( @"Parsed %d of %d lines. Displaying transcript...",
                                @"Parsed %d of %d lines. Displaying transcript..." ),
                                ( total - 1 ), ( total - 1 ) ]];

    [ self setTranscriptContents: pLines ];
    [ pLines release ];
    [ self calculateAndDisplayLoadsetSize ];
    
    if ( unlink(( char * )[ path UTF8String ] ) < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"unlink %@: %s", @"unlink %@: %s" ),
                NSLocalizedString( @"OK", @"OK" ), @"", @"", path, strerror( errno ));
    }
}

/*
 * see DOSetupCompleteInThreadWithID: below.
 */
- ( BOOL )readFromFile: ( NSString * )fileName ofType: ( NSString * )type
{
    return( YES );
}

- ( BOOL )readFromURL: ( NSURL * )url ofType: ( NSString * )type
{
    return( [ self readFromFile: [ url path ] ofType: type ] );
}

- ( void )readFile: ( NSDictionary * )contextInfo
{
    int                 action = [[ contextInfo objectForKey: @"action" ] intValue ];
    int                 recurse = [[[ NSUserDefaults standardUserDefaults ]
                                            objectForKey: @"FsdiffType" ] intValue ];
    NSString            *arg1 = [ contextInfo objectForKey: @"filea" ];
    NSString            *arg2 = [ contextInfo objectForKey: @"fileb" ];
    NSString            *fsdiffType = @"-1";
    NSArray             *args = nil;
    
    switch ( action ) {
    case FSDIFF:
        if ( recurse ) {
            fsdiffType = @"-K/dev/null";
        }
        args = [ NSArray arrayWithObjects: @"-AExecuteCommand",
					[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]],
					@"--", @"/usr/local/bin/fsdiff",
                                        @"-csha1", fsdiffType, arg1, nil ];
        break;
	
    case TREAD:
        args = [ NSArray arrayWithObjects: @"-AReadTranscript", 
					[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]],
					@"--", arg1, arg2, nil ];
        break;
	
    case TWRITE:
        args = [ NSArray arrayWithObjects: @"-A", @"WriteTranscript", 
					[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]],
					@"--", arg1, arg2, nil ];
        break;
    }

    [ self executeAuthorizedAction: action arguments: args ];
}

- ( void )readFinishedFromThreadWithID: ( int )ID
{
    if ( ID != [ self helperThreadID ] ) {
        return;
    }

    [ self updateIndices ];

    [ loadProgressIndicator setIndeterminate: YES ];
    [ loadProgressIndicator stopAnimation: nil ];
    [ loadProgressIndicator retain ];
    [ loadProgressIndicator removeFromSuperview ];

    [ lineCountField setStringValue: [ NSString stringWithFormat: @"%d line%@",
                                        [ parsedLines count ],
                                        ( [ parsedLines count ] == 1 ? @"" : @"s" ) ]];
    //[ tContentsTableView reloadData ];
    [[ tContentsTableView window ] makeFirstResponder: tContentsTableView ];
    
    if ( INITIAL_READ ) INITIAL_READ = 0;
}

- ( void )writeFinishedFromThreadWithID: ( int )ID
{
    if ( ID != [ self helperThreadID ] ) {
        return;
    }

    [ loadProgressIndicator stopAnimation: nil ];
    [ loadProgressIndicator retain ];
    [ loadProgressIndicator removeFromSuperview ];
}

- ( oneway void )authorizationFailedInThreadWithID: ( int )ID
{
    if ( ID != [ self helperThreadID ] ) {
        return;
    }
    
    NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
            NSLocalizedString( @"Authorization failed.", @"Authorization failed." ),
            NSLocalizedString( @"OK", @"OK" ), @"", @"" );
}

- ( oneway void )DOSetupCompleteInThreadWithID: ( int )ID
{
    char        tmp[ MAXPATHLEN ] = "/tmp/rte-trans.T-XXXXX";
    NSArray     *args = nil;
    
    if ( ID != [ self helperThreadID ] || [ self fileName ] == nil ) {
        return;
    }

    if ( mktemp( tmp ) < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
                NSLocalizedString( @"mktemp: %s", @"mktemp: %s" ),
                NSLocalizedString( @"OK", @"OK" ), @"", @"", strerror( errno ));
        exit( 2 );
    }

    if ( [ self fileName ] != nil ) {
        args = [ NSArray arrayWithObjects: @"-AReadTranscript",
			[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]],
			@"--", [ self fileName ],
                        [ NSString stringWithUTF8String: tmp ], nil ];
        [ self executeAuthorizedAction: TREAD arguments: args ];
    }
}

- ( BOOL )executeAuthorizedAction: ( int )action arguments: ( NSArray * )arguments
{
    /* if DO isn't completely set up yet, wait for notification */
    if ( arw == nil ) {
        return( NO );
    }
    
    if ( [ loadProgressIndicator superview ] == nil ) {
        [ transcriptHeaderView addSubview: loadProgressIndicator ];
        [ loadProgressIndicator setUsesThreadedAnimation: YES ];
        [ loadProgressIndicator setFrameOrigin:
                    NSMakePoint(( [ transcriptHeaderView frame ].size.width -
                                    [ loadProgressIndicator frame ].size.width - 12 ),
                                    [ loadProgressIndicator frame ].origin.y ) ];
    }
    [ loadProgressIndicator startAnimation: nil ];
    [ loadProgressIndicator displayIfNeeded ];

    [ arw executeAction: action arguments: arguments
            controller: self ];
            
    return( YES );
}

- ( IBAction )saveDocument: ( id )sender
{
    if ( [ self fileName ] == nil ) {
        [ self saveDocumentAs: sender ];
    } else {
        [ super saveDocument: sender ];
    }
    [ tWindow setDocumentEdited: NO ];
}

- ( void )saveDocumentWithDelegate: ( id )delegate
        didSaveSelector: ( SEL )didSaveSelector contextInfo: ( void * )contextInfo
{
    [ self writeWithBackupToFile: [ self fileName ]
            ofType: @"T" saveOperation: NSSaveOperation ];
            
    /* since we've overridden the default clear, do it here */
    [ self updateChangeCount: NSChangeCleared ];
}

- ( BOOL )writeWithBackupToFile: ( NSString * )filename
            ofType: ( NSString * )filetype
            saveOperation: ( NSSaveOperationType )saveOp
{
    return( [ self writeToFile: filename ofType: filetype ] );
}

- ( BOOL )writeToFile: ( NSString * )filename ofType: ( NSString * )type
{
    int			i, fd, count, pct = 0;
    float               increment;
    FILE		*outtran;
    char		tmpfile[ MAXPATHLEN ] = "/tmp/rte-trans.T.XXXXX";
    char                *line = NULL;
    NSArray             *saveArgs = nil, *lines = [ self transcriptContents ];
    NSAutoreleasePool   *poole = nil;
    NSDictionary        *d = nil;
    
    if (( count = [ lines count ] ) == 0 ) {
        NSBeep();
        goto WRITEFAILED;
    }

    if (( fd = mkstemp( tmpfile )) < 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
            @"mkstemp %s: %s", NSLocalizedString( @"OK", @"OK" ), @"", @"", tmpfile, strerror( errno ));
        goto WRITEFAILED;
    }
    
    if (( outtran = fdopen( fd, "w" )) == NULL ) {
        ( void )close( fd );
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
            @"fdopen %s: %s", NSLocalizedString( @"OK", @"OK" ), @"", @"", tmpfile, strerror( errno ) );
        goto WRITEFAILED;
    }
    
    [ lineCountField setStringValue: NSLocalizedString( @"Saving...", @"Saving..." ) ];
    [ lineCountField displayIfNeeded ];

    if ( [ loadProgressIndicator superview ] == nil ) {
        [ transcriptHeaderView addSubview: loadProgressIndicator ];
        [ loadProgressIndicator setIndeterminate: NO ];
        [ loadProgressIndicator setUsesThreadedAnimation: YES ];
        [ loadProgressIndicator setFrameOrigin:
                    NSMakePoint(( [ transcriptHeaderView frame ].size.width - [ loadProgressIndicator frame ].size.width - 12 ),
                                    [ loadProgressIndicator frame ].origin.y ) ];;
    }
    [ loadProgressIndicator setMinValue: 0.0 ];
    [ loadProgressIndicator setMaxValue: 100.0 ];
    [ loadProgressIndicator setDoubleValue: 0.0 ];
    [ loadProgressIndicator displayIfNeeded ];
    
    increment = ( count * 0.05 );
    
    poole = [[ NSAutoreleasePool alloc ] init ];
    for ( i = 0; i < count; i++ ) {
        d = [ lines objectAtIndex: i ];
        
        /* this must be a cString for it to be a clean radmind transcript line */
        line = strdup( [[ NSString transcriptLineFromDictionary: d ] cString ] );
        
        if ( line == NULL ) {
            NSLog( @"strdup failed! Error: %s", strerror( errno ));
            exit( 2 );
        }
        
        fprintf( outtran, "%s", line );

        free( line );
        line = NULL;
        d = nil;

        if ( i % ( int )increment == 0 ) {
            [ loadProgressIndicator setDoubleValue: ( double )pct ];
            [ loadProgressIndicator displayIfNeeded ];
            [ poole release ];
            poole = [[ NSAutoreleasePool alloc ] init ];
            pct += 5;
        }
    }
    [ poole release ];
    
    [ loadProgressIndicator stopAnimation: nil ];
    [ loadProgressIndicator retain ];
    [ loadProgressIndicator removeFromSuperview ];
    
    if ( fclose( outtran ) != 0 ) {
        NSRunAlertPanel( NSLocalizedString( @"Error", @"Error" ),
            @"fclose %s: %s", NSLocalizedString( @"OK", @"OK" ),
	    @"", @"", tmpfile, strerror( errno ) );
        goto WRITEFAILED;
    }
    
    saveArgs = [ NSArray arrayWithObjects: @"-A", @"WriteTranscript", 
					[ NSString stringWithFormat: @"-U%@",
					    [[ self delegate ] sessionUserName ]], @"--",
                [ NSString stringWithUTF8String: tmpfile ], filename, nil ];

    [ self executeAuthorizedAction: TWRITE arguments: saveArgs ];

    [ self setFileName: filename ];
    [ lineCountField setStringValue:
            [ NSString stringWithFormat: @"%d line%@", count, ( count > 1 ? @"s" : @"" ) ]];
    
    return( YES );
    
WRITEFAILED:
    return( NO );
}

- ( BOOL )writeToURL: ( NSURL * )url ofType: ( NSString * )type
{
    return( [ self writeToFile: [ url path ] ofType: type ] );
}

- ( NSString * )retrievePassword
{
    return( @"NoPass" );
}

/* tableview data source methods */
- ( int )numberOfRowsInTableView: ( NSTableView * )aTableView
{
    int		count = 0;
    
    switch ( mode ) {
    case SEARCH:
        if ( searchResults == nil ) return( 0 );
        count = [ searchResults count ];
        break;
        
    default:
    case NORMAL:
        if ( [ self transcriptContents ] == nil ) return( 0 );
        count = [[ self transcriptContents ] count ];
    }
    
    return( count );
}

- ( void )tableView: ( NSTableView * )tableView willDisplayCell: ( id )aCell
        forTableColumn: ( NSTableColumn * )aTableColumn row: ( int )rowIndex
{
    NSMutableArray      *lines = ( mode == SEARCH ? searchResults : [ self transcriptContents ] );
    
    if ( rowIndex < 0 ) {
        NSLog( @"%d: invalid row.", rowIndex );
        return;
    }
    
    [ aCell setFont: [ NSFont systemFontOfSize: fontsize ]];
    
    if ( [[ aTableColumn identifier ] isEqualToString: @"Type" ] ) {
        NSString	*type = [[ lines objectAtIndex: rowIndex ]
                                    objectForKey: @"type" ];
        if ( [ type characterAtIndex: 0 ] == '0' ) [ aCell setStringValue: @"" ];
        else [ aCell setStringValue: type ];
    } else if ( [[ aTableColumn identifier ] isEqualToString: @"Path" ] ) {
        NSString        *path = [[ lines objectAtIndex: rowIndex ]
                                    objectForKey: @"path" ];
        if ( [ path characterAtIndex: 0 ] == '#' ) {
            [ aCell setTextColor: [ NSColor blueColor ]];
        }
        return;
    } else if ( [[ aTableColumn identifier ] isEqualToString: @"PlusMinus" ] ) {
        NSString        *pm = [[ lines objectAtIndex: rowIndex ] objectForKey: @"pm" ];

        if ( pm != nil && [ pm characterAtIndex: 0 ] != '0' ) {
            [ aCell setStringValue: [[ lines objectAtIndex: rowIndex ] objectForKey: @"pm" ]];
        } else {
            [ aCell setStringValue: @"" ];
        }
    } else if ( [[ aTableColumn identifier ] isEqualToString: @"linenumbers" ] ) {
        [ aCell setStringValue: [ NSString stringWithFormat: @"%d", ( rowIndex + 1 ) ]];
    }
    [ aCell setTextColor: [ NSColor blackColor ]];
}

- ( NSDragOperation )tableView: ( NSTableView * )aTableView validateDrop:( id <NSDraggingInfo> )info
    proposedRow: ( int )row proposedDropOperation: ( NSTableViewDropOperation )operation
{
    if ( operation == NSTableViewDropOn && row == -1 ) {
        return( NSDragOperationCopy );
    } else if ( operation == NSTableViewDropAbove ) {
        [ aTableView setDropRow: -1 dropOperation: NSTableViewDropOn ];
        return( NSDragOperationCopy );
    }

    return( NSDragOperationNone );
}

- ( BOOL )tableView: ( NSTableView * )aTableView acceptDrop: ( id <NSDraggingInfo> )info
    row: ( int )row dropOperation: ( NSTableViewDropOperation )operation
{
    NSPasteboard		*pb = [ info draggingPasteboard ];
    id				dragData = nil;
    NSString			*type = [ pb availableTypeFromArray:
                                            [ NSArray arrayWithObject:
                                                NSStringPboardType ]];
    int				i;

    if ( type != nil ) {
        NSArray		*lines;
        
        dragData = [ pb stringForType: type ];
        
        if ( ! [ dragData isKindOfClass: [ NSString class ]] ) return( NO );

        lines = [ dragData componentsSeparatedByString: @"\n" ];
        for ( i = 0; i < [ lines count ]; i++ ) {
            [ self addLines: [ NSArray arrayWithObject:
                    [ NSMutableDictionary dictionaryForLine: [ lines objectAtIndex: i ]]]
                    fromThreadWithID: [ self helperThreadID ]];
        }
        [ tWindow setDocumentEdited: YES ];
        [ lineCountField setStringValue:
            [ NSString stringWithFormat: @"%d lines", [ parsedLines count ]]];
        [ aTableView reloadData ];
        [ self updateChangeCount: NSChangeDone ];
        
        return( YES );
    } else {
        NSString        *path = [[ NSUserDefaults standardUserDefaults ]
                                        objectForKey: @"RTEBasePath" ];
        BOOL		isDir;
        int		rc;
        
        dragData = [ pb propertyListForType: [ pb availableTypeFromArray:
                        [ NSArray arrayWithObject: NSFilenamesPboardType ]]];

        if ( dragData == nil ) return( NO );
        
        if ( ! [[ NSFileManager defaultManager ] fileExistsAtPath:
                                                    [ dragData objectAtIndex: 0 ]
                                                isDirectory: &isDir ] ) {
            NSRunAlertPanel( @"Error", @"%@: no such file or directory.",
                    @"OK", @"", @"", [ dragData objectAtIndex: 0 ] );
            return( NO );
        }
        if ( isDir ) {
            [ NSApp activateIgnoringOtherApps: YES ];
            rc = NSRunAlertPanel( @"What would you like to add to this transcript?",
                @"You can add only the directory, or the directory and its contents",
                @"Add Directory", @"Cancel", @"Add Directory and Contents" );
            
            switch ( rc ) {
            default:
            case NSAlertDefaultReturn:
                [[ NSUserDefaults standardUserDefaults ] setObject:
                                                [ NSNumber numberWithInt: 0 ]
                                                        forKey: @"FsdiffType" ];
                break;
                
            case NSAlertAlternateReturn:
                return( NO );
            
            case NSAlertOtherReturn:
                [[ NSUserDefaults standardUserDefaults ] setObject:
                                                [ NSNumber numberWithInt: 1 ]
                                                        forKey: @"FsdiffType" ];
                break;
            }
        }
        
        if ( path == nil || [ path isEqualToString: @"/" ] ) {
            path = [ dragData objectAtIndex: 0 ];
        } else {
            path = [ path stringByAppendingString: [ dragData objectAtIndex: 0 ]];
        }
                
        [ self readFile: [ NSDictionary dictionaryWithObjectsAndKeys:
                                    [ NSNumber numberWithInt: FSDIFF ], @"action",
                                    path, @"filea",
                                    @"", @"fileb", nil ]];
        
        [ aTableView noteNumberOfRowsChanged ];
        [ tWindow setDocumentEdited: YES ];
        [ self updateChangeCount: NSChangeDone ];
        return( YES );
    }
    
    return( NO );
}

- ( id )tableView: ( NSTableView * )aTableView
        objectValueForTableColumn: ( NSTableColumn * )aTableColumn
        row: ( int )rowIndex
{
    NSMutableArray      *lines = ( mode == SEARCH ? searchResults : [ self transcriptContents ] );
    
    if ( [[ aTableColumn identifier ] isEqualToString: @"Path" ] ) {
        NSString            *path = [[ lines objectAtIndex: rowIndex ] objectForKey: @"path" ];
        NSAttributedString  *attrString = [[[ NSAttributedString alloc ] initWithString: path ] autorelease ];
        double              width = [ aTableColumn width ];

        attrString = [ attrString ellipsisAbbreviatedStringForWidth: width ];
        
        return( attrString );
    }
    
    /* rest handled by willDisplayCell: */
    return( @"" );
}

- ( void )tableViewSelectionDidChange: ( NSNotification * )notification
{
    if ( ! [ transcriptSplitView isSubviewCollapsed: infoTabView ] ) {
        [ self showInfo: nil ];
    }
    
    if ( [ transcriptLineDrawer state ] == NSDrawerOpenState ) {
        int         row = [ tContentsTableView selectedRow ];
        
        if ( row >= 0 ) {
            [ transcriptLineView setEditable: YES ];
            [ transcriptLineView setString:
                [ NSString transcriptLineFromDictionary:
                [ ( mode == SEARCH ? searchResults : parsedLines ) objectAtIndex: row ]]];
            [ transcriptLineView setEditable: NO ];
        }
    }
}

/* window delegate */
- ( void )windowWillClose: ( NSNotification * )notification
{
    id              window = [ notification object ];
    NSConnection    *conn = nil;
    
    if ( [ window isEqual: tWindow ] ) {
        conn = [[ ( id )arw connectionForProxy ] retain ];
        
        [ window setDelegate: nil ];
        
        if ( [ conn isValid ] ) {
            [ arw disconnect ];
            [ conn release ];
        }
        
        if ( parsedLines != nil ) {
            [ parsedLines release ];
        }
        [ searchResults release ];

        [ self autorelease ];
    }
}

/* splitview delegate methods */
- ( BOOL )splitView: ( NSSplitView * )splitview
            canCollapseSubview: ( NSView * )subview
{
    return(( subview == infoTabView ));
}

- ( float )splitView: ( NSSplitView * )splitview
            constrainMaxCoordinate: ( float )proposedMax
            ofSubviewAt: ( int )offset
{
    return(( proposedMax - 316.0 ));
}

- ( float )splitView: ( NSSplitView * )splitview
            constrainMinCoordinate: ( float )proposedMin
            ofSubviewAt: ( int )offset
{
    return(( proposedMin + 316.0 ));
}

@end
