/********************************************************************************
*                                                                               *
*              FoxSweeper game                                                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 2002 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This program is free software; you can redistribute it and/or modify          *
* it under the terms of the GNU General Public License as published by          *
* the Free Software Foundation; either version 2 of the License, or             *
* (at your option) any later version.                                           *
*                                                                               *
* This program is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
* GNU General Public License for more details.                                  *
*                                                                               *
* You should have received a copy of the GNU General Public License             *
* along with this program; if not, write to the Free Software                   *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#include <fox/fx.h>
#include "fxex.h"
#include "Icons.h"
#include "GridPointerArray.h"
#include "fsButton.h"
#include "fsGridPointerArray.h"
#include "Settings.h"
#include "HighScores.h"
#include "FoxSweeper.h"

extern FXbool playAgain;

FXDEFMAP(FoxSweeper) FoxSweeperMap [] = {
  FXMAPFUNC (SEL_CLOSE  , FoxSweeper::ID_MAINWINDOW,  FoxSweeper::onCmdQuit       ),
  FXMAPFUNC (SEL_SIGNAL , FoxSweeper::ID_QUIT,        FoxSweeper::onCmdQuit       ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_QUIT,        FoxSweeper::onCmdQuit       ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_NEW_GAME,    FoxSweeper::onCmdNewGame    ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_PAUSE,       FoxSweeper::onCmdPause      ),
  FXMAPFUNC (SEL_UPDATE , FoxSweeper::ID_PAUSE,       FoxSweeper::onUpdPause      ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_SETTINGS,    FoxSweeper::onCmdSettings   ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_HIGH_SCORE,  FoxSweeper::onCmdHighScore  ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_ABOUT,       FoxSweeper::onCmdAbout      ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_LEFT_CLICK,  FoxSweeper::onCmdLeftClick  ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_MIDDLE_CLICK,FoxSweeper::onCmdMiddleClick),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_RIGHT_CLICK, FoxSweeper::onCmdRightClick ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_NROWS,       FoxSweeper::onCmdNRows      ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_NCOLS,       FoxSweeper::onCmdNCols      ),
  FXMAPFUNC (SEL_COMMAND, FoxSweeper::ID_NBOMBS,      FoxSweeper::onCmdNBombs     ),
  FXMAPFUNC (SEL_UPDATE , FoxSweeper::ID_NROWS,       FoxSweeper::onUpdNRows      ),
  FXMAPFUNC (SEL_UPDATE , FoxSweeper::ID_NCOLS,       FoxSweeper::onUpdNCols      ),
  FXMAPFUNC (SEL_UPDATE , FoxSweeper::ID_NBOMBS,      FoxSweeper::onUpdNBombs     ),
  FXMAPFUNC (SEL_TIMEOUT, FoxSweeper::ID_TIMER,       FoxSweeper::onTimer         ),
  FXMAPFUNCS(SEL_COMMAND, FoxSweeper::ID_BEGINNER,    FoxSweeper::ID_CUSTOM, FoxSweeper::onCmdMode),
  FXMAPFUNCS(SEL_UPDATE , FoxSweeper::ID_BEGINNER,    FoxSweeper::ID_CUSTOM, FoxSweeper::onUpdMode),
  FXMAPFUNCS(SEL_UPDATE , FoxSweeper::ID_HS_BEGINNER, FoxSweeper::ID_HS_CUSTOM, FoxSweeper::onUpdHighScore),
  };
FXIMPLEMENT(FoxSweeper,FXMainWindow,FoxSweeperMap,ARRAYNUMBER(FoxSweeperMap))

FXString FoxSweeper::gameLevel[4] = {"Beginner","Intermediate","Advanced","Custom"};
FXuint FoxSweeper::maxTime = 999;
FXuint FoxSweeper::maxBombs = 99;
FXuint FoxSweeper::minBombs = 10;
FXuint FoxSweeper::maxSize = 99;
FXuint FoxSweeper::minSize = 9;
FXuint FoxSweeper::defaultGridSize = 9;
FXuint FoxSweeper::defaultButtonSize = 25;
FXuint FoxSweeper::clockUpdateRate = 1000;

// Make the main window
FoxSweeper::FoxSweeper(FXApp* a) : FXMainWindow(a,"FoxSweeper",NULL,NULL,DECOR_TITLE|DECOR_MINIMIZE|DECOR_CLOSE){
  buttonTotal=0;
  buttonCount=0;
  bombTotal=0;
  bombCount=0;
  currentTime=0;
  mode=Beginner;

  // Set myself as the target, make some icons, make some tool tips
  icons=new Icons(getApp());
  tooltip=new FXToolTip(getApp(),TOOLTIP_PERMANENT );
  setTarget(this);
  setSelector(ID_MAINWINDOW);
  setMiniIcon(icons->Flag);
  setIcon(icons->Bomb);

  // Toolbar
  toolbarShell=new FXToolBarShell(this,FRAME_NONE);
  toolbar=new FXToolBar(this,toolbarShell,FRAME_RAISED|FRAME_THICK|LAYOUT_FILL_X, 0,0,0,0, 2,2,0,0, 0,0 );
  exitButton=new FXButton(toolbar,"Exit\tExit FoxSweeper",icons->DeleteIt,this,ID_QUIT,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  newButton=new FXButton(toolbar,"New\tNew Game",icons->Smile,this,ID_NEW_GAME,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  pauseButton=new FXButton(toolbar,"Pause\tTemporarily stops the timer.\nIt will restart when you click on a button.",icons->Clock,this,ID_PAUSE,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  settingsButton=new FXButton(toolbar,"Settings\tDisplay game settings.",icons->Properties,this,ID_SETTINGS,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  highscoreButton=new FXButton(toolbar,"Scores\tDisplay the high score window.",icons->Bomb,this,ID_HIGH_SCORE,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  aboutButton=new FXButton(toolbar,"About\tDisplay the 'About' window.",icons->MiniPenguin,this,ID_ABOUT,BUTTON_TOOLBAR|FRAME_RAISED|ICON_ABOVE_TEXT);
  countTarget=new FXDataTarget(bombCount);
  bombField=new FXLCDLabel(toolbar,3,countTarget,FXDataTarget::ID_VALUE,FRAME_SUNKEN|LAYOUT_RIGHT|JUSTIFY_RIGHT);
  bombField->setThickness(4);
  bombField->setVertical(12);
  bombField->setHorizontal(12);
  bombField->setGroove(1);
  flagLabel=new FXLabel(toolbar,"Flags:",NULL,LAYOUT_RIGHT);

  // variables
  randomSeed=fxrandomseed();
  }

FoxSweeper::~FoxSweeper(){
  delete flagLabel;
  delete bombField;
  delete countTarget;
  delete aboutButton;
  delete highscoreButton;
  delete settingsButton;
  delete pauseButton;
  delete newButton;
  delete exitButton;
  delete toolbar;
  delete toolbarShell;
  delete tooltip;
  delete icons;
  }

// create FoxApp
void FoxSweeper::create(){
  loadSettings();
  buttonTotal = gridSize.x * gridSize.y;
  gridData.size(gridSize.x,gridSize.y);
  matrix=new FXMatrix(this,gridData.ysize(),MATRIX_BY_ROWS|LAYOUT_CENTER_X|LAYOUT_CENTER_Y);
  generateBombSet();
  makeGrid();
  bombCount = bombTotal;
  FXMainWindow::create();
  show();
  }

// A random number generator for integer numbers 0..(scale-1)
FXint FoxSweeper::rand(FXint scale){
  FXint value=-1;
  while (value < 0 || value > (scale-1)){
    value = (FXint) rint(((fxrandom(randomSeed) / (RAND_MAX * 2.0) ) - 0.05) * scale);
    }
  return value;
  }

// A very simple message box
FXbool FoxSweeper::questionBox(const FXString& title,const FXString& contents){
  if (FXMessageBox::question(this,MBOX_YES_NO|PLACEMENT_OWNER,title.text(),contents.text()) == MBOX_CLICKED_YES) return TRUE;
  else return FALSE;
  }

// Save settings
void FoxSweeper::saveSettings(){
  // Write new window size back to registry
  getApp()->reg().writeIntEntry("Window Location","x",getX());
  getApp()->reg().writeIntEntry("Window Location","y",getY());

  // Grid size
  getApp()->reg().writeIntEntry("Grid Size", "x", gridSize.x);
  getApp()->reg().writeIntEntry("Grid Size", "y", gridSize.y);

  // save remaining settings
  getApp()->reg().writeIntEntry("Grid Data", "Total Bombs", bombTotal);
  getApp()->reg().writeIntEntry("Grid Data", "mode", mode);
  }

// Load settings
void FoxSweeper::loadSettings(){
  // Get last window position
  setX ( getApp()->reg().readIntEntry ( "Window Location", "x", 0 ) );
  setY ( getApp()->reg().readIntEntry ( "Window Location", "y", 0 ) );

  // get the last grid size
  gridSize.x = getApp()->reg().readIntEntry ( "Grid Size", "x", defaultGridSize );
  gridSize.y = getApp()->reg().readIntEntry ( "Grid Size", "y", defaultGridSize );

  // get remaining settings
  bombTotal = getApp()->reg().readIntEntry ( "Grid Data", "Total Bombs", minBombs );
  mode = (GameLevel) getApp()->reg().readIntEntry ( "Grid Data", "mode", Beginner );

  switch (mode){
    case Beginner:     modeBeginner(); break;
    case Intermediate: modeIntermediate(); break;
    case Advanced:     modeAdvanced(); break;
    default:           break;
    }
  }

// Generate the bomb set
void FoxSweeper::generateBombSet(){
  FXuint x,y;
  fsButtonData *b;

  while (bombCount < bombTotal){
    x = rand (gridData.xsize());
    y = rand (gridData.ysize());
    b=gridData.find(x,y);
    if (!b->isBomb){
      b->isBomb = TRUE;
      b->bombCount = -1;
      bombCount++;
      }
    }

  FXuint xm = gridData.xsize() - 1;
  FXuint ym = gridData.ysize() - 1;

  fsButtonData *tb;
  for (y=0; y<gridData.ysize(); y++){
    for (x=0; x<gridData.xsize(); x++){
      b=gridData.find(x,y);
      if (!b->isBomb){
        if (x>0  && y>0 ){ tb=gridData.find(x-1,y-1); if(tb->isBomb) b->bombCount++; }
        if (        y>0 ){ tb=gridData.find(x  ,y-1); if(tb->isBomb) b->bombCount++; }
        if (x<xm && y>0 ){ tb=gridData.find(x+1,y-1); if(tb->isBomb) b->bombCount++; }
        if (x>0         ){ tb=gridData.find(x-1,y  ); if(tb->isBomb) b->bombCount++; }
        if (x<xm        ){ tb=gridData.find(x+1,y  ); if(tb->isBomb) b->bombCount++; }
        if (x>0  && y<ym){ tb=gridData.find(x-1,y+1); if(tb->isBomb) b->bombCount++; }
        if (        y<ym){ tb=gridData.find(x  ,y+1); if(tb->isBomb) b->bombCount++; }
        if (x<xm && y<ym){ tb=gridData.find(x+1,y+1); if(tb->isBomb) b->bombCount++; }
        }
      }
    }
  }

// Build the grid
void FoxSweeper::makeGrid(){
  FXuint x,y;
  fsButtonData *b;
  FXint buttonSize = defaultButtonSize;
  for (x=0; x < gridData.xsize(); x++){
    for (y=0; y < gridData.ysize(); y++){
      b=gridData.find(x,y);
      b->theButton = new fsButton(matrix,NULL,this,ID_LEFT_CLICK,ID_MIDDLE_CLICK,ID_RIGHT_CLICK,buttonSize,buttonSize);
      }
    }
  setWidth (matrix->getDefaultWidth() + buttonSize);
  FXint width=toolbar->getDefaultWidth()+10;
  if (getWidth()<width) setWidth(width);
  setHeight(matrix->getDefaultHeight() + toolbar->getDefaultHeight() + buttonSize);
  }

// Remove the grid
void FoxSweeper::deleteGrid(){
  FXuint x,y;
  fsButtonData *f;
  fsButton *b;
  for (x=0; x < gridData.xsize(); x++){
    for (y=0; y < gridData.ysize(); y++){
      f=gridData.find(x,y);
      b=f->theButton;
      b->hide();
      delete b;
      }
    }
  }

// evaluate if this x,y location contains a bomb,
// if so, do the game finish handling for wrong click
FXbool FoxSweeper::evaluateThisLocation(FXuint x, FXuint y){
  fsButtonData *b=gridData.find(x,y);
  if (b->state != hidden) return TRUE;
  if (b->isBomb){
    // *** its a bomb!!! ***
    FXString txt = "X";
    b->theButton->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
    b->theButton->setIcon(icons->Bomb);
    b->theButton->handle(this,MKUINT(FXWindow::ID_CHECK,SEL_COMMAND),(void*) TRUE);
    getApp()->removeTimeout(this,ID_TIMER);
    txt = "You are dead (in ";
    txt += FXStringVal(currentTime);
    if (currentTime==1) txt += " second";
    else txt += " seconds";
    txt += ")!\nWould you like to play again?";
    showRemainingBombs();
    playAgain = questionBox("Argh!",txt);
    handle(this,MKUINT(ID_QUIT,SEL_COMMAND),NULL);
    return FALSE;
    }
  else{
    // *** its not a bomb - whew - better show how many bombs around ***
    evaluateSurroundingLocation(x,y);
    }
  // check if this click has caused the game end condition (eg run out of buttons)
  evalEndGame();
  return TRUE;
  }

// evaluate the surrounding locations, displaying the appropriate image.
// if the value is zero, do the same thing again
void FoxSweeper::evaluateSurroundingLocation(FXuint x, FXuint y){
  fsButtonData *b=gridData.find(x,y);
  if (b->state == opened) return;
  FXString txt = " ";
  b->theButton->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
  if (b->state == flagged){
    b->theButton->setIcon(NULL);
    bombCount++;
    buttonCount--;
    }
  b->state = opened;
  buttonCount++;

  b->theButton->handle(this,MKUINT(FXWindow::ID_CHECK,SEL_COMMAND), (void*) TRUE);
  if (b->bombCount != 0){
    txt = FXStringVal(b->bombCount);
    FXColor clr;
    switch(b->bombCount){
      case 3 : clr=FXRGB(150,100,50);  break;  // Brown
      case 7 : clr=FXRGB(255,0,0);     break;  // Red
      case 1 : clr=FXRGB(255,160,50);  break;  // Orange
      case 4 : clr=FXRGB(200,200,0);   break;  // Yellow
      case 2 : clr=FXRGB(0,200,0);     break;  // Green
      case 5 : clr=FXRGB(0,0,255);     break;  // Blue
      case 6 : clr=FXRGB(225,50,225);  break;  // Purple
      case 8 : clr=FXRGB(100,100,100); break;  // Grey
      default: clr=FXRGB(0,0,0);       break;  // Black      (resistor color codes...!)
      }
    b->theButton->setTextColor(clr);
    b->theButton->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
    return;
    }

  FXuint xm = gridData.xsize() - 1;
  FXuint ym = gridData.ysize() - 1;
  fsButtonData *tb;
  if (x>0  && y>0 ){ tb=gridData.find(x-1,y-1); if(tb->state != opened) evaluateSurroundingLocation(x-1,y-1); }
  if (        y>0 ){ tb=gridData.find(x  ,y-1); if(tb->state != opened) evaluateSurroundingLocation(x  ,y-1); }
  if (x<xm && y>0 ){ tb=gridData.find(x+1,y-1); if(tb->state != opened) evaluateSurroundingLocation(x+1,y-1); }
  if (x>0         ){ tb=gridData.find(x-1,y  ); if(tb->state != opened) evaluateSurroundingLocation(x-1,y  ); }
  if (x<xm        ){ tb=gridData.find(x+1,y  ); if(tb->state != opened) evaluateSurroundingLocation(x+1,y  ); }
  if (x>0  && y<ym){ tb=gridData.find(x-1,y+1); if(tb->state != opened) evaluateSurroundingLocation(x-1,y+1); }
  if (        y<ym){ tb=gridData.find(x  ,y+1); if(tb->state != opened) evaluateSurroundingLocation(x  ,y+1); }
  if (x<xm && y<ym){ tb=gridData.find(x+1,y+1); if(tb->state != opened) evaluateSurroundingLocation(x+1,y+1); }
  }

// have we found an end-of-game condition (eg not buttons left to click on)
void FoxSweeper::evalEndGame(){
  if (buttonCount == buttonTotal){
    if (bombCount == 0){
      getApp()->removeTimeout(this,ID_TIMER);
      addHighScore();
      FXString txt = "You won in ";
      txt += FXStringVal(currentTime);
      if (currentTime==1) txt += " second";
      else txt += " seconds";
      txt += "!\nWould you like to play again?";
      playAgain = questionBox ("Whew...!",txt);
      handle(this,MKUINT(ID_QUIT,SEL_COMMAND),NULL);
      }
    else{ // bombCount < 0
      FXMessageBox::warning (this,MBOX_OK,"What are you doing?","More flags exist than bombs...!");
      }
    }
  }

// set to hard coded beginner mode
void FoxSweeper::modeBeginner(){
  gridSize.x = minSize;
  gridSize.y = minSize;
  bombTotal = minBombs;
  }

// set to hard coded intermediate mode
void FoxSweeper::modeIntermediate(){
  gridSize.x = 16;
  gridSize.y = 16;
  bombTotal = 40;
  }

// set to hard coded advanced mode
void FoxSweeper::modeAdvanced(){
  gridSize.x = 30;
  gridSize.y = 16;
  bombTotal = 99;
  }

// show how many bombs left on the grid, after the game has completed (ie shows their location)
void FoxSweeper::showRemainingBombs(){
  FXuint x,y;
  fsButtonData *b;
  for (x=0; x < gridData.xsize(); x++){
    for (y=0; y < gridData.ysize(); y++){
      b=gridData.find(x,y);
      if ( b->isBomb && (b->state==hidden || b->state==questioned) ){
        b->theButton->setIcon(icons->Bomb);
        FXString txt = " ";
        b->theButton->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
        }
      if ( !b->isBomb && (b->state==flagged || b->state==questioned) ){
        FXString txt = "X";
        b->theButton->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
        }
      }
    }
  }

// if we managed to win, wite the time of the game to the highscore table
// provided that it was less than the previous score
void FoxSweeper::addHighScore(){
  if ( getApp()->reg().readUnsignedEntry("High Scores",gameLevel[mode].text(),maxTime+1) > currentTime )
    getApp()->reg().writeUnsignedEntry("High Scores",gameLevel[mode].text(),currentTime);
  }

// Quit
long FoxSweeper::onCmdQuit(FXObject*,FXSelector,void*){
  getApp()->removeTimeout(this,ID_TIMER);
  saveSettings();
  deleteGrid();
  getApp()->exit();
  return 1;
  }

// About
long FoxSweeper::onCmdAbout(FXObject*,FXSelector,void*){
  FXMessageBox::information(this,MBOX_OK,"About FoxSweeper", "FoxSweeper is a minesweeper copy,\nwritten by Mathew Robertson.\n\nThis software uses the FOX Toolkit Library (http://www.fox-toolkit.org),\nand the FXEX extension library (http://foXdesktop.sf.net)." );
  return 1;
  }

// start a new game
long FoxSweeper::onCmdNewGame(FXObject*,FXSelector,void*){
  playAgain = TRUE;
  handle(this,MKUINT(ID_QUIT,SEL_COMMAND),NULL);
  return 1;
  }

// pause the game
long FoxSweeper::onCmdPause(FXObject*,FXSelector,void*){
  if (getApp()->hasTimeout(this,ID_TIMER)) getApp()->removeTimeout(this,ID_TIMER);
  else onTimer(NULL,0,NULL);
  return 1;
  }

// update pause button
long FoxSweeper::onUpdPause( FXObject* sender, FXSelector, void* ){
  if (getApp()->hasTimeout(this,ID_TIMER))
    sender->handle(this,MKUINT(FXWindow::ID_ENABLE,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(FXWindow::ID_DISABLE,SEL_COMMAND),NULL);
  return 1;
  }

// show settings dialog
long FoxSweeper::onCmdSettings(FXObject*,FXSelector,void*){
  point size = gridSize;
  FXint bombs = bombTotal;
  GameLevel currentMode=mode;

  Settings settings(this);
  if (settings.execute()){
    playAgain = TRUE;
    handle(this,MKUINT(ID_QUIT,SEL_COMMAND),NULL);
    }
  else{
    gridSize = size;
    bombTotal = bombs;
    mode = currentMode;
    }
  return 1;
  }

// show highscore dialog
long FoxSweeper::onCmdHighScore(FXObject*,FXSelector,void*){
  HighScores hs(this);
  hs.execute();
  return 1;
  }

// handle left click - show number/bomb under button
long FoxSweeper::onCmdLeftClick(FXObject* o,FXSelector,void*){
  if(!getApp()->hasTimeout(this,ID_TIMER)) onTimer(NULL,0,NULL);
  fsButton* button = (fsButton*) o;
  FXuint y = matrix->rowOfChild(button);
  FXuint x = matrix->colOfChild(button);
  evaluateThisLocation(x,y);
  return 1;
  }

// handle middle click - if button is opened, open surrounding locations
long FoxSweeper::onCmdMiddleClick(FXObject* o,FXSelector,void*) {
  if(!getApp()->hasTimeout(this,ID_TIMER)) onTimer(NULL,0,NULL);
  fsButton* button = (fsButton*) o;
  FXuint y = matrix->rowOfChild(button);
  FXuint x = matrix->colOfChild(button);
  FXbool statOk=evaluateThisLocation(x,y);

  FXuint xm = gridData.xsize() - 1;
  FXuint ym = gridData.ysize() - 1;
  fsButtonData *tb;
  if (statOk && x>0  && y>0 ){ tb=gridData.find(x-1,y-1); if(tb->state != opened) statOk=evaluateThisLocation(x-1,y-1); }
  if (statOk &&         y>0 ){ tb=gridData.find(x  ,y-1); if(tb->state != opened) statOk=evaluateThisLocation(x  ,y-1); }
  if (statOk && x<xm && y>0 ){ tb=gridData.find(x+1,y-1); if(tb->state != opened) statOk=evaluateThisLocation(x+1,y-1); }
  if (statOk && x>0         ){ tb=gridData.find(x-1,y  ); if(tb->state != opened) statOk=evaluateThisLocation(x-1,y  ); }
  if (statOk && x<xm        ){ tb=gridData.find(x+1,y  ); if(tb->state != opened) statOk=evaluateThisLocation(x+1,y  ); }
  if (statOk && x>0  && y<ym){ tb=gridData.find(x-1,y+1); if(tb->state != opened) statOk=evaluateThisLocation(x-1,y+1); }
  if (statOk &&         y<ym){ tb=gridData.find(x  ,y+1); if(tb->state != opened) statOk=evaluateThisLocation(x  ,y+1); }
  if (statOk && x<xm && y<ym){ tb=gridData.find(x+1,y+1); if(tb->state != opened) statOk=evaluateThisLocation(x+1,y+1); }
  return 1;
  }

// show flag/question on right click
long FoxSweeper::onCmdRightClick(FXObject* o,FXSelector,void*) {
  if(!getApp()->hasTimeout(this,ID_TIMER)) onTimer(NULL,0,NULL);
  FXString txt = "";
  fsButton* button = (fsButton*) o;
  FXuint y = matrix->rowOfChild(button);
  FXuint x = matrix->colOfChild(button);
  fsButtonData *b = gridData.find(x,y);
  switch (b->state){
    case hidden:{
      b->state = flagged;
      button->setIcon(icons->Flag);
      bombCount--;
      buttonCount++;
      o->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
      break;
      }
    case flagged:{
      b->state = questioned;
      button->setIcon(NULL);
      txt = "?";
      bombCount++;
      buttonCount--;
      o->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
      break;
      }
    case questioned:{
      b->state = hidden;
      txt = " ";
      o->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND), (void*) &txt);
      break;
      }
    default:{
      break;
      }
    }
  evalEndGame();
  return 1;
  }

// set to custom rows
long FoxSweeper::onCmdNRows(FXObject* sender,FXSelector,void*) {
  mode=Custom;
  FXuint rows;
  sender->handle(this,MKUINT(ID_GETINTVALUE,SEL_COMMAND), (void*) &rows);
  if ( rows > maxSize ) rows = maxSize;
  if ( rows < minSize ) rows = minSize;
  gridSize.y=rows;
  return 1;
  }

// set to custom colums
long FoxSweeper::onCmdNCols(FXObject* sender,FXSelector,void*) {
  mode=Custom;
  FXuint cols;
  sender->handle(this,MKUINT(ID_GETINTVALUE,SEL_COMMAND), (void*) &cols);
  if ( cols > maxSize ) cols = maxSize;
  if ( cols < minSize ) cols = minSize;
  gridSize.x=cols;
  return 1;
  }

// set to custom bombs
long FoxSweeper::onCmdNBombs(FXObject* sender,FXSelector,void*) {
  mode=Custom;
  FXuint bombs;
  sender->handle(this,MKUINT(ID_GETINTVALUE,SEL_COMMAND), (void*) &bombs);
  if ( bombs > maxBombs ) bombs = maxBombs;
  if ( bombs < minBombs ) bombs = minBombs;
  bombTotal = bombs;
  return 1;
  }

// update number of rows
long FoxSweeper::onUpdNRows(FXObject* sender,FXSelector,void*) {
  FXuint rows=gridSize.y;
  sender->handle(this,MKUINT(ID_SETINTVALUE,SEL_COMMAND), (void*) &rows);
  return 1;
  }

// update number of columns
long FoxSweeper::onUpdNCols(FXObject* sender,FXSelector,void*) {
  FXuint cols=gridSize.x;
  sender->handle(this,MKUINT(ID_SETINTVALUE,SEL_COMMAND), (void*) &cols);
  return 1;
  }

// update number of bombs
long FoxSweeper::onUpdNBombs(FXObject* sender,FXSelector,void*) {
  if ( (unsigned)bombTotal > (gridSize.x * gridSize.y) ) bombTotal=gridSize.x * gridSize.y;
  FXuint bombs=bombTotal;
  sender->handle(this,MKUINT(ID_SETINTVALUE,SEL_COMMAND), (void*) &bombs);
  return 1;
  }

// update the clock - check if we ran out of time
long FoxSweeper::onTimer( FXObject*, FXSelector, void* ){
  if (currentTime > maxTime){
    getApp()->removeTimeout(this,ID_TIMER);
    FXString txt = "You ran out of time (after ";
    txt += FXStringVal(currentTime);
    if (currentTime==1) txt += " second";
    else txt += " seconds";
    txt += ")!\nWould you like to play again?";
    showRemainingBombs();
    playAgain = questionBox("Argh!",txt);
    handle(this,MKUINT(ID_QUIT,SEL_COMMAND),NULL);
    }
  else{
    currentTime++;
    FXString title = getApp()->getAppName();
    title += ": ";
    title += FXStringVal(currentTime);
    if (currentTime==1) title += " second";
    else title += " seconds";
    setTitle(title);
    getApp()->addTimeout( this, ID_TIMER, clockUpdateRate );
    }
  return 1;
  }

// set to specific game mode
long FoxSweeper::onCmdMode( FXObject*, FXSelector sel, void* ){
  GameLevel level = (GameLevel) ( FXSELID(sel) - ID_BEGINNER );
  switch(level){
    case Beginner:     mode=Beginner;     modeBeginner();     break;
    case Intermediate: mode=Intermediate; modeIntermediate(); break;
    case Advanced:     mode=Advanced;     modeAdvanced();     break;
    case Custom:       mode=Custom;       break;
    default:           break;
    }
  return 1;
  }

// update game mode
long FoxSweeper::onUpdMode( FXObject* sender, FXSelector sel, void* ){
  GameLevel level = (GameLevel) ( FXSELID(sel) - ID_BEGINNER );
  if (mode==level)
    sender->handle(this,MKUINT(FXWindow::ID_CHECK,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(FXWindow::ID_UNCHECK,SEL_COMMAND),NULL);
  return 1;
  }

// update highscore value
long FoxSweeper::onUpdHighScore( FXObject* sender, FXSelector sel, void* ){
  GameLevel level = (GameLevel) ( FXSELID(sel) - ID_HS_BEGINNER );
  FXint score = getApp()->reg().readIntEntry("High Scores", gameLevel[level].text(), maxTime+1);
  sender->handle(this,MKUINT(FXWindow::ID_SETINTVALUE,SEL_COMMAND),(void*)&score);
  return 1;
  }
